﻿Imports Microsoft.VisualBasic
Imports System.Windows.Forms.DataVisualization.Charting

Public Class Graphed_Signal
    Public Enum Signal_Type
        Unassigned
        Stack_Summary
        Board_Summary
        Cell_Voltage
        Temperature
        Num_Types
    End Enum

    Public Enum Summary_Signal_SubType
        Sum
        Ave
        Max_Cell
        Min_Cell
        Max_Temp
        Min_Temp
        Num_SubTypes
    End Enum

    Public Const NUM_PRINTED_FIELDS As Integer = 3  ' the time, value, and balancing are written to/read from the CSV files
    Public Structure Data_Point
        Public time As Single
        Public value As Single
        Public is_balancing As Boolean
        Public valid As Boolean
    End Structure

    Private graph As Chart
    Public data() As Data_Point
    Public type As Signal_Type
    Public board_num As Integer
    Public other_num As Integer
    Public secondary_y_axis As Boolean
    Public name As String
    
    Public Sub New(ByVal num_datapoints As Integer, ByVal graph As Chart)

        ' Assign the chart and number of data points to this object
        ReDim Me.data(num_datapoints - 1)
        Me.graph = graph

        ' Clear the data storage for this object
        Me.Reset()
    End Sub

    Public Sub Reset(Optional ByVal data_only As Boolean = False)

        If data_only = False Then
            ' Remove graphed signal from chart if it's present
            Dim chart_series_index As Integer = Me.graph.Series.IndexOf(Me.name)
            If chart_series_index >= 0 Then
                Me.graph.Series.RemoveAt(chart_series_index)
            End If

            ' Mark signal as unassigned
            Me.type = Signal_Type.Unassigned
            Me.board_num = 0
            Me.other_num = 0
            Me.secondary_y_axis = 0
            Me.name = ""
        End If

        ' Mark all of the data as invalid
        For dp_num As Integer = 0 To Me.data.Length - 1
            Me.data(dp_num).valid = False
        Next
    End Sub

    Public Sub Start(ByVal type As Graphed_Signal.Signal_Type, Optional ByVal board_num As Integer = 0, Optional ByVal other_num As Integer = 0, Optional ByVal secondary_y_axis As Boolean = False)

        ' Save parameters necessary for accessing data later
        Me.type = type
        Me.board_num = board_num
        Me.other_num = other_num
        Me.secondary_y_axis = secondary_y_axis

        ' Create the name of the signal
        Select Case type
            Case Signal_Type.Stack_Summary
                Me.name = "Summary Stack"
            Case Signal_Type.Board_Summary
                Me.name = "Summary Board " + CStr(board_num + 1)
            Case Signal_Type.Cell_Voltage
                ' todo - gross way to reuse this code for the Cap Demo graph
                If D33001.Board_ID_Data(D33001.DC2100A_PIC_BOARD_NUM).Cap_Demo = True Then
                    Me.name = "Cell " + CStr(other_num + 1)
                Else
                    Me.name = "Board " + CStr(board_num + 1) + " Cell " + CStr(other_num + 1)
                End If
            Case Signal_Type.Temperature
                If D33001.numBoards = 1 Then
                    Me.name = "Temp " + CStr(other_num + 1)
                Else
                    Me.name = "Board " + CStr(board_num + 1) + " Temp " + CStr(other_num + 1)
                End If
        End Select

        If (type = Signal_Type.Stack_Summary) Or (type = Signal_Type.Board_Summary) Then
            If (other_num = Summary_Signal_SubType.Sum) Then
                Me.name += " Sum Cells"
            ElseIf (other_num = Summary_Signal_SubType.Ave) Then
                Me.name += " Ave Cell"
            ElseIf (other_num = Summary_Signal_SubType.Max_Cell) Then
                Me.name += " Max Cell"
            ElseIf (other_num = Summary_Signal_SubType.Min_Cell) Then
                Me.name += " Min Cell"
            ElseIf (other_num = Summary_Signal_SubType.Max_Temp) Then
                Me.name += " Max Temp"
            ElseIf (other_num = Summary_Signal_SubType.Min_Temp) Then
                Me.name += " Min Temp"
            End If
        End If

        ' Add the signal to the graph and set its options
        Me.graph.Series.Add(Me.name)
        Me.graph.Series(Me.name).ChartType = DataVisualization.Charting.SeriesChartType.FastLine
        Me.graph.Series(Me.name).BorderWidth = 2

        ' Set up the primary and secondary axis
        If Me.secondary_y_axis = False Then
            Me.graph.Series(Me.name).YAxisType = AxisType.Primary
        Else
            Me.graph.Series(Me.name).YAxisType = AxisType.Secondary
            Me.graph.Series(Me.name).BorderDashStyle = ChartDashStyle.Dash
            Me.graph.ChartAreas(0).AxisY2.LineColor = Drawing.Color.Transparent
            Me.graph.ChartAreas(0).AxisY2.MajorGrid.Enabled = False
            Me.graph.ChartAreas(0).AxisY2.Enabled = AxisEnabled.True
            Me.graph.ChartAreas(0).AxisY2.Title = "(°C or Volts)"
        End If


    End Sub

    ' Plot the valid datapoints in a range.  Note that the first_datapoint and num_datapoints are not tracked by the ojbect
    ' itself, because I want all the signals to stay in sync even if they are started/stopped at different times.
    Public Sub Plot(ByVal datapoint_index As Integer, ByVal max_datapoints As Integer)

        ' Plot the graphed signal on the chart if it's present
        Dim chart_series_index As Integer = Me.graph.Series.IndexOf(Me.name)
        If chart_series_index >= 0 Then

            ' If more than the max points, remove the old point
            If graph.Series(chart_series_index).Points.Count > max_datapoints Then
                graph.Series(chart_series_index).Points.RemoveAt(0)
            End If

            ' Add the new point
            If Me.data(datapoint_index).valid = True Then
                graph.Series(chart_series_index).Points.AddXY(Me.data(datapoint_index).time, Me.data(datapoint_index).value)
            End If

        End If

    End Sub

End Class

Public Class Graphed_Signal_Group

    Private graph As Chart
    Private hscrollbar As HScrollBar
    Private debug_textbox As TextBox
    Public Graphed_Signals() As Graphed_Signal

    Public Const MAX_DATAPOINTS As Integer = 10000

    Public Const X_AXIS_FORMAT As String = "0.00"
    Public Const Y_AXIS_MAX As Single = 5.5
    Public Const Y_AXIS_MIN As Single = -0.5
    Public Const Y_AXIS_INTERVAL_COARSE As Single = 0.5
    Public Const Y_AXIS_INTERVAL_FINE As Single = 0.1
    Public Const Y_AXIS_FORMAT As String = "0.0"

    Public num_signals As Integer

    ' Flag that the graph was started or a signal was added, and we want to make sure this isn't done in the middle of a communication cycle.
    Public prime_datapoints As Boolean

    ' important indexes in the wraparound buffer of datapoints, so that we don't have to keep calculating them as we add/plot/get min and max/etc datapoints.
    Public first_datapoint As Integer
    Public last_datapoint As Integer
    Public next_datapoint As Integer
    Public num_datapoints As Integer

    ' timestamp at the time the data collection is started, to be subtracted from subsequent timestamps so that the graph always starts at 0 sec
    Public timestamp_start As Single

    ' Data range is a max + min, with bit indicating whether a range existed to calculate the min/max.
    Public Structure Range
        Public min As Single
        Public max As Single
        Public valid As Boolean
    End Structure

    Public Sub New(ByVal num_signals As Integer,
                   ByVal graph As Chart,
                   ByVal hscrollbar As HScrollBar,
                   Optional ByVal debug_textbox As TextBox = Nothing)

        ' Store the controls and indicators for this graphed signal group
        Me.graph = graph
        Me.hscrollbar = hscrollbar
        Me.debug_textbox = debug_textbox

        ' Store the number of signals contained in this group
        Me.num_signals = num_signals
        ReDim Graphed_Signals(num_signals - 1)

        ' Create the Graphed Signals.
        For signal_num As Integer = 0 To num_signals - 1
            Graphed_Signals(signal_num) = New Graphed_Signal(MAX_DATAPOINTS, graph)
        Next signal_num

        ' Reset the Graphed Signals and the variables used to graph them
        Me.Reset()

        ' Initialize the GUI controls
        Me.Update_GUI(True)

    End Sub

    Public Sub Reset(Optional ByVal data_only As Boolean = False)

        Me.prime_datapoints = True ' If graphing is started in the middle of receiving data, avoid the situation where we try to plot data when it's only populated for some of the signals.

        Me.first_datapoint = 0
        Me.next_datapoint = 0
        Me.last_datapoint = 0
        Me.num_datapoints = 0
        Me.timestamp_start = -1

        ' Reset the graphed signals
        For signal_num As Integer = 0 To Me.num_signals - 1
            Me.Graphed_Signals(signal_num).Reset(data_only)
        Next signal_num

        ' Reset the Scrollbar

        Me.hscrollbar.Minimum = 0
        Me.hscrollbar.Maximum = 0
        Me.hscrollbar.Value = 0

    End Sub

    ' Attempt to add a signal to the chart, if any signals are currently unassigned
    Public Function Add_Signal(ByVal type As Graphed_Signal.Signal_Type, Optional ByVal board_num As Integer = 0, Optional ByVal other_num As Integer = 0) As Boolean

        Me.prime_datapoints = True  ' If a signal is added in the middle of receiving data, avoid the situation where we try to plot data when it's only populated for some of the signals. 

        For signal_num As Integer = 0 To Me.num_signals - 1
            If Graphed_Signals(signal_num).type = Graphed_Signal.Signal_Type.Unassigned Then

                ' Pick an axis for this signal
                ' Primary Y Axis is for cell voltages.  Everything else goes on the Secondary axis.
                Dim secondary_y_axis As Boolean

                If type = Graphed_Signal.Signal_Type.Cell_Voltage Then
                    secondary_y_axis = False
                ElseIf (type = Graphed_Signal.Signal_Type.Stack_Summary) Or (type = Graphed_Signal.Signal_Type.Board_Summary) Then
                    If (other_num = Graphed_Signal.Summary_Signal_SubType.Ave) Or
                        (other_num = Graphed_Signal.Summary_Signal_SubType.Min_Cell) Or
                        (other_num = Graphed_Signal.Summary_Signal_SubType.Max_Cell) Then
                        secondary_y_axis = False
                    Else
                        secondary_y_axis = True
                    End If
                Else
                    secondary_y_axis = True
                End If

                ' Start taking data for this signal
                Graphed_Signals(signal_num).Start(type, board_num, other_num, secondary_y_axis)

                ' Adjust the Legend based upon how many signals are present
                If Me.num_signals <= 12 Then
                    Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 10.0!)
                Else
                    If Me.num_signals - Me.Get_Num_Unassigned() > 9 Then
                        Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 6.25!)
                    ElseIf Me.num_signals - Me.Get_Num_Unassigned() > 6 Then
                        Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 7.25!)
                    Else
                        Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 8.25!)
                    End If
                End If

                ' Return now that signal has been started
                Return True
            End If
        Next signal_num

        ' Return that the signal has not been started
        Return False

    End Function

    ' Attempt to remove a signal from the chart, if that signal is currently in the chart
    Public Function Remove_Signal(ByVal type As Graphed_Signal.Signal_Type, Optional ByVal board_num As Integer = 0, Optional ByVal other_num As Integer = 0) As Boolean

        For signal_num As Integer = 0 To Me.num_signals - 1
            If (Me.Graphed_Signals(signal_num).type = type) And
               (Me.Graphed_Signals(signal_num).board_num = board_num) And
               (Me.Graphed_Signals(signal_num).other_num = other_num) Then

                If Me.Get_Num_Unassigned() + 1 = num_signals Then
                    ' If this is the last signal to get removed, then reset the whole graphed signal group
                    Me.Reset()
                Else
                    ' If this is not the last signal to get removed, then just reset this signal
                    Me.Graphed_Signals(signal_num).Reset()

                    ' Adjust the Legend based upon how many signals are present
                    If Me.num_signals <= 12 Then
                        Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 10.0!)
                    Else
                        If Me.num_signals - Me.Get_Num_Unassigned() > 9 Then
                            Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 6.25!)
                        ElseIf Me.num_signals - Me.Get_Num_Unassigned() > 6 Then
                            Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 7.25!)
                        Else
                            Me.graph.Legends(0).Font = New System.Drawing.Font("Microsoft Sans Serif", 8.25!)
                        End If
                    End If

                End If
                Return True

            End If
        Next signal_num
        Return False
    End Function

    ' Set the data points to the most recent values read from the PIC
    Public Sub Update_Data_Points(Optional ByVal voltage_only As Boolean = False)

        ' If no signals are assigned, then exit sub
        If Me.Get_Num_Unassigned = Me.num_signals Then
            Exit Sub
        End If

        ' If a signal was added between this and the last time that datapoints were added (or if the window was just opened), skip adding this line
        ' of data as it might not be complete.  It's very unlikely that the graph activity coincided exactly with the beginning of a comm cycle.
        If (Me.prime_datapoints = True) Then
            If (D33001.voltage_timestamp(D33001.DC2100A_PIC_BOARD_NUM).time <> 0) Then  ' If time is 0, then we likely have not received valid data yet.
                Me.prime_datapoints = False
            End If
            Exit Sub
        End If

        ' If this is the first datapoint taken, remember the timestamp so that we can make graph start at 0
        If timestamp_start < 0 Then
            Dim min_timestamp As Single = D33001.voltage_timestamp(D33001.DC2100A_PIC_BOARD_NUM).time

            For board_num As Integer = 0 To D33001.numBoards - 1
                min_timestamp = Math.Min(min_timestamp, D33001.voltage_timestamp(board_num).time)
                If voltage_only = False Then
                    min_timestamp = Math.Min(min_timestamp, D33001.temperature_timestamp(board_num).time)
                End If
            Next board_num

            timestamp_start = min_timestamp
        End If

        ' Get the data to be added to the graph for each signal
        Dim any_signals_unassigned As Boolean = False
        For signal_num As Integer = 0 To Me.num_signals - 1

            Dim time As Single
            Dim value As Single
            Dim is_balancing As Boolean

            ' Only update a signal if it's assigned to a source of data
            If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then

                ' Flag that signals are be plotted
                any_signals_unassigned = True

                ' Map between signals that can be graphed, and their source in the main form
                Select Case Graphed_Signals(signal_num).type

                    Case Graphed_Signal.Signal_Type.Stack_Summary
                        If (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Sum) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Volt_Sum
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Ave) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Volt_Average
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Max_Cell) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Volt_Max
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Min_Cell) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Volt_Min
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Max_Temp) Then
                            time = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Temp_Max
                            is_balancing = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Min_Temp) Then
                            time = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Stack_Summary_Data.Temp_Min
                            is_balancing = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        End If

                    Case (Graphed_Signal.Signal_Type.Board_Summary)
                        If (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Sum) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Volt_Sum
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Ave) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Volt_Average
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Max_Cell) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Volt_Max
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Min_Cell) Then
                            time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Volt_Min
                            is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Max_Temp) Then
                            time = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Temp_Max
                            is_balancing = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        ElseIf (Graphed_Signals(signal_num).other_num = Graphed_Signal.Summary_Signal_SubType.Min_Temp) Then
                            time = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                            value = D33001.Board_Summary_Data(Graphed_Signals(signal_num).board_num).Temp_Min
                            is_balancing = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                        End If

                    Case Graphed_Signal.Signal_Type.Cell_Voltage
                        time = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                        value = D33001.Voltages(Graphed_Signals(signal_num).board_num, Graphed_Signals(signal_num).other_num)
                        is_balancing = D33001.voltage_timestamp(Graphed_Signals(signal_num).board_num).is_balancing

                    Case Graphed_Signal.Signal_Type.Temperature
                        time = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).time - timestamp_start
                        value = D33001.Temperatures(Graphed_Signals(signal_num).board_num, Graphed_Signals(signal_num).other_num)
                        is_balancing = D33001.temperature_timestamp(Graphed_Signals(signal_num).board_num).is_balancing
                End Select

                ' Put the data in the graphed signal
                Graphed_Signals(signal_num).data(next_datapoint).time = time
                Graphed_Signals(signal_num).data(next_datapoint).value = value
                Graphed_Signals(signal_num).data(next_datapoint).is_balancing = is_balancing
                Graphed_Signals(signal_num).data(next_datapoint).valid = True

            End If

        Next signal_num

        ' If signals are being graphed, update the number of datapoints and location where next data should be added
        If any_signals_unassigned = True Then

            ' Update the number of datapoints written
            num_datapoints += 1

            ' Save index to the datapoint we just wrote
            last_datapoint = next_datapoint

            ' Update index to next datapoint
            If (next_datapoint < MAX_DATAPOINTS - 1) Then
                next_datapoint += 1
            Else
                next_datapoint = 0
            End If

            ' First datapoint only moves once wrapping occurs
            If (num_datapoints > MAX_DATAPOINTS) Then
                If (first_datapoint < MAX_DATAPOINTS - 1) Then
                    first_datapoint += 1
                Else
                    first_datapoint = 0
                End If
            End If

        End If

    End Sub

    ' Set the data points to specific values (probably from an input file)
    Public Sub Set_Data_Points(ByVal data() As Graphed_Signal.Data_Point)

        ' If no signals are assigned, then exit sub
        If Me.Get_Num_Unassigned = Me.num_signals Then
            Exit Sub
        End If

        ' Set the data to be added to the graph for each signal
        For signal_num As Integer = 0 To Me.num_signals - 1

            ' Only update a signal if it's assigned to a source of data
            If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then

                ' Put the data in the graphed signal
                If (data(signal_num).valid = True) Then
                    Graphed_Signals(signal_num).data(next_datapoint).time = data(signal_num).time
                    Graphed_Signals(signal_num).data(next_datapoint).value = data(signal_num).value
                    Graphed_Signals(signal_num).data(next_datapoint).is_balancing = data(signal_num).is_balancing
                    Graphed_Signals(signal_num).data(next_datapoint).valid = True
                End If
            End If

        Next signal_num

        ' Update the number of datapoints written
        num_datapoints += 1

        ' Save index to the datapoint we just wrote
        last_datapoint = next_datapoint

        ' Update index to next datapoint
        If (next_datapoint < MAX_DATAPOINTS - 1) Then
            next_datapoint += 1
        Else
            next_datapoint = 0
        End If

        ' First datapoint only moves once wrapping occurs
        If (num_datapoints > MAX_DATAPOINTS) Then
            If (first_datapoint < MAX_DATAPOINTS - 1) Then
                first_datapoint += 1
            Else
                first_datapoint = 0
            End If
        End If

    End Sub

    ' Update the chart and scrollbar.
    Public Sub Update_GUI(ByVal controls_only As Boolean)

        Try
            ' Update the graphed signals, unless we're just updating the controls (by actions such as moving the scrollbar)
            If controls_only = False Then
                For signal_num As Integer = 0 To Me.num_signals - 1
                    ' Only update a signal if it's assigned to a source of data
                    If (Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned) Then
                        Graphed_Signals(signal_num).Plot(last_datapoint, MAX_DATAPOINTS)
                    End If
                Next signal_num
            End If

            ' Adjust x-axis and scroll bar for the new data points
            Dim x_range As Range = Get_Min_and_Max_Time()

            If (x_range.valid = True) Then

                ' Adjust scrollbar to the new min/max
                hscrollbar.Minimum = x_range.min
                hscrollbar.Maximum = x_range.max

                ' If in scope mode, adjust x-axis just to show the portion selected by the scroll bar
                If GraphOptionPopup.scopeMode = True Then
                    ' If snapped to the front or if there aren't enough datapoints in the desired range, adjust the scrollbar value to the max
                    If (GraphOptionPopup.snappedToFront = True) Or
                       (GraphOptionPopup.scopeRange > (x_range.max - x_range.min)) Then
                        hscrollbar.Value = hscrollbar.Maximum
                    End If

                    ' Adjust the window displayed in the graph
                    x_range.max = Math.Min(hscrollbar.Value, x_range.max)
                    x_range.min = x_range.max - Math.Min(GraphOptionPopup.scopeRange, (x_range.max - x_range.min))

                    ' Readjust max if the scrollbar Value reduces the window size below the desired range.
                    If (x_range.max - x_range.min) < GraphOptionPopup.scopeRange Then
                        x_range.max = Math.Min(x_range.min + GraphOptionPopup.scopeRange, hscrollbar.Maximum)
                    End If

                    If x_range.max < x_range.min Then
                        x_range.max = x_range.min
                    End If
                Else
                    hscrollbar.Value = hscrollbar.Maximum
                End If

            End If

            If (CDbl(x_range.min.ToString(X_AXIS_FORMAT)) <= graph.ChartAreas(0).Axes(0).Maximum) Then
                graph.ChartAreas(0).Axes(0).Minimum = CDbl(x_range.min.ToString(X_AXIS_FORMAT))
                graph.ChartAreas(0).Axes(0).Maximum = CDbl(x_range.max.ToString(X_AXIS_FORMAT))
            Else
                graph.ChartAreas(0).Axes(0).Maximum = CDbl(x_range.max.ToString(X_AXIS_FORMAT))
                graph.ChartAreas(0).Axes(0).Minimum = CDbl(x_range.min.ToString(X_AXIS_FORMAT))
            End If
            graph.ChartAreas(0).AxisX.LabelStyle.Format = X_AXIS_FORMAT

            ' Adjust Primary Y-axis scale depending upon X-Axis and graph options
            Dim y_range As Range = Get_Min_and_Max_Value(x_range, False)
            If (y_range.valid = True) Then

                Dim y_interval As Single = If((y_range.max - y_range.min) > (2 * Y_AXIS_INTERVAL_COARSE), Y_AXIS_INTERVAL_COARSE, Y_AXIS_INTERVAL_FINE)
                If (GraphOptionPopup.ShowZeroOnY = True) Or (x_range.valid = False) Then
                    y_interval = Y_AXIS_INTERVAL_COARSE
                    y_range.min = Y_AXIS_MIN
                    y_range.max = Y_AXIS_MAX
                Else
                    y_interval = If((y_range.max - y_range.min) > (2 * Y_AXIS_INTERVAL_COARSE), Y_AXIS_INTERVAL_COARSE, Y_AXIS_INTERVAL_FINE)
                    y_range.min = Math.Max(Y_AXIS_MIN, y_range.min - y_interval)
                    y_range.max = Math.Min(Y_AXIS_MAX, y_range.max + y_interval)
                End If

                If y_range.max < y_range.min Then
                    y_range.max = y_range.min
                End If

                If (CDbl(y_range.min.ToString(Y_AXIS_FORMAT)) <= graph.ChartAreas(0).AxisY.Maximum) Then
                    graph.ChartAreas(0).AxisY.Minimum = CDbl(y_range.min.ToString(Y_AXIS_FORMAT))
                    graph.ChartAreas(0).AxisY.Maximum = CDbl(y_range.max.ToString(Y_AXIS_FORMAT))
                Else
                    graph.ChartAreas(0).AxisY.Maximum = CDbl(y_range.max.ToString(Y_AXIS_FORMAT))
                    graph.ChartAreas(0).AxisY.Minimum = CDbl(y_range.min.ToString(Y_AXIS_FORMAT))
                End If
                graph.ChartAreas(0).AxisY.Interval = CDbl(y_interval.ToString(Y_AXIS_FORMAT))

            End If

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    ' Scroll the scrollbar.
    Public Sub ScrollBar_Scroll()
        Dim snapped_to_front_threshold As Integer = hscrollbar.Maximum - 9
        '        Dim snapped_to_front_threshold As Integer = hscrollbar.Minimum + ((hscrollbar.Maximum - hscrollbar.Minimum) * 850) \ 1000

        'If we're manipulating the scrollbar, it's most likely that we will not be snapped to the front any more
        GraphOptionPopup.snappedToFront = False

        'If the scrollbar is selected withing the visible range's distance from the the end and we're in scope mode, then lock at the front
        If (GraphOptionPopup.scopeMode = True) AndAlso (hscrollbar.Value >= snapped_to_front_threshold) AndAlso (graph.ChartAreas(0).AxisX.Maximum <> 0) Then
            hscrollbar.Value = graph.ChartAreas(0).AxisX.Maximum
            GraphOptionPopup.snappedToFront = True
        ElseIf (GraphOptionPopup.scopeMode = False) AndAlso (hscrollbar.Value >= snapped_to_front_threshold) Then
            hscrollbar.Value = hscrollbar.Maximum
        End If

        debug_textbox.Text = "Range :" + hscrollbar.Minimum.ToString() + " - " + hscrollbar.Maximum.ToString() + "; "
        debug_textbox.Text += hscrollbar.Value.ToString()
        If (hscrollbar.Value > snapped_to_front_threshold) Then
            debug_textbox.Text += " > "
        ElseIf (hscrollbar.Value < snapped_to_front_threshold) Then
            debug_textbox.Text += " < "
        Else
            debug_textbox.Text += " = "
        End If
        debug_textbox.Text += snapped_to_front_threshold.ToString()
    End Sub

    ' Print the contents of this signal group to a CSV
    Public Sub Output_CSV(ByVal filename As String, ByVal x As Range)
        Try
            Dim textline As String
            Dim writer As System.IO.StreamWriter

            ' If full path is not passed in, then append filename to path where we are executing as it's where we're going to write the file.
            If (filename.IndexOf(System.IO.Path.DirectorySeparatorChar) < 0) Then
                filename = D33001.filePath + filename
            End If

            ' Open the writer.  Overwrite old file.
            writer = New System.IO.StreamWriter(filename, False)

            ' Save the filename, date, time
            writer.WriteLine(filename)
            writer.WriteLine(String.Format("{0:MM/dd/yyyy}", DateTime.Now))
            writer.WriteLine(String.Format("{0:HH:mm}", DateTime.Now))

            ' Next save the signals present in this data
            textline = ""
            For signal_num As Integer = 0 To Me.num_signals - 1
                ' Only update a signal if it's assigned to a source of data
                If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then
                    ' Write each signal multiple times, as there will be a column for time, value, etc
                    For field_num As Integer = 0 To Graphed_Signal.NUM_PRINTED_FIELDS - 1
                        textline += Graphed_Signals(signal_num).name + ","
                    Next field_num
                End If
            Next signal_num
            writer.WriteLine(textline)

            ' Next save the headings for the X and Y data 
            textline = ""
            For signal_num As Integer = 0 To Me.num_signals - 1
                ' Only update a signal if it's assigned to a source of data
                If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then
                    textline += "time,value,balancing,"
                End If
            Next signal_num
            writer.WriteLine(textline)

            ' Finally, save the datapoints for these signals
            Dim datapoint_index As Integer = first_datapoint
            For datapoint_num As Integer = 0 To Math.Min(num_datapoints, MAX_DATAPOINTS) - 1

                Dim data_written As Boolean = False
                textline = ""
                For signal_num As Integer = 0 To Me.num_signals - 1
                    ' Only update a signal if it's assigned to a source of data
                    If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then

                        ' Only write the valid datapoints that are inside a range of time (if a range of time is specified)
                        If (Graphed_Signals(signal_num).data(datapoint_index).valid = True) AndAlso
                            ((x.min <= 0) Or (x.min <= Graphed_Signals(signal_num).data(datapoint_index).time)) AndAlso
                            ((x.max <= 0) Or (x.max >= Graphed_Signals(signal_num).data(datapoint_index).time)) Then
                            textline += Graphed_Signals(signal_num).data(datapoint_index).time.ToString() + ","
                            textline += Graphed_Signals(signal_num).data(datapoint_index).value.ToString() + ","
                            textline += Graphed_Signals(signal_num).data(datapoint_index).is_balancing.ToString() + ","
                            data_written = True
                        Else
                            textline += ", , "
                        End If

                    End If
                Next signal_num

                ' Only write the line if it contains some data
                If data_written = True Then
                    writer.WriteLine(textline)
                End If

                ' update the index, watching for wraparound
                If datapoint_index < MAX_DATAPOINTS - 1 Then
                    datapoint_index += 1
                Else
                    datapoint_index = 0
                End If

            Next datapoint_num

            writer.Flush()
            writer.Close()

            ' Report that file has been saved
            Dim popup_msg As String
            popup_msg = Math.Min(num_datapoints, MAX_DATAPOINTS).ToString() + " data points were saved for "
            popup_msg += num_signals.ToString() + " signals to the file "
            popup_msg += filename
            MsgBox(popup_msg)

        Catch ex As Exception
            D33001.Handle_Exception(ex)
            MsgBox(filename + " could not be opened.")
        End Try

    End Sub

    ' Print the contents of this signal group to a CSV
    Public Sub Input_CSV(ByVal filename As String)
        Try
            Dim reader As System.IO.StreamReader
            Dim header_strings() As String
            Dim date_string As String
            Dim time_string As String
            Dim num_signals As Integer
            Dim num_points As Integer

            ' First clear out all of the signals
            Me.Reset()

            ' Next read out data about the new signals
            Try
                reader = New System.IO.StreamReader(filename)
            Catch ex As Exception
                D33001.Handle_Exception(ex)

                MsgBox(filename + " could not be opened.")
                Exit Sub

            End Try

            ' Read the filename, date, time
            filename = reader.ReadLine()
            date_string = reader.ReadLine()
            time_string = reader.ReadLine()

            ' Next read the signals present in this data
            header_strings = reader.ReadLine().Split(",")

            ' Each Signal is listed multiple times for the time/value/etc
            num_signals = header_strings.Length \ Graphed_Signal.NUM_PRINTED_FIELDS

            For signal_num As Integer = 0 To num_signals - 1

                Try
                    Dim type As Graphed_Signal.Signal_Type
                    Dim board_num As Integer
                    Dim other_num As Integer

                    ' If a signal type can be extracted from the signal name, add the signal
                    If Me.Get_Signal_Type_From_Name(header_strings(signal_num * Graphed_Signal.NUM_PRINTED_FIELDS), type, board_num, other_num) = True Then
                        Me.Add_Signal(type, board_num, other_num)
                    End If

                Catch ex As Exception
                    D33001.Handle_Exception(ex)

                End Try
            Next signal_num

            ' Read the time/value headings, but don't do anything with them
            reader.ReadLine()

            ' Then read out the data
            num_points = 0
            Dim data(num_signals - 1) As Graphed_Signal.Data_Point
            Dim data_strings(num_signals * Graphed_Signal.NUM_PRINTED_FIELDS - 1) As String

            While (Not reader.EndOfStream)

                ' Get a line of data
                data_strings = reader.ReadLine().Split(",")

                ' Convert line of data into data points for each signal
                Try

                    ' Set the data to be added to the graph for each signal
                    For signal_num As Integer = 0 To num_signals - 1

                        ' Start off assuming the data is valid
                        data(signal_num).valid = True

                        For field_num As Integer = 0 To Graphed_Signal.NUM_PRINTED_FIELDS - 1
                            If (data_strings(signal_num * Graphed_Signal.NUM_PRINTED_FIELDS + field_num) = "") Then
                                data(signal_num).valid = False
                            End If
                        Next field_num

                        If data(signal_num).valid = True Then
                            ' Get the time.
                            Try
                                data(signal_num).time = CSng(data_strings(signal_num * Graphed_Signal.NUM_PRINTED_FIELDS))
                            Catch
                                data(signal_num).valid = False
                            End Try

                            ' Get the value.
                            Try
                                data(signal_num).value = CSng(data_strings(signal_num * Graphed_Signal.NUM_PRINTED_FIELDS + 1))
                            Catch
                                data(signal_num).valid = False
                            End Try

                            ' Get if balancing was active.
                            Try
                                data(signal_num).is_balancing = CBool(data_strings(signal_num * Graphed_Signal.NUM_PRINTED_FIELDS + 2))
                            Catch
                                data(signal_num).valid = False
                            End Try
                        End If

                    Next signal_num

                    Me.Set_Data_Points(data)    ' Update the datapoints
                    Me.Update_GUI(False)        ' Plot the last datapoints

                    num_points += 1

                Catch ex As Exception
                    D33001.Handle_Exception(ex)
                End Try

            End While

            ' File has been converted to data.  Close it.
            reader.Close()

            ' Report that file has been loaded
            Dim popup_msg As String
            popup_msg = num_points.ToString() + " data points were read for "
            popup_msg += num_signals.ToString() + " signals from the file saved on "
            popup_msg += date_string + " at " + time_string + " as "
            popup_msg += filename
            MsgBox(popup_msg)

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    ' Gets the number of unassigned signals in this group
    Public Function Get_Num_Unassigned() As Integer
        Dim num_unassigned As Integer = 0

        For signal_num As Integer = 0 To Me.num_signals - 1
            If Graphed_Signals(signal_num).type = Graphed_Signal.Signal_Type.Unassigned Then
                num_unassigned += 1
            End If
        Next signal_num

        Return num_unassigned
    End Function

    ' Get Min and Max Time in Graphed Signals
    Public Function Get_Min_and_Max_Time() As Range
        Dim x As Range
        x.min = 0
        x.max = 0
        x.valid = False

        If Me.num_datapoints <> 0 Then

            ' Search for the max/min in each signal
            For signal_num As Integer = 0 To Me.num_signals - 1

                ' If signal is unassigned, ignore it for max/min
                If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then

                    ' Datapoints available, search the datapoints for max/min
                    For datapoint_index As Integer = 0 To MAX_DATAPOINTS - 1

                        ' If datapoint is valid, check to see if it's the max or min
                        If Graphed_Signals(signal_num).data(datapoint_index).valid = True Then

                            ' If this if the first valid point that we've seen, set it as the max/min
                            If x.valid <> True Then
                                x.valid = True
                                x.min = Graphed_Signals(signal_num).data(datapoint_index).time
                                x.max = Graphed_Signals(signal_num).data(datapoint_index).time
                            Else
                                ' Check to see if this data point is the new max/min
                                x.min = Math.Min(Graphed_Signals(signal_num).data(datapoint_index).time, x.min)
                                x.max = Math.Max(Graphed_Signals(signal_num).data(datapoint_index).time, x.max)
                            End If

                        End If

                    Next datapoint_index

                End If

            Next signal_num

        End If

        Return x

    End Function

    ' Get Min and Max Value in Graphed Signals For a Specified Region of Time
    Public Function Get_Min_and_Max_Value(ByVal x As Range, ByVal secondary_y_axis As Boolean) As Range
        Dim y As Range
        y.min = 0
        y.max = 0
        y.valid = False

        If Me.num_datapoints <> 0 Then

            ' Search for the max/min in each signal
            For signal_num As Integer = 0 To Me.num_signals - 1

                ' If signal is unassigned, ignore it for max/min
                If Graphed_Signals(signal_num).type <> Graphed_Signal.Signal_Type.Unassigned Then

                    ' If signal is not for this axis, ignore it for max/min
                    If Graphed_Signals(signal_num).secondary_y_axis = secondary_y_axis Then

                        ' Datapoints available, search the datapoints for max/min
                        For datapoint_index As Integer = 0 To MAX_DATAPOINTS - 1

                            ' If datapoint is valid, check to see if it's inside the specified time range
                            If Graphed_Signals(signal_num).data(datapoint_index).valid = True Then

                                ' If datapoint is inside the specified time range, check to see if it's the max or min
                                If (Graphed_Signals(signal_num).data(datapoint_index).time >= x.min) And
                                   (Graphed_Signals(signal_num).data(datapoint_index).time <= x.max) Then

                                    ' If this if the first valid point that we've seen, set it as the max/min
                                    If y.valid <> True Then
                                        y.valid = True
                                        y.min = Graphed_Signals(signal_num).data(datapoint_index).value
                                        y.max = Graphed_Signals(signal_num).data(datapoint_index).value
                                    Else
                                        ' Check to see if this data point is the new max/min
                                        If y.min > Graphed_Signals(signal_num).data(datapoint_index).value Then
                                            y.min = Graphed_Signals(signal_num).data(datapoint_index).value
                                        ElseIf y.max < Graphed_Signals(signal_num).data(datapoint_index).value Then
                                            y.max = Graphed_Signals(signal_num).data(datapoint_index).value
                                        End If
                                    End If

                                End If

                            End If

                        Next datapoint_index

                    End If

                End If

            Next signal_num

        End If

        Return y

    End Function

    ' todo - the string-to-signalname relationship needs to be better controlled
    Public Function Get_Signal_Type_From_Name(ByVal signal_num As Integer, ByRef type As Graphed_Signal.Signal_Type, ByRef board_num As Integer, ByRef other_num As Integer) As Boolean
        If Me.Graphed_Signals(signal_num).type = Graphed_Signal.Signal_Type.Unassigned Then
            Return False
        Else
            Return Get_Signal_Type_From_Name(Me.Graphed_Signals(signal_num).name, type, board_num, other_num)
        End If

    End Function
    Public Function Get_Signal_Type_From_Name(ByVal name As String, ByRef type As Graphed_Signal.Signal_Type, ByRef board_num As Integer, ByRef other_num As Integer) As Boolean

        ' First get rid of "_" and " " so that object names and displayed text can be parsed with the same function.
        name = name.Replace("_", "")
        name = name.Replace(" ", "")

        ' First look for the signal type
        If name.IndexOf("Summary") >= 0 Then
            ' This is a summary signal.  Get whether it is the stack or the board
            If name.IndexOf("Stack") >= 0 Then
                type = Graphed_Signal.Signal_Type.Stack_Summary
            ElseIf name.IndexOf("Board") >= 0 Then
                type = Graphed_Signal.Signal_Type.Board_Summary
            End If
        ElseIf name.IndexOf("Cell") >= 0 Then
            type = Graphed_Signal.Signal_Type.Cell_Voltage
        ElseIf name.IndexOf("Temp") >= 0 Then
            type = Graphed_Signal.Signal_Type.Temperature
        Else
            type = Graphed_Signal.Signal_Type.Unassigned
        End If

        ' If this is a board specific signal type, get the board number
        If name.IndexOf("Board") >= 0 Then
            ' First try parsing a 2 digit number
            Try
                board_num = Integer.Parse(name.Substring(name.IndexOf("Board") + "Board".Length, 2))
            Catch
                ' Try parsing a 1 digit number if that didn't work
                Try
                    board_num = Integer.Parse(name.Substring(name.IndexOf("Board") + "Board".Length, 1))
                Catch
                    board_num = 0   ' todo - parsing the checkbox names, will not have a board number
                End Try
            End Try

            board_num = board_num - 1 ' number found. convert to index to account for arrays being 0 indexed
        End If

        ' Then get the details for that signal type
        If (type = Graphed_Signal.Signal_Type.Stack_Summary) Or (type = Graphed_Signal.Signal_Type.Board_Summary) Then

            If name.IndexOf("SumCells") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Sum
            ElseIf name.IndexOf("Ave") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Ave
            ElseIf name.IndexOf("MaxCell") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Max_Cell
            ElseIf name.IndexOf("MinCell") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Min_Cell
            ElseIf name.IndexOf("MaxTemp") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Max_Temp
            ElseIf name.IndexOf("MinTemp") >= 0 Then
                other_num = Graphed_Signal.Summary_Signal_SubType.Min_Temp
            Else
                type = Graphed_Signal.Signal_Type.Unassigned    ' failed to find any known subtype
            End If

        ElseIf (type = Graphed_Signal.Signal_Type.Cell_Voltage) Then

            ' First try parsing a 2 digit number
            Try
                other_num = Integer.Parse(name.Substring(name.IndexOf("Cell") + "Cell".Length, 2))
            Catch
                ' Try parsing a 1 digit number if that didn't work
                Try
                    other_num = Integer.Parse(name.Substring(name.IndexOf("Cell") + "Cell".Length, 1))
                Catch ex As Exception
                    D33001.Handle_Exception(ex)
                    type = Graphed_Signal.Signal_Type.Unassigned    ' failed to find a valid cell number
                End Try
            End Try

            other_num = other_num - 1 ' number found. convert to index to account for arrays being 0 indexed

        ElseIf (type = Graphed_Signal.Signal_Type.Temperature) Then

            ' First try parsing a 2 digit number
            Try
                other_num = Integer.Parse(name.Substring(name.IndexOf("Temp") + "Temp".Length, 2))
            Catch
                ' Try parsing a 1 digit number if that didn't work
                Try
                    other_num = Integer.Parse(name.Substring(name.IndexOf("Temp") + "Temp".Length, 1))
                Catch ex As Exception
                    D33001.Handle_Exception(ex)
                    type = Graphed_Signal.Signal_Type.Unassigned    ' failed to find a valid temperature number
                End Try
            End Try
            other_num = other_num - 1 ' number found. convert to index to account for arrays being 0 indexed
        Else
            board_num = 0
            other_num = 0
        End If

        If type <> Graphed_Signal.Signal_Type.Unassigned Then
            Return True
        Else
            Return False
        End If

    End Function

End Class
