﻿Public Class Data_Log
    Public Const DEFAULT_INTERVAL As Integer = 1
    Public Const DEFAULT_LENGTH As Integer = 30

    Public Const NUM_PRINTED_FIELDS As Integer = 3  ' the time, value, and balancing are written to/read from the CSV files
    Public Const HEADER_SIZE As Integer = 200       ' Estimated
    Public Const TIMESTAMP_FORMAT As String = "0.00"
    Public Const VOLTAGE_FORMAT As String = "0.0000"
    Public Const BALANCE_FORMAT As String = "0000"  ' For now, 1 if any cell balancing and 0 if none.  todo - this should indicate if charging/discharging, and if supplying/receiving balance currents.
    Public Const TEMPERATURE_FORMAT As String = "0000"
    Public Const TIMEDATE As Integer = 20

    Public interval As Integer     ' in s
    Public length As Integer       ' in s
    Public logging As Boolean

    Public filename As String
    Public num_datapoints As Long
    Public datapoint_count As Integer

    Private timestamp_start As Single

    Public Sub New()
        Me.filename = ""
        Me.interval = DEFAULT_INTERVAL
        Me.length = DEFAULT_LENGTH
        Me.logging = False
    End Sub

    Public Function Update_Size(ByVal interval As Integer, ByVal length As Integer) As Long

        Dim memUsed As Long

        Try
            Me.interval = interval
            Me.length = length

            Me.num_datapoints = CInt(length / interval)
            memUsed = HEADER_SIZE
            memUsed += TIMEDATE * D33001.MAX_CELLS * D33001.numBoards * Me.num_datapoints
            memUsed += (TIMESTAMP_FORMAT.Length + VOLTAGE_FORMAT.Length + BALANCE_FORMAT.Length) * D33001.MAX_CELLS * D33001.numBoards * Me.num_datapoints
            memUsed += (TIMESTAMP_FORMAT.Length + TEMPERATURE_FORMAT.Length + BALANCE_FORMAT.Length) * D33001.NUM_TEMPS * D33001.numBoards * Me.num_datapoints

        Catch ex As Exception
            D33001.Handle_Exception(ex)
            memUsed = 0
        End Try

        Return memUsed

    End Function

    Public Sub Start()

        Try
            ' Find the minimum timestamp at which data collection begun.
            Dim min_timestamp As Single = D33001.voltage_timestamp(0).time
            For board_num As Integer = 0 To D33001.numBoards - 1
                min_timestamp = Math.Min(min_timestamp, D33001.voltage_timestamp(board_num).time)
                min_timestamp = Math.Min(min_timestamp, D33001.temperature_timestamp(board_num).time)
            Next board_num
            Me.timestamp_start = min_timestamp

            ' Clear the datapoint counter
            datapoint_count = 0

            ' Create the header
            Me.Create_Header()

            ' Take the first set of data
            Me.Add_Data()

            ' Start the timer to continue taking datapoints at the specified interval
            Me.logging = True
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Public Sub Start(ByVal filename As String, ByVal interval As Integer, ByVal length As Integer)
        Me.filename = filename
        Me.Update_Size(interval, length)
        Me.Start()
    End Sub

    Public Sub Stop_Log()
        Me.logging = False
    End Sub

    Public Sub Create_Header()

        Try
            Dim textline As String
            Dim writer As System.IO.StreamWriter
            ' Open the writer.  Overwrite old file.
            writer = New System.IO.StreamWriter(Me.filename, False)

            ' Save the filename, date, time, interval, and length
            writer.WriteLine(Me.filename)
            writer.WriteLine("Date Started:," + String.Format("{0:MM/dd/yyyy}", DateTime.Now))
            writer.WriteLine("Time Started:," + String.Format("{0:HH:mm}", DateTime.Now))
            Dim time_interval As Integer = Me.interval
            Dim time_interval_units As String = ""
            D33001.Get_Time_From_Seconds(time_interval, time_interval_units)
            writer.WriteLine("Logging Interval:," + time_interval.ToString() + "," + time_interval_units)
            Dim time_remaining As Long = (Me.num_datapoints - Me.datapoint_count) * Me.interval
            Dim time_remaining_units As String = ""
            D33001.Get_Time_From_Seconds(time_remaining, time_remaining_units)
            writer.WriteLine("Logging Length:," + time_remaining.ToString() + "," + time_remaining_units)

            ' Next save the cell voltage signals present in this data
            writer.Write("Datapoint,")
            For board_num As Integer = 0 To D33001.numBoards - 1
                For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                    ' Write each signal multiple times, as there will be a column for time and value
                    textline = ""
                    For field_num As Integer = 0 To Graphed_Signal.NUM_PRINTED_FIELDS - 1
                        textline += "Board " + (board_num + 1).ToString() + "Cell " + (cell_num + 1).ToString() + ","
                    Next field_num
                    writer.Write(textline)
                Next cell_num
            Next board_num

            ' Next save the temperature signals present in this data
            For board_num As Integer = 0 To D33001.numBoards - 1
                For temp_num As Integer = 0 To D33001.NUM_TEMPS - 1
                    ' Write each signal multiple times, as there will be a column for time and value
                    textline = ""
                    For field_num As Integer = 0 To Graphed_Signal.NUM_PRINTED_FIELDS - 1
                        textline += "Board " + (board_num + 1).ToString() + "Temp " + (temp_num + 1).ToString() + ","
                    Next field_num
                    writer.Write(textline)
                Next temp_num
            Next board_num
            writer.WriteLine("")

            ' Next save the headings for the cell voltage data 
            writer.Write("Number,")
            For board_num As Integer = 0 To D33001.numBoards - 1
                textline = "time (s),Voltage (V),Balancing (1=On/0=Off),"
                For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                    writer.Write(textline)
                Next cell_num
            Next board_num

            ' Finally save the headings for the temperature data 
            For board_num As Integer = 0 To D33001.numBoards - 1
                textline = "time (s),Temp (°C),Balancing (1=On/0=Off),"
                For temp_num As Integer = 0 To D33001.NUM_TEMPS - 1
                    writer.Write(textline)
                Next temp_num
            Next board_num
            writer.WriteLine("")

            writer.Flush()
            writer.Close()

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Public Sub Add_Data()
        Try
            Dim textline As String
            Dim writer As System.IO.StreamWriter

            ' Open the writer.  Append to file.
            writer = System.IO.File.AppendText(Me.filename)

            ' Increment the datapoint count, and add to the file.
            Me.datapoint_count += 1
            writer.Write(Me.datapoint_count.ToString() + ",")

            ' Save the cell voltage data 
            For board_num As Integer = 0 To D33001.numBoards - 1
                For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                    textline = (D33001.voltage_timestamp(board_num).time - Me.timestamp_start).ToString(TIMESTAMP_FORMAT) + ","
                    textline += D33001.Voltages(board_num, cell_num).ToString(VOLTAGE_FORMAT) + ","
                    textline += (-CInt(D33001.voltage_timestamp(board_num).is_balancing)).ToString(BALANCE_FORMAT) + ","
                    writer.Write(textline)
                Next cell_num
            Next board_num

            ' Save the temperature data 
            For board_num As Integer = 0 To D33001.numBoards - 1
                For temp_num As Integer = 0 To D33001.NUM_TEMPS - 1
                    textline = (D33001.temperature_timestamp(board_num).time - Me.timestamp_start).ToString(TIMESTAMP_FORMAT) + ","
                    textline += D33001.Temperatures(board_num, temp_num).ToString(TEMPERATURE_FORMAT) + ","
                    textline += (-CInt(D33001.temperature_timestamp(board_num).is_balancing)).ToString(BALANCE_FORMAT) + ","
                    writer.Write(textline)
                Next temp_num
            Next board_num
            writer.WriteLine("")

            writer.Flush()
            writer.Close()

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try
    End Sub

End Class
