﻿' Cell Balancer Class contains the balance action and balance time for a cell balancer.
' * One of the annoying things that it tries to manage, is that there are multiple copies of the balance actions and balance times:
'   1. The GUI display has radio buttons and text boxes indicating the balance action and balance time.
'   2. The FW has values stored for the balance action and balance time, which will be different from the GUI display until buttons are pushed and USB comms are sent to make the equal.
'   3. Each cell balancer has a balance action and balance time even if it is not currently displayed in the GUI.
' * Another one of the annoying things that it tries to manage, is that the balance actions can be set through different interfaces:
'   1. The balance action can be set directly by a human, as they manipulate the controls that also serve as indicators.
'   2. They can be set by Configuration Loads and Demo buttons that essentially act as a human manipulating all of the controls quickly.
'   3. They can be set by the FW, such as when it first connects and the GUI should be displaying what the board is actually doing.
'   4. Another case of being set by the FW, is as the FW is counting down balance times.  This is only going to become more important as the FW starts implementing its own balance algorithms.
Public Class Cell_Balancer

    Public Const BALANCE_TIME_RESOLUTION As Single = 0.25 ' s per bit
    Public Const BALANCE_TIME_MAX As Single = ((1 << 15) - 1) * BALANCE_TIME_RESOLUTION ' 15 bit time
    Public Const BALANCE_TIME_FORMAT As String = "0.00"
    Public Const BALANCE_COMMAND_SIZE As Integer = 2

    Public DISABLED_COLOR As System.Drawing.Color = Drawing.Color.Silver
    Public ENABLED_COLOR As System.Drawing.Color = Drawing.Color.White
    Public CHANGED_COLOR As System.Drawing.Color = Drawing.Color.LightGray

    ' The actions allowed for a balancer.  These values are different from the actual balancing comand codes used by the LTC3300.
    Public Enum Balance_Action
        None
        Discharge
        Charge
        IC_Error
    End Enum

    ' Controls and Indicators for this Cell
    Private none_radio_button As System.Windows.Forms.RadioButton
    Private discharge_radio_button As System.Windows.Forms.RadioButton
    Private charge_radio_button As System.Windows.Forms.RadioButton
    Private write_command_text_box(2) As System.Windows.Forms.TextBox
    Private read_command_text_box(2) As System.Windows.Forms.TextBox
    Private status_text_box As System.Windows.Forms.TextBox
    Private balance_timer_text_box As System.Windows.Forms.TextBox

    ' Copies of values written to the FW, so that the GUI can detect when they're changed and color the GUI appropriately.
    Public write_balance_action_fw As Balance_Action
    Public balance_time_fw As Single

    ' Copies of values that would be stored in GUI, if this board was the selected board.  They're copied to the indicators when it's the right time to do so.
    Public write_balance_action_copy As Balance_Action
    Public read_balance_action_copy As Balance_Action
    Public read_status_copy As Boolean
    Public balance_time_copy As Single

    Public radio_button_eventhandler As EventHandler

    Public Sub New(ByVal cell_num As Integer, ByVal balance_timer_text_box As System.Windows.Forms.TextBox, _
                   ByVal none_radio_button As System.Windows.Forms.RadioButton, _
                   ByVal discharge_radio_button As System.Windows.Forms.RadioButton, _
                   ByVal charge_radio_button As System.Windows.Forms.RadioButton,
                   ByVal write_command_text_box_bit1 As System.Windows.Forms.TextBox,
                   ByVal write_command_text_box_bit0 As System.Windows.Forms.TextBox,
                   ByVal read_command_text_box_bit1 As System.Windows.Forms.TextBox,
                   ByVal read_command_text_box_bit0 As System.Windows.Forms.TextBox,
                   ByVal status_text_box As System.Windows.Forms.TextBox,
                   ByRef radio_button_eventhandler As EventHandler)

        Me.balance_timer_text_box = balance_timer_text_box
        Me.none_radio_button = none_radio_button
        Me.discharge_radio_button = discharge_radio_button
        Me.charge_radio_button = charge_radio_button
        Me.write_command_text_box(0) = write_command_text_box_bit0
        Me.write_command_text_box(1) = write_command_text_box_bit1
        Me.read_command_text_box(0) = read_command_text_box_bit0
        Me.read_command_text_box(1) = read_command_text_box_bit1
        Me.status_text_box = status_text_box

        Me.radio_button_eventhandler = radio_button_eventhandler
    End Sub

    ' Init the GUI display for this balancer
    ' If initializing GUI for a new system, such as when systems are detatched and reattached, clear everything
    ' If initializing GUI because we are selecting a new board in a system, temporarily blank out data so that it will be repopulated by USB data (if it comes)
    Public Sub Init(Optional ByVal Clear As Boolean = True, Optional ByVal Timed As Boolean = False)

        ' Turn off radio button event handler as we initialize the radio buttons as it looks ridiculous when the handler fires 24 times as we switch boards.
        Me.Disable_Handler()

        If Clear = True Then
            ' If clearing, reset all of balancing data and the GUI displaying the data
            Me.write_balance_action_fw = Balance_Action.None
            Me.balance_time_fw = 0

            Me.write_balance_action_copy = Balance_Action.None
            Me.read_balance_action_copy = Balance_Action.None
            Me.read_status_copy = False
            Me.balance_time_copy = 0

            Me.none_radio_button.Checked = True
            Me.discharge_radio_button.Checked = False
            Me.charge_radio_button.Checked = False

            Me.write_command_text_box(0).Text = ""
            Me.write_command_text_box(1).Text = ""
            Me.read_command_text_box(0).Text = ""
            Me.read_command_text_box(1).Text = ""
            Me.status_text_box.Text = ""

            If Timed = True Then
                Me.balance_timer_text_box.Text = BALANCE_TIME_FORMAT
            Else
                Me.balance_timer_text_box.Text = ""
            End If
        Else
            ' If switching boards, set the GUI for the write command to the last written data, but clear the GUI 
            ' for the read command, status, and balancer timers as that data will be refreshed by the FW
            If Me.write_balance_action_copy = Balance_Action.None Then
                Me.none_radio_button.Checked = True
            ElseIf Me.write_balance_action_copy = Balance_Action.Discharge Then
                Me.discharge_radio_button.Checked = True
            Else
                Me.charge_radio_button.Checked = True
            End If

            Me.read_command_text_box(0).Text = ""
            Me.read_command_text_box(1).Text = ""
            Me.status_text_box.Text = ""
            Me.balance_timer_text_box.Text = ""
        End If

        Me.Enable_Handler()

    End Sub

    Public Sub Enable_Handler()
        AddHandler Me.none_radio_button.CheckedChanged, Me.radio_button_eventhandler
        AddHandler Me.discharge_radio_button.CheckedChanged, Me.radio_button_eventhandler
        AddHandler Me.charge_radio_button.CheckedChanged, Me.radio_button_eventhandler
    End Sub

    Public Sub Disable_Handler()
        RemoveHandler Me.none_radio_button.CheckedChanged, Me.radio_button_eventhandler
        RemoveHandler Me.discharge_radio_button.CheckedChanged, Me.radio_button_eventhandler
        RemoveHandler Me.charge_radio_button.CheckedChanged, Me.radio_button_eventhandler
    End Sub

    ' Enabled the balancer controls for an unpopulated cell.  Timer only enabled if cell is populated.
    Public Sub Enable(Optional ByVal Timed As Boolean = False)
        Me.none_radio_button.Enabled = True
        Me.discharge_radio_button.Enabled = True
        Me.charge_radio_button.Enabled = True

        Me.write_command_text_box(0).BackColor = ENABLED_COLOR
        Me.write_command_text_box(1).BackColor = ENABLED_COLOR
        Me.read_command_text_box(0).BackColor = ENABLED_COLOR
        Me.read_command_text_box(1).BackColor = ENABLED_COLOR
        Me.status_text_box.BackColor = ENABLED_COLOR

        If Timed = True Then
            Me.balance_timer_text_box.Enabled = True
            Me.balance_timer_text_box.BackColor = ENABLED_COLOR
        Else
            Me.balance_timer_text_box.Enabled = False
            Me.balance_timer_text_box.BackColor = DISABLED_COLOR
        End If
    End Sub

    ' Disables the balancer controls for an unpopulated cell
    Public Sub Disable()
        Me.none_radio_button.Enabled = False
        Me.discharge_radio_button.Enabled = False
        Me.charge_radio_button.Enabled = False

        Me.write_command_text_box(0).BackColor = DISABLED_COLOR
        Me.write_command_text_box(1).BackColor = DISABLED_COLOR
        Me.read_command_text_box(0).BackColor = DISABLED_COLOR
        Me.read_command_text_box(1).BackColor = DISABLED_COLOR
        Me.status_text_box.BackColor = DISABLED_COLOR

        Me.balance_timer_text_box.Enabled = False
        Me.balance_timer_text_box.BackColor = DISABLED_COLOR

        Me.write_balance_action_fw = Balance_Action.None
    End Sub

    ' Gets the write action set by the human controlling the GUI
    Public Function Get_Write_Action() As Balance_Action
        Return Me.write_balance_action_copy
    End Function

    ' Copy the LTC3300 write balancer command bits into the balancer boxes, and mark color to indicate if they're different compared to what's stored in the FW
    Public Sub Update_Write_Action_GUI(ByVal bits As Integer)

        ' Set the radio buttons with Balance Action codes
        If Me.write_balance_action_copy = Balance_Action.None Then
            Me.none_radio_button.Checked = True
        ElseIf Me.write_balance_action_copy = Balance_Action.Discharge Then
            Me.discharge_radio_button.Checked = True
        Else
            Me.charge_radio_button.Checked = True
        End If

        ' Set the text boxes with LTC3300 Balance Command codes
        Me.write_command_text_box(0).Text = (bits >> 0 And &H1).ToString()
        Me.write_command_text_box(1).Text = (bits >> 1 And &H1).ToString()

        ' Indicate with color if values still need to be written to the FW
        If write_balance_action_fw <> Get_Write_Action() Then
            Me.write_command_text_box(0).BackColor = CHANGED_COLOR
            Me.write_command_text_box(1).BackColor = CHANGED_COLOR
        Else
            Me.write_command_text_box(0).BackColor = ENABLED_COLOR
            Me.write_command_text_box(1).BackColor = ENABLED_COLOR
        End If

    End Sub

    ' Asks the balancer that its write action has been written to the FW
    Public Function Is_Write_Action_Set() As Boolean
        If write_balance_action_fw = Get_Write_Action() Then
            Return True
        Else
            Return False
        End If
    End Function

    ' Tells the balancer that its write action has been written to the FW
    Public Sub Write_Action_Set()
        write_balance_action_fw = Get_Write_Action()
    End Sub

    ' Sets the write command for this balancer
    Public Sub Set_Write_Action(ByVal balance_action As Balance_Action)
        Me.write_balance_action_copy = balance_action
    End Sub

    ' Sets the write command for this balancer from the radio buttons
    Public Sub Human_Set_Write_Action()
        If Me.none_radio_button.Checked = True Then
            Me.write_balance_action_copy = Balance_Action.None
        ElseIf Me.discharge_radio_button.Checked = True Then
            Me.write_balance_action_copy = Balance_Action.Discharge
        Else
            Me.write_balance_action_copy = Balance_Action.Charge
        End If
    End Sub

    ' Sets the read command for this balancer
    Public Sub Set_Read_Action(ByVal balance_action As Balance_Action)
        Me.read_balance_action_copy = balance_action
    End Sub

    ' Sets the status for this balancer
    Public Sub Set_Status(ByVal status As Boolean)
        Me.read_status_copy = status
    End Sub

    ' Copy the LTC3300 read balancer command bits into the balancer boxes
    Public Sub Update_Read_Action_GUI(ByVal bits As Integer)
        Me.read_command_text_box(0).Text = (bits >> 0 And &H1).ToString()
        Me.read_command_text_box(1).Text = (bits >> 1 And &H1).ToString()
    End Sub

    ' Copy the LTC3300 read balancer command bits into the balancer boxes
    Public Sub Update_Status_GUI(ByVal bits As Integer)
        Me.status_text_box.Text = (bits >> 0 And &H1).ToString()
    End Sub

    ' Bounds checks the values in the balancer timer text boxes, and marks color to indicate if they're different compared to what's stored in the FW
    Public Sub Human_Set_Balance_Time()
        Dim new_time As Single

        Try
            new_time = CSng(balance_timer_text_box.Text)

            ' Bound at upper and lower values
            If new_time > BALANCE_TIME_MAX Then
                new_time = BALANCE_TIME_MAX
            ElseIf new_time < 0 Then
                new_time = 0
            End If

            ' Round to nearest value
            If ((new_time Mod BALANCE_TIME_RESOLUTION) < BALANCE_TIME_RESOLUTION / 2) Then
                new_time = new_time - (new_time Mod BALANCE_TIME_RESOLUTION)
            Else
                new_time = new_time - (new_time Mod BALANCE_TIME_RESOLUTION) + BALANCE_TIME_RESOLUTION
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)

            new_time = 0
        End Try

        Me.balance_time_copy = new_time.ToString(BALANCE_TIME_FORMAT)

        If Me.balance_time_fw <> new_time Then
            Me.balance_timer_text_box.BackColor = CHANGED_COLOR
        End If

    End Sub

    ' Tells the balancer that its balancer timer and write action has been written to the FW
    Public Function Get_Balance_Time() As Single
        Return Me.balance_time_copy
    End Function

    ' Tells the balancer that its balancer timer has been written to the FW
    Public Sub Balance_Time_Set()
        Me.balance_time_fw = Me.balance_time_copy
    End Sub

    ' Sets the balancer timer for this balancer
    Public Sub Set_Balance_Time(ByVal Timer As Single)
        Me.balance_time_copy = Timer
    End Sub

    ' Asks the balancer if its balance time has been written to the FW
    Public Function Is_Balance_Time_Set() As Boolean
        If Me.balance_time_fw = Me.balance_time_copy Then
            Return True
        Else
            Return False
        End If
    End Function

    ' Copy the balance time into the text boxes, and mark color to indicate if they're different compared to what's stored in the FW
    Public Sub Update_Balance_Time_GUI()

        Me.balance_timer_text_box.Text = Me.balance_time_copy.ToString(BALANCE_TIME_FORMAT)

        ' Indicate with color if values still need to be written to the FW
        If Me.balance_time_fw <> Me.balance_time_copy Then
            Me.balance_timer_text_box.BackColor = CHANGED_COLOR
        Else
            Me.balance_timer_text_box.BackColor = ENABLED_COLOR
        End If

    End Sub

End Class

' LTC3300 Class contains the methods and supplementary bits associated with translating the balance actions of 6 cell balancers into balance
' commands for the LTC3300 Battery Balancer IC.
Public Class LTC3300
    Public Const NUM_CELLS As Integer = 6
    Public Const CRC_SIZE As Integer = 4
    Public Const BALANCE_COMMAND_SIZE As Integer = 16
    Public Const STATUS_UNUSED_SIZE As Integer = 3
    Public Enabled As Boolean
    Private ic_num As Integer
    Private balancer(NUM_CELLS) As Cell_Balancer
    Private write_command_crc_text_box(CRC_SIZE) As System.Windows.Forms.TextBox
    Private read_command_crc_text_box(CRC_SIZE) As System.Windows.Forms.TextBox
    Private read_read_status_crc_copy_text_box(CRC_SIZE) As System.Windows.Forms.TextBox
    Private read_status_cells_not_ov_copy_text_box As System.Windows.Forms.TextBox
    Private read_status_stack_not_ov_copy_text_box As System.Windows.Forms.TextBox
    Private read_status_temp_ok_copy_text_box As System.Windows.Forms.TextBox
    Private read_status_unused_copy_text_box(3) As System.Windows.Forms.TextBox

    ' Copies of values that would be stored in GUI, if this board was the selected board.  They're copied to the indicators when it's the right time to do so.
    Private read_balance_crc_copy As Integer
    Private read_status_crc_copy As Integer
    Private read_status_cells_not_ov_copy As Boolean
    Private read_status_stack_not_ov_copy As Boolean
    Private read_status_temp_ok_copy As Boolean
    Private read_status_unused_copy(STATUS_UNUSED_SIZE) As Boolean

    ' None of the status bits are valid unless the LTC3300 is attempting to execute a balance command.
    Private status_valid As Boolean

    ' These are the commands used to write/read balancing command and read status registers to/from the LTC3300.
    Public Enum Command
        Write_Balance = &HA9
        Read_Balance = &HAA
        Read_Status = &HAC
        Execute = &HAF
        Suspend = &HAE
    End Enum

    ' These are the from the actual balancing comand codes used by the LTC3300.  These values are different from the balancing actions defined by the radio buttons for each balancer.
    Public Enum Balance_Command
        None = &H0  ' Balancing Action: None
        Discharge_Sync = &H1 ' Balancing Action: Discharge Cell n (Nonsynchronous)
        Discharge_Nonsync = &H2 ' Balancing Action: Discharge Cell n (Synchronous)
        Charge = &H3 ' Balancing Action: Charge Cell n
    End Enum

    Public Sub New(ByVal ic_num As Integer, ByVal balancer() As Cell_Balancer, _
                   ByVal write_command_crc_text_box() As System.Windows.Forms.TextBox,
                   ByVal read_command_crc_text_box() As System.Windows.Forms.TextBox,
                   ByVal read_read_status_crc_copy_text_box() As System.Windows.Forms.TextBox,
                   ByVal ltc_read_status_crc_copy_text_box() As System.Windows.Forms.TextBox)
        Me.Enabled = True
        Me.ic_num = ic_num
        Me.balancer = balancer
        Me.write_command_crc_text_box = write_command_crc_text_box
        Me.read_command_crc_text_box = read_command_crc_text_box
        Me.read_read_status_crc_copy_text_box = read_read_status_crc_copy_text_box
        Me.read_status_cells_not_ov_copy_text_box = ltc_read_status_crc_copy_text_box(0)
        Me.read_status_stack_not_ov_copy_text_box = ltc_read_status_crc_copy_text_box(1)
        Me.read_status_temp_ok_copy_text_box = ltc_read_status_crc_copy_text_box(2)
        Me.read_status_unused_copy_text_box(0) = ltc_read_status_crc_copy_text_box(3)
        Me.read_status_unused_copy_text_box(1) = ltc_read_status_crc_copy_text_box(4)
        Me.read_status_unused_copy_text_box(2) = ltc_read_status_crc_copy_text_box(5)
        Me.status_valid = False
    End Sub

    ' Init the GUI display for this LTC3300
    ' If initializing GUI for a new system, such as when systems are detatched and reattached, clear everything
    ' If initializing GUI because we are selecting a new board in a system, temporarily blank out data so that it will be repopulated by USB data (if it comes)
    Public Sub Init(Optional ByVal Clear As Boolean = True, Optional ByVal Timed As Boolean = False)

        For cell_num As Integer = 0 To NUM_CELLS - 1
            Me.balancer(cell_num).Init(Clear, Timed)
        Next cell_num

        If Clear = True Then
            For crc_bit As Integer = 0 To CRC_SIZE - 1
                write_command_crc_text_box(crc_bit).Text = ""
            Next crc_bit
        Else
            Me.Update_Write_Command_GUI()
        End If

    End Sub

    Private Function Convert_Balance_Command_To_Balance_Action(ByVal balance_command As Balance_Command) As Cell_Balancer.Balance_Action
        Dim balance_action As Cell_Balancer.Balance_Action

        If balance_command = balance_command.None Then
            balance_action = Cell_Balancer.Balance_Action.None
        ElseIf balance_command = balance_command.Discharge_Nonsync Then
            balance_action = Cell_Balancer.Balance_Action.Discharge
        ElseIf balance_command = balance_command.Charge Then
            balance_action = Cell_Balancer.Balance_Action.Charge
        End If

        Return balance_action

    End Function

    Private Function Convert_Balance_Action_To_Balance_Command(ByVal balance_action As Cell_Balancer.Balance_Action) As Balance_Command
        Dim balance_command As Balance_Command

        If balance_action = Cell_Balancer.Balance_Action.None Then
            balance_command = balance_command.None
        ElseIf balance_action = Cell_Balancer.Balance_Action.Discharge Then
            balance_command = balance_command.Discharge_Nonsync
        ElseIf balance_action = Cell_Balancer.Balance_Action.Charge Then
            balance_command = balance_command.Charge
        End If

        Return balance_command

    End Function

    Public Function Is_Write_Command_Set() As Boolean
        For cell_num As Integer = 0 To NUM_CELLS - 1
            If Me.balancer(cell_num).Is_Write_Action_Set() = False Then
                Return False
            End If
        Next cell_num
        Return True
    End Function

    Public Function Get_Write_Command() As String     'Creates appropriate hex value string to be used in Write command
        Dim balance_command As Integer = 0

        For cell_num As Integer = 0 To NUM_CELLS - 1
            balance_command = balance_command << Cell_Balancer.BALANCE_COMMAND_SIZE
            balance_command += Me.Convert_Balance_Action_To_Balance_Command(Me.balancer(cell_num).Get_Write_Action())
        Next cell_num

        For crc_bit As Integer = CRC_SIZE - 1 To 0 Step -1
            balance_command <<= 1
            balance_command += Integer.Parse(write_command_crc_text_box(crc_bit).Text)
        Next crc_bit

        Return balance_command.ToString("X4")
    End Function

    Public Function Get_Error() As Boolean     'Returns True if any errors are indicated by the status register
        If Me.status_valid = False Then
            Return False
        Else
            Return Not (Me.read_status_cells_not_ov_copy And Me.read_status_stack_not_ov_copy And Me.read_status_temp_ok_copy)
        End If
    End Function

    'Validates and stores the data for LTC3300 read balance command or status register
    Public Sub Set_Read_Register(ByVal command As Command, ByVal register As Integer)     'Sets the received value of the Read command and verifies the CRC
        Dim crc_read As Integer
        Dim crc_check As Integer

        crc_read = register And ((1 << CRC_SIZE) - 1)
        crc_check = calculateCRC(register >> CRC_SIZE)

        ' If CRC does not check, flag an error
        If (crc_read <> crc_check) Then
            D33001.Error_File.Add(Error_File.Error_Code.LTC3300_CRC, register.ToString("X4") & " : calculated CRC =" & crc_check.ToString("X2"))
        Else
            ' If CRC does check, save the value for this balancer
            If (command = command.Read_Balance) Then

                ' Save the CRC for display in GUI
                Me.read_balance_crc_copy = crc_read
                register >>= CRC_SIZE

                ' Save the balance command for display in GUI
                For cell_num As Integer = NUM_CELLS - 1 To 0 Step -1
                    Dim balancer_command As Balance_Command
                    balancer_command = register And ((1 << Cell_Balancer.BALANCE_COMMAND_SIZE) - 1)
                    Me.balancer(cell_num).Set_Read_Action(Convert_Balance_Command_To_Balance_Action(balancer_command))
                    register >>= Cell_Balancer.BALANCE_COMMAND_SIZE
                Next cell_num

            ElseIf (command = command.Read_Status) Then

                ' Save the CRC for display in GUI
                Me.read_status_crc_copy = crc_read
                register >>= CRC_SIZE

                ' todo - This is a hack to determine if any of the status bits should be trusted.
                '        Basically none of the status bits are valid unless a balance command is being executed, and then the Gate OK bits aren't valid unless
                '        that particular cell is balancing.  Mark V suggested that using the presence of any bits being set in the status register could indicate
                '        that a balance command is being executed, as opposed to keeping track of this in FW.
                If register = 0 Then
                    Me.status_valid = False
                Else
                    Me.status_valid = True
                End If

                ' Save the status for display in GUI
                For bit_num As Integer = 0 To STATUS_UNUSED_SIZE - 1
                    If register And 1 Then
                        Me.read_status_unused_copy(bit_num) = True
                    Else
                        Me.read_status_unused_copy(bit_num) = False
                    End If
                    register >>= 1
                Next bit_num

                If register And 1 Then
                    Me.read_status_temp_ok_copy = True
                Else
                    Me.read_status_temp_ok_copy = False
                End If
                register >>= 1

                If register And 1 Then
                    Me.read_status_stack_not_ov_copy = True
                Else
                    Me.read_status_stack_not_ov_copy = False
                End If
                register >>= 1

                If register And 1 Then
                    Me.read_status_cells_not_ov_copy = True
                Else
                    Me.read_status_cells_not_ov_copy = False
                End If
                register >>= 1

                For cell_num As Integer = NUM_CELLS - 1 To 0 Step -1
                    If register And 1 Then
                        Me.balancer(cell_num).Set_Status(True)
                    Else
                        Me.balancer(cell_num).Set_Status(False)
                    End If
                    register >>= 1
                Next cell_num
            Else
                D33001.Error_File.Add(Error_File.Error_Code.LTC3300_CRC, "Unknown command = " & command.ToString("X2"))
            End If
        End If
    End Sub

    'Updates the GUI display of LTC3300 read balance command or status register
    Public Sub Update_Register_GUI(ByVal command As Command)
        ' If CRC does check, save the value for this balancer
        If (command = command.Read_Balance) Then

            For cell_num As Integer = 0 To NUM_CELLS - 1
                command = Me.Convert_Balance_Action_To_Balance_Command(Me.balancer(cell_num).read_balance_action_copy)
                Me.balancer(cell_num).Update_Read_Action_GUI(command)
            Next cell_num

            For crc_bit As Integer = 0 To CRC_SIZE - 1
                read_command_crc_text_box(crc_bit).Text = ((Me.read_balance_crc_copy >> crc_bit) And 1).ToString()
            Next crc_bit
        ElseIf (command = command.Read_Status) Then

            For cell_num As Integer = 0 To NUM_CELLS - 1
                Me.balancer(cell_num).Update_Status_GUI(Me.balancer(cell_num).read_status_copy)
            Next cell_num

            If Me.read_status_cells_not_ov_copy = True Then
                read_status_cells_not_ov_copy_text_box.Text = "1"
            Else
                read_status_cells_not_ov_copy_text_box.Text = "0"
            End If

            If Me.read_status_stack_not_ov_copy = True Then
                read_status_stack_not_ov_copy_text_box.Text = "1"
            Else
                read_status_stack_not_ov_copy_text_box.Text = "0"
            End If

            If Me.read_status_temp_ok_copy = True Then
                read_status_temp_ok_copy_text_box.Text = "1"
            Else
                read_status_temp_ok_copy_text_box.Text = "0"
            End If

            For bit_num As Integer = 0 To STATUS_UNUSED_SIZE - 1
                If Me.read_status_unused_copy(bit_num) = True Then
                    read_status_unused_copy_text_box(bit_num).Text = "1"
                Else
                    read_status_unused_copy_text_box(bit_num).Text = "0"
                End If
            Next bit_num

            For crc_bit As Integer = 0 To CRC_SIZE - 1
                read_read_status_crc_copy_text_box(crc_bit).Text = ((Me.read_status_crc_copy >> crc_bit) And 1).ToString()
            Next crc_bit
        Else
            D33001.Error_File.Add(Error_File.Error_Code.LTC3300_CRC, "Unknown command = " & command.ToString("X2"))
        End If

    End Sub

    'Updates the GUI display of LTC3300 write balance command register
    Public Sub Update_Write_Command_GUI()

        Dim command As Balance_Command

        ' Turn off radio button event handler as we change the radio buttons as it looks ridiculous when the handler fires 24 times as we switch boards.
        For cell_num As Integer = 0 To NUM_CELLS - 1
            Me.balancer(cell_num).Disable_Handler()
        Next cell_num

        ' Update the balance command text boxes and radio buttons
        For cell_num As Integer = 0 To NUM_CELLS - 1
            command = Me.Convert_Balance_Action_To_Balance_Command(Me.balancer(cell_num).Get_Write_Action())
            Me.balancer(cell_num).Update_Write_Action_GUI(command)
        Next cell_num

        ' Update the Command CRC
        Dim balance_command As Integer = 0
        Dim command_crc As Integer

        For cell_num As Integer = 0 To NUM_CELLS - 1
            balance_command <<= Cell_Balancer.BALANCE_COMMAND_SIZE
            balance_command += Me.Convert_Balance_Action_To_Balance_Command(Me.balancer(cell_num).Get_Write_Action())
        Next cell_num

        command_crc = calculateCRC(balance_command)
        For crc_bit As Integer = 0 To CRC_SIZE - 1
            write_command_crc_text_box(crc_bit).Text = ((command_crc >> crc_bit) And 1).ToString()
        Next crc_bit

        For cell_num As Integer = 0 To NUM_CELLS - 1
            Me.balancer(cell_num).Enable_Handler()
        Next cell_num

    End Sub

    Private Function calculateCRC(ByVal balance_command As Integer) As Integer
        Dim dividend As Integer = balance_command
        Dim divisor As Integer = &H980
        Dim CRC As Integer = 0

        CRC = dividend And &HF80

        For i As Integer = 1 To Cell_Balancer.BALANCE_COMMAND_SIZE * NUM_CELLS
            If CRC >= &H800 Then
                CRC = CRC Xor divisor
            End If
            CRC = CRC << 1

            CRC = CRC Or (dividend << i) And &H80
        Next

        CRC = (CRC >> 8)

        CRC = ((&HF - CRC) And &HF)

        Return CRC

    End Function

    ' Combines the read balance command and status to indicate whether the balancer is actually charging, discharging, or in an error state
    Public Function Get_Combined_Balance_Action(ByVal cell_num As Integer) As Cell_Balancer.Balance_Action
        If Me.status_valid = False Then
            ' If status isn't valid, then all of the balancers on this IC are off
            Return Cell_Balancer.Balance_Action.None
        ElseIf balancer(cell_num).read_status_copy = True Then
            ' If status bits for this balancer are true, the the balance action is the current state of the cell.
            Return balancer(cell_num).read_balance_action_copy
        Else
            ' If status bit for this balancer is false, either it has a balance command of None or it's an error.
            If balancer(cell_num).read_balance_action_copy = Cell_Balancer.Balance_Action.None Then
                Return Cell_Balancer.Balance_Action.None
            Else
                Return Cell_Balancer.Balance_Action.IC_Error
            End If
        End If
    End Function


End Class
