﻿Imports Microsoft.VisualBasic
Imports System.Collections.Generic

Public Class CapBoardPopup

    Public ACTIVATED_COLOR As System.Drawing.Color = Drawing.Color.PaleGoldenrod
    Public DEACTIVATED_COLOR As System.Drawing.Color = Drawing.Color.LightGray

    Private Const MAX_SIGNALS As Integer = D33001.MAX_CELLS
    Private Const VOLTAGE_MATCH_DEFAULT As Integer = 25             ' in mV
    Private Const VOLTAGE_MATCH_MIN As Integer = 10                 ' in mV

    Public voltage_match_checked_copy As Boolean = False            ' copy of checkbox setting, so that it can be restored after tests that force it to its own position.
    Public voltage_match_copy As Integer = VOLTAGE_MATCH_DEFAULT    ' copy of value in text box, so that old value can be restored if garbage is entered
    Public charger_on_copy As Boolean                               ' copy of charger state, so that it can be restored after demo stops
    Public discharger_on_copy As Boolean                            ' copy of discharger state, so that it can be restored after demo stops
    Public vmin_copy As Single                                      ' copy of value in text box, so that old value can be restored after demo stops
    Public vmax_copy As Single                                      ' copy of value in text box, so that old value can be restored after demo stops

    Public is_balancing_last As Boolean                             ' copy of whether we were balancing the last time GUI was updated, so that edges can be detected as this value changes.
    Public nonzero_balance_time_last As Boolean                     ' flag set when a non-zero balance time has been calculate.  This accounts for the special case, where the GUI does not receive any voltage samples while balancing is active (since BALANCE_TIME_RESOLUTION < auto_read_counter_period)
    Public stack_charge_last As Single                              ' copy of charge stored in battery stack the last time Cap Demo was updated.
    Public balance_stopped_sent_last As Boolean                     ' todo - this is a hack to prevent us from ever making a "started/stopped" pair because we thought balancing occurred in between voltage samples.
    Public Const CHARGE_FORMAT As String = "0"


    Private CapDemo_Signal_Group As Graphed_Signal_Group

    Public Opened As Boolean = False
    Public Loaded_From_File As Boolean = False

    ' Balance Line Definition
    Private Const MAX_BALANCE_LINES As Integer = 50
    Public Enum Balance_Line_Type
        Balance_Started
        Balance_Started_Voltage_Matching
        Balance_Stopped
        Test
    End Enum
    Public Structure Balance_Line_Data_Structure
        Public number As Integer
        Public name As String
        Public type As Balance_Line_Type
        Public x_coord As Single
        Public text As String
    End Structure
    Public Balance_Line_List As New List(Of Balance_Line_Data_Structure)
    Public Balance_Line_Text_Disable As Boolean                     ' Setting this causes the text above balance lines to be disabled until the condition arises for a Balance_Stopped balance line.

    ' Properties of each operational demo button
    Public Enum Cap_Demo_Button_Type
        Charge
        Discharge
        Stop_Chg_Dchg
        Balance_While_Charging
        Balance_While_Discharging
        Charge_All_Cells
        Discharge_All_Cells
        Alternate_Chg_Dchg
        Cell_7_Chg_Cell_6_Dchg
        Num_Cap_Demo_Buttons
        None_Selected
    End Enum
    Public Structure Cap_Demo_Button_Data_Structure
        Public button As System.Windows.Forms.Button
        Public button_activated_text As String
        Public button_deactivated_text As String
    End Structure
    Public Cap_Demo_Button_Data(Cap_Demo_Button_Type.Num_Cap_Demo_Buttons - 1) As Cap_Demo_Button_Data_Structure
    Public cap_demo_button_activated As Cap_Demo_Button_Type = Cap_Demo_Button_Type.None_Selected

    ' Trade Show Demo Variables
    Private Const TRADE_SHOW_DEMO_MIN_VOLTAGE As Single = 2.0               ' in V, the point at which cells are being damaged due to undervoltage
    Private Const TRADE_SHOW_DEMO_LOW_VOLTAGE As Single = 2.5               ' in V, the point to which cells are considered to have 0% SOC
    Private Const TRADE_SHOW_DEMO_HIGH_VOLTAGE As Single = 3.3              ' in V, the point to which cells are considered to have 100% SOC
    Private Const TRADE_SHOW_DEMO_MAX_VOLTAGE As Single = 4.5               ' in V, the point at which cells are being damaged due to overvoltage

    Private Const TRADE_SHOW_DEMO_FLOAT_VOLTAGE_DELTA As Single = 0.1       ' in V, the stack voltage must change by less than this in order to be considered floating.
    Private Const TRADE_SHOW_DEMO_FLOAT_TIME As Integer = 5000              ' in ms, how long the stack voltage must be changing by less than TRADE_SHOW_DEMO_FLOAT_VOLTAGE_DELTA to be considered floating.

    Private Const TRADE_SHOW_DEMO_BALANCED_VOLTAGE_DELTA As Single = 0.025  ' in V, the max and min cells must be less than this in order to be considered balanced.
    Private Const TRADE_SHOW_DEMO_BALANCED_TIME As Integer = 5000           ' in ms, how long the max and min cells must be less than TRADE_SHOW_DEMO_BALANCED_VOLTAGE_DELTA to be considered balanced.

    Private Const TRADE_SHOW_DEMO_DISCHARGED_TIME As Integer = 1000         ' in ms, how long the min cell must be less than TRADE_SHOW_DEMO_LOW_VOLTAGE to be considered discharged.
    Private Const TRADE_SHOW_DEMO_CHARGED_TIME As Integer = 1000            ' in ms, how long the max cell must be less than TRADE_SHOW_DEMO_HIGH_VOLTAGE to be considered charged.

    Private Const TRADE_SHOW_DELAY_TIME As Integer = 10000                  ' in ms, how long the trade show demo pauses during delay states.

    Private Const TRADE_SHOW_DEMO_TIME_BETWEEN_STATES As Integer = 100      ' in ms, the estimate for the a brief period when we are not performing any action when switching from one mode to the next.

    Public Trade_Show_Demo_Button_Data As Cap_Demo_Button_Data_Structure
    Public Enum Trade_Show_Demo_State_Type
        Idle
        Init_Charging
        Init_Charging_With_Balancing
        Init_Charging_Delay
        Discharge_Without_Balancing
        Discharge_With_Balancing
        Discharge_Delay
        Charge_Without_Balancing
        Charge_With_Balancing
        Charge_Delay
        Num_Trade_Show_Demo_States
    End Enum
    Public trade_show_demo_state As Trade_Show_Demo_State_Type
    Public trade_show_demo_timer As Integer
    Public trade_show_demo_timestamp_last As Single
    Public trade_show_demo_high_voltage_measured As Single                  ' Since each cap demo system is different, the actual voltage at which we float is not nominal
    Public trade_show_demo_display_time_start As Single                     ' Capture the best time to start the trade show demo graph such that it looks good whether caps started at 0V or float voltage.

    Public trade_show_demo_float_voltage_list As New List(Of Single)

    Public trade_show_demo_discharge_time_wo_balancing As Integer
    Public trade_show_demo_discharge_time_w_balancing As Integer
    Public trade_show_demo_charge_wo_balancing As Single
    Public trade_show_demo_charge_w_balancing As Single


#Region "Form Startup and Shutdown"
    Private Sub frmMain_Load(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles MyBase.Load
        Me.Init()
        Me.Opened = True
        Me.Loaded_From_File = False

        ' Enable graphing upon opening window
        Enable_CheckBox.Checked = True

        ' Create objects for Cap Demo Buttons
        ' Set the properties of each operational demo button so that they can be set in a loop

        ' Manual Charge/Discharge Buttons
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge).button = ChgBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge).button_activated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge).button_deactivated_text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge).button = DchgBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge).button_activated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge).button_deactivated_text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Stop_Chg_Dchg).button = NoneBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Stop_Chg_Dchg).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Stop_Chg_Dchg).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Stop_Chg_Dchg).button_activated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Stop_Chg_Dchg).button_deactivated_text

        ' Operational Demos
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button = ChgBalBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button_activated_text = "Suspend"
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button = DchgBalBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button_activated_text = "Suspend"

        ' Imbalance Demos
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button = ChgAllBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button_activated_text = "Suspend"
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button = DchgAllBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button_activated_text = "Suspend"
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button = AltChgBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button_activated_text = "Suspend"
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button = DemoChgBtn
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button_deactivated_text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button.Text
        Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button_activated_text = "Suspend"

        ' Init to reflect that no button is activated
        cap_demo_button_activated = Cap_Demo_Button_Type.None_Selected

        ' Trade Show Demo
        Trade_Show_Demo_Button_Data.button = opDemoBtn
        Trade_Show_Demo_Button_Data.button_deactivated_text = Trade_Show_Demo_Button_Data.button.Text
        Trade_Show_Demo_Button_Data.button_activated_text = "Stop Trade Show Demonstration"

        ' Display what the Cap Demo system is actually doing in the charge and discharge checkboxes
        ChgBox.Checked = D33001.is_charging
        DchgBox.Checked = D33001.is_discharging

        ' Click the button for what the Cap Demo system is actually doing with its charger/discharger
        charger_on_copy = ChgBox.Checked
        discharger_on_copy = DchgBox.Checked

        ' If timed balancing is already underway when window is started, open form with buttons properly enabled/disabled
        If D33001.isTimedBalancing = True Then
            Disable_All_Cap_Demo_Buttons()
            balStartBtn.Enabled = False
            balStopBtn.Enabled = True
            VMBox.Enabled = False
            AlgoBtn.Enabled = False
            is_balancing_last = True
            If D33001.Max_Balance_Time_Value <> 0 Then
                nonzero_balance_time_last = True
            Else
                nonzero_balance_time_last = False
            End If
            balance_stopped_sent_last = False
        Else
            Restore_Charger_and_Discharger_Settings()
            is_balancing_last = False
            nonzero_balance_time_last = False
            balance_stopped_sent_last = False
        End If

        ' Initialize the Trade Show Demo
        trade_show_demo_state = Trade_Show_Demo_State_Type.Idle
        trade_show_demo_timer = 0

        trade_show_demo_high_voltage_measured = TRADE_SHOW_DEMO_HIGH_VOLTAGE
        trade_show_demo_display_time_start = 0

        trade_show_demo_float_voltage_list.Clear()

        trade_show_demo_discharge_time_w_balancing = 0
        trade_show_demo_discharge_time_wo_balancing = 0

    End Sub

    Private Sub frmMain_Closed(ByVal eventSender As System.Object, ByVal eventArgs As System.EventArgs) Handles MyBase.Closed
        Me.Opened = False
    End Sub

    Private Sub CapBoardPopup_FormClosing(ByVal sender As System.Object, ByVal e As System.Windows.Forms.FormClosingEventArgs) Handles MyBase.FormClosing

        ' Disable graphing upon closing window
        Enable_CheckBox.Checked = False

        ' Disable voltage matching upon closing window
        VMBox.Checked = False

        If cap_demo_button_activated <> Cap_Demo_Button_Type.None_Selected Then
            ' Stop any tests that are in progress
            Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

            ' Restore the charger/discharger settings
            Restore_Charger_and_Discharger_Settings()

            ' Stop balancing
            D33001.Timed_Balance_Stop(False)
        End If

    End Sub

    Private Sub CapBoardPopup_Deactivate(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Deactivate
        D33001.CapBoardBtn.Enabled = True
    End Sub

    Private Sub CapBoardPopup_Activated(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Activated
        D33001.CapBoardBtn.Enabled = False
    End Sub

#End Region

#Region "Graph"

    Public Sub Init()
        ' Connect the graph to the signals that are plotted on it
        CapDemo_Signal_Group = New Graphed_Signal_Group(MAX_SIGNALS, CapGraph, HScrollBar, Debug_TextBox)

        ' Create the signals to be graphed
        Me.Setup_Graphed_Signals()

        ' Show debug fields 
        Debug_TextBox.Visible = D33001.DEBUG

        ' Enable graphing upon opening window
        Enable_CheckBox.Checked = False

    End Sub

    Public Sub Reset()

        ' Reset Graph Data
        CapDemo_Signal_Group.Reset()

        ' Reset Balance lines
        Clear_Balance_Lines()
        Balance_Line_Text_Disable = False
        stack_charge_last = Calculate_Stack_Charge(D33001.vMin)

        ' Reset Annotations
        While (CapGraph.Annotations.Count > 0)
            CapGraph.Annotations.RemoveAt(0)
        End While

    End Sub

    ' The equivalent of checking the 12 Cell Voltage CheckBoxes in GraphData.vb, as the Cap Demo will always be looking at the 12 cell voltages and nothing else.
    Public Sub Setup_Graphed_Signals()
        Dim type As Graphed_Signal.Signal_Type = Graphed_Signal.Signal_Type.Cell_Voltage
        Dim board_num As Integer = D33001.selectedBoard

        For cell_num As Integer = 0 To MAX_SIGNALS - 1
            If CapDemo_Signal_Group.Add_Signal(type, D33001.selectedBoard, cell_num) = False Then
                ' todo if this fails, we have a code bug in Get_Num_Unassigned()
            End If
        Next
    End Sub

    Public Sub Update_GUI_Plot(ByVal controls_only As Boolean)

        ' Do nothing if GUI not opened
        If (Me.Opened = False) Then
            Exit Sub
        End If

        ' Update datapoints if still taking data
        If Enable_CheckBox.Checked = True Then
            CapDemo_Signal_Group.Update_Data_Points(True)
        End If

        ' Update the graph
        CapDemo_Signal_Group.Update_GUI(controls_only)

        ' Show balance lines
        If GraphOptionPopup.showBalLines = False Then
            ' Remove all balance lines and annotations if user unchecked the box to show them
            Clear_Balance_Lines_GUI()
        Else
            ' Add all balance lines and annotations if user checked the box to show them
            Add_Balance_Lines_GUI()
        End If

        ' Set Legend
        If GraphOptionPopup.showLegend = True Then
            CapGraph.Legends(0).Enabled = True
        Else
            CapGraph.Legends(0).Enabled = False
        End If

        ' Invalidate chart
        ' todo - the Microsoft example code does this.  What's it for?
        CapGraph.Invalidate()

    End Sub

    Private Sub ScrollBar_Scroll(ByVal sender As System.Object, ByVal e As System.Windows.Forms.ScrollEventArgs) Handles HScrollBar.Scroll

        Try
            CapDemo_Signal_Group.ScrollBar_Scroll()

            If (Me.Enable_CheckBox.Checked = True) Then
                Update_GUI_Plot(True)
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub GraphOptionBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles GraphOptionBtn.Click
        Try
            If GraphOptionPopup.ShowDialog() = DialogResult.OK Then
                Update_GUI_Plot(True)
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try
    End Sub

    Private Sub CapGraph_MouseEnter(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles scrollPanel.MouseEnter
        HScrollBar.Show()
    End Sub

    Private Sub CapGraph_MouseLeave(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles scrollPanel.MouseLeave
        HScrollBar.Hide()
    End Sub

    Private Sub Enable_CheckBox_CheckedChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Enable_CheckBox.CheckedChanged
        Try
            If Enable_CheckBox.Checked = True Then
                ' if re-enabling logging after loading a file, reset the data as it won't make sense with the timestamps for new data (but keep the signals that are being graphed)
                If (Loaded_From_File = True) Then
                    Loaded_From_File = False    ' Flag that this is no longer the loaded data.
                    CapDemo_Signal_Group.Reset(True)
                End If
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try
    End Sub

#End Region

#Region "Data Load/Save"
    Private Sub ClearLogBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ClearLogBtn.Click
        Try
            ' Clear all data
            Me.Reset()

            ' Set up the graphed signals again.
            Me.Setup_Graphed_Signals()
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub SaveLogBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SaveLogBtn.Click
        Try
            SaveLogPopup.Set_Graph_Objects(CapDemo_Signal_Group, CapGraph, SplitContainer1.Panel1)
            SaveLogPopup.ShowDialog()
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try
    End Sub

    'Loads a selected file BUT the way it is done is very inefficient and slow. If I had time to fix it I would read data with a "StreamReader"
    '(for streamreader implementation, see the code for the error log)
    Private Sub LoadLogBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles LoadLogBtn.Click
        Dim open_file_dialog As New OpenFileDialog()
        Dim open_file_name As String

        Try

            open_file_dialog.Filter = "CSV Files (*.csv)|*.csv|All Files (*.*)|*.*"
            open_file_dialog.Title = "Choose File Location"
            If open_file_dialog.ShowDialog() = DialogResult.OK Then
                If open_file_dialog.FileName.Substring(open_file_dialog.FileName.Length - ".csv".Length, ".csv".Length) = ".csv" Then
                    open_file_name = open_file_dialog.FileName
                Else
                    open_file_name = open_file_dialog.FileName + ".csv"
                End If
            Else
                Exit Sub
            End If

            ' Mark data as having been loaded from a file
            Loaded_From_File = True

            ' Do not keep updating graph, as it'll make no sense with the old and new data together
            Enable_CheckBox.Checked = False

            ' Load the data from the input file
            CapDemo_Signal_Group.Input_CSV(open_file_name)

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub
#End Region

#Region "Balance lines"
    Public Sub Add_Balance_Line(ByVal type As Balance_Line_Type, ByVal text As String, Optional ByVal datapoints_back As Integer = 0)

        ' Don't create balance lines if window not opened, or if trade show demo is in progress
        If (Me.Opened = False) Or (Enable_CheckBox.Checked = False) Or (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
            Exit Sub
        End If

        ' if this is a set of lines started with the charger or discharger on, do to make any more lines until a stop condition is detected.
        If ((type = Balance_Line_Type.Balance_Started) Or (type = Balance_Line_Type.Balance_Started_Voltage_Matching)) And _
            ((D33001.is_charging = True) Or (D33001.is_discharging = True)) Then
            Balance_Line_Text_Disable = True
        End If

        ' We can't add an infinite number of lines.  Remove the first one if we try to add too many
        If Balance_Line_List.Count >= MAX_BALANCE_LINES Then
            Dim chart_series_index As Integer = CapGraph.Series.IndexOf(Balance_Line_List(0).name)
            If chart_series_index >= 0 Then
                CapGraph.Series.RemoveAt(chart_series_index)
            End If

            chart_series_index = CapGraph.Annotations.IndexOf(Balance_Line_List(0).name)
            If chart_series_index >= 0 Then
                CapGraph.Annotations.RemoveAt(chart_series_index)
            End If

            Balance_Line_List.Remove(Balance_Line_List(0))
        End If

        ' Add the new line
        Dim temp_bal_line As Balance_Line_Data_Structure
        If (Balance_Line_List.Count = 0) Then
            temp_bal_line.number = 1
        Else
            temp_bal_line.number = Balance_Line_List(Balance_Line_List.Count - 1).number + 1
        End If
        temp_bal_line.name = "Line " + temp_bal_line.number.ToString()
        temp_bal_line.type = type
        ' Set the x-coordinate value for the line, accounting for wraparound of the data.
        If CapDemo_Signal_Group.next_datapoint >= (1 + datapoints_back) Then
            temp_bal_line.x_coord = CapDemo_Signal_Group.Graphed_Signals(0).data(CapDemo_Signal_Group.next_datapoint - (1 + datapoints_back)).time
        Else
            temp_bal_line.x_coord = CapDemo_Signal_Group.Graphed_Signals(0).data(CapDemo_Signal_Group.num_datapoints - (1 + datapoints_back)).time
        End If
        If Balance_Line_Text_Disable = True Then
            temp_bal_line.text = ""
        Else
            temp_bal_line.text = text
        End If
        Balance_Line_List.Add(temp_bal_line)

        ' if this is the stopped line for a set of lines started with the charger or discharger on, start allowing lines to be made again.
        If (type = Balance_Line_Type.Balance_Stopped) And (Balance_Line_Text_Disable = True) Then
            Balance_Line_Text_Disable = False
        End If

    End Sub

    Public Sub Clear_Balance_Lines()
        Try
            While Balance_Line_List.Count > 0
                Dim chart_series_index As Integer = CapGraph.Series.IndexOf(Balance_Line_List(0).name)
                If chart_series_index >= 0 Then
                    CapGraph.Series.RemoveAt(chart_series_index)
                End If

                chart_series_index = CapGraph.Annotations.IndexOf(Balance_Line_List(0).name)
                If chart_series_index >= 0 Then
                    CapGraph.Annotations.RemoveAt(chart_series_index)
                End If

                Balance_Line_List.Remove(Balance_Line_List(0))
            End While
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Public Sub Add_Balance_Lines_GUI()

        If Balance_Line_List.Count <> 0 Then

            For line_num As Integer = 0 To Balance_Line_List.Count - 1

                ' Check to see if line is already in the graph

                If CapGraph.Series.IndexOf(Balance_Line_List(line_num).name) >= 0 Then
                    ' Line is already on the graph.  Don't redraw it.
                Else
                    ' Line is not already on the graph.  Draw it.
                    CapGraph.Series.Add(Balance_Line_List(line_num).name)

                    Select Case Balance_Line_List(line_num).type
                        Case Balance_Line_Type.Balance_Started
                            CapGraph.Series(Balance_Line_List(line_num).name).Color = Drawing.Color.Green
                        Case Balance_Line_Type.Balance_Started_Voltage_Matching
                            CapGraph.Series(Balance_Line_List(line_num).name).Color = Drawing.Color.Red
                        Case Balance_Line_Type.Balance_Stopped
                            CapGraph.Series(Balance_Line_List(line_num).name).Color = Drawing.Color.Blue
                        Case Else
                            CapGraph.Series(Balance_Line_List(line_num).name).Color = Drawing.Color.Black
                    End Select
                    CapGraph.Series(Balance_Line_List(line_num).name).ChartType = DataVisualization.Charting.SeriesChartType.Line
                    CapGraph.Series(Balance_Line_List(line_num).name).IsVisibleInLegend = False
                    CapGraph.Series(Balance_Line_List(line_num).name).BorderDashStyle = DataVisualization.Charting.ChartDashStyle.Dash
                    CapGraph.Series(Balance_Line_List(line_num).name).Points.AddXY(Balance_Line_List(line_num).x_coord, Graphed_Signal_Group.Y_AXIS_MIN)
                    CapGraph.Series(Balance_Line_List(line_num).name).Points.AddXY(Balance_Line_List(line_num).x_coord, Graphed_Signal_Group.Y_AXIS_MAX)

                    ' Add the text annotation to the line if it isn't already there
                    Dim newText As DataVisualization.Charting.TextAnnotation = New DataVisualization.Charting.TextAnnotation
                    newText.Name = Balance_Line_List(line_num).name
                    newText.Text = Balance_Line_List(line_num).text
                    CapGraph.Annotations.Add(newText)
                    CapGraph.Annotations(newText.Name).ClipToChartArea = "ChartArea1"
                    CapGraph.Annotations(newText.Name).Font = New System.Drawing.Font("Microsoft Sans Serif", 8.25!, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, CType(0, Byte))
                    CapGraph.Annotations(newText.Name).AxisX = CapGraph.ChartAreas(0).AxisX
                    CapGraph.Annotations(newText.Name).AnchorX = Balance_Line_List(line_num).x_coord
                    Select Case Balance_Line_List(line_num).type
                        Case Balance_Line_Type.Balance_Started
                            CapGraph.Annotations(newText.Name).ForeColor = Drawing.Color.DarkGreen
                            CapGraph.Annotations(newText.Name).AnchorOffsetX = -2.75
                        Case Balance_Line_Type.Balance_Started_Voltage_Matching
                            CapGraph.Annotations(newText.Name).ForeColor = Drawing.Color.DarkRed
                            CapGraph.Annotations(newText.Name).AnchorOffsetX = -2.75
                        Case Balance_Line_Type.Balance_Stopped
                            CapGraph.Annotations(newText.Name).ForeColor = Drawing.Color.DarkBlue
                            CapGraph.Annotations(newText.Name).AnchorOffsetX = 2.75
                        Case Else
                            CapGraph.Annotations(newText.Name).ForeColor = Drawing.Color.Black
                    End Select
                    CapGraph.Annotations(newText.Name).Y = 12 - 3 * (Balance_Line_List(line_num).number Mod 3)
                End If
            Next line_num
        End If
    End Sub

    Public Sub Clear_Balance_Lines_GUI()

        For line_num As Integer = 0 To Balance_Line_List.Count - 1
            Dim chart_series_index As Integer = CapGraph.Series.IndexOf(Balance_Line_List(line_num).name)
            If chart_series_index >= 0 Then
                CapGraph.Series.RemoveAt(chart_series_index)
            End If

            chart_series_index = CapGraph.Annotations.IndexOf(Balance_Line_List(line_num).name)
            If chart_series_index >= 0 Then
                CapGraph.Annotations.RemoveAt(chart_series_index)
            End If
        Next line_num

    End Sub

#End Region

#Region "Balance Alogrithm"
    Private Sub AlgoBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles AlgoBtn.Click
        D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_ALGORITHM_COMMAND + "W")
    End Sub

    Private Sub VMBox_CheckedChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles VMBox.CheckedChanged
        Try
            If VMBox.Checked = True Then
                VMTextBox.BackColor = Drawing.Color.White
                VMTextBox.Enabled = True
                VMTextBox.Text = voltage_match_copy.ToString()
            Else
                VMTextBox.BackColor = Drawing.Color.Silver
                VMTextBox.Enabled = False
                VMTextBox.Text = ""
            End If

            voltage_match_checked_copy = VMBox.Checked
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub VMTextBox_Enter(ByVal sender As System.Object, ByVal e As KeyPressEventArgs) Handles VMTextBox.KeyPress
        If e.KeyChar = Convert.ToChar(Keys.Enter) Then
            VMTextBox_LostFocus(sender, e)
        End If
    End Sub

    Private Sub VMTextBox_LostFocus(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles VMTextBox.LostFocus

        Try
            If CInt(VMTextBox.Text) < VOLTAGE_MATCH_MIN Then
                VMTextBox.Text = VOLTAGE_MATCH_MIN.ToString()
            End If
            voltage_match_copy = CInt(VMTextBox.Text)
        Catch ex As Exception
            D33001.Handle_Exception(ex)

            If VMTextBox.Text <> "" Then
                VMTextBox.Text = voltage_match_copy.ToString()
            End If
        End Try
    End Sub

    Private Sub balStartBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles balStartBtn.Click

        Try
            ' If not currently balancing, then start it if there's not an UV/OV condition.
            Dim board_num As Integer
            If D33001.Get_SystemOVorUV(board_num) = True Then
                ' If OV or UV present, do not start the balancing
                ' Popup a message about OV or UV preventing balancing
                If OVorUVPopup.Visible = False Then
                    OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                    OVorUVPopup.Show()
                End If
            Else
                D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_ALGORITHM_COMMAND + "W")

                ' Start balancing
                D33001.Timed_Balance_Start()

            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub balStopBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles balStopBtn.Click
        Try
            D33001.Timed_Balance_Stop(False)
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Function Calculate_Stack_Charge(Optional ByVal min_voltage As Single = 0) As Single
        Dim temp_charge As Single = 0

        For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
            temp_charge += (D33001.Voltages(D33001.DC2100A_PIC_BOARD_NUM, cell_num) - min_voltage) * D33001.EEPROM_Data(D33001.DC2100A_PIC_BOARD_NUM).Capacity(cell_num)
        Next cell_num

        Return temp_charge
    End Function
#End Region

#Region "Cap Demo Button Interlocks"

    ' This function will configure the Cap Demo Window given the demo being performed.  It is in a subroutine so that these settings
    ' can be consolidated into one place, and so that functions (such as OV/UV) can call something to stop whatever demo is in progess.
    ' This essentially creates a drop down menu (combobox) for the manual charge/discharge and demos buttons.
    Public Sub Update_GUI_Cap_Demo_Buttons(Optional ByVal demo_to_start As Cap_Demo_Button_Type = Cap_Demo_Button_Type.None_Selected)
        ' Start with all controls enabled.
        VMBox.Enabled = True
        AlgoBtn.Enabled = True
        balStartBtn.Enabled = True

        ' Set voltage matching back to whatever it was before the test started.
        VMBox.Checked = voltage_match_checked_copy

        ' Re-enable controls on main window.
        D33001.Balance_Control_Timed_CheckBox.Enabled = True
        D33001.Balance_Control_Manual_CheckBox.Enabled = True

        If D33001.Balance_Control_Manual_CheckBox.Checked = True Then
            D33001.Balance_Control_Manual_Write_Command_Button.Enabled = True
            D33001.Balance_Control_Manual_Execute_Button.Enabled = True
            D33001.Balance_Control_Manual_Suspend_Button.Enabled = True
            D33001.Balance_Control_Manual_Read_Command_Button.Enabled = True
            D33001.Balance_Control_Manual_Read_Status_Button.Enabled = True
        End If

        If D33001.Balance_Control_Timed_CheckBox.Checked = True Then
            D33001.Balance_Control_Timed_Write_Button.Enabled = True
            D33001.Balance_Control_Timed_StartStop_Button.Enabled = True
            D33001.Balance_Control_Timed_Reset_Button.Enabled = True
        End If

        ' Set the button for the selected op demo to be enabled and colored, set all other buttons appropriately locked out.
        For op_demo_num As Cap_Demo_Button_Type = 0 To Cap_Demo_Button_Type.Num_Cap_Demo_Buttons - 1
            ' Configure button for the selected op demo, and all other buttons to their non-selected text and deactivated color.
            If op_demo_num = demo_to_start Then
                Cap_Demo_Button_Data(op_demo_num).button.Text = Cap_Demo_Button_Data(op_demo_num).button_activated_text
                Cap_Demo_Button_Data(op_demo_num).button.BackColor = ACTIVATED_COLOR
                Cap_Demo_Button_Data(op_demo_num).button.Enabled = True
            Else
                Cap_Demo_Button_Data(op_demo_num).button.Text = Cap_Demo_Button_Data(op_demo_num).button_deactivated_text
                Cap_Demo_Button_Data(op_demo_num).button.BackColor = DEACTIVATED_COLOR

                ' All buttons enabled when nothing selected, or when performing a basic charger control.  Otherwise, only the selected button is enabled.
                If (demo_to_start = Cap_Demo_Button_Type.None_Selected) OrElse
                    (demo_to_start = Cap_Demo_Button_Type.Charge) OrElse
                    (demo_to_start = Cap_Demo_Button_Type.Discharge) OrElse
                    (demo_to_start = Cap_Demo_Button_Type.Stop_Chg_Dchg) Then
                    Cap_Demo_Button_Data(op_demo_num).button.Enabled = True
                Else
                    Cap_Demo_Button_Data(op_demo_num).button.Enabled = False
                End If
            End If
        Next op_demo_num

        ' Now handle the special case stuff.  It all related to the balancing/algorithm/voltage match controls.
        Dim temp_vmbox As Boolean
        Select Case demo_to_start

            Case Cap_Demo_Button_Type.Charge, _
                 Cap_Demo_Button_Type.Discharge, _
                 Cap_Demo_Button_Type.Stop_Chg_Dchg
                ' The basic charger controls do not affect our ability to balance

            Case Cap_Demo_Button_Type.Balance_While_Charging, _
                 Cap_Demo_Button_Type.Balance_While_Discharging
                ' The demos that use voltage matching must lock matching on and prevent user control of the balancers

                ' Set up the GUI for voltage matching
                temp_vmbox = VMBox.Checked
                VMBox.Checked = True
                VMTextBox.Text = voltage_match_copy.ToString()
                voltage_match_checked_copy = temp_vmbox              ' remember the old setting so that it can be restored when demo is ended

                ' This demo controls the balancers using voltage matching.  Do not let user change the controls for votlage matching, 
                ' although allow them to change the voltage match target value.
                AlgoBtn.Enabled = False
                VMBox.Enabled = False
                balStartBtn.Enabled = False
                balStopBtn.Enabled = False

                ' Set up controls on main window to be appropriate for timed balancing, but lock out so that user can't mess with controls.
                D33001.Balance_Control_Timed_CheckBox.Checked = True
                D33001.Balance_Control_Timed_Write_Button.Enabled = False
                D33001.Balance_Control_Timed_StartStop_Button.Enabled = False
                D33001.Balance_Control_Timed_Reset_Button.Enabled = False
                D33001.Balance_Control_Manual_CheckBox.Enabled = False
                D33001.Balance_Control_Timed_CheckBox.Enabled = False

                ' Remember the state of the charger before this button was pressed
                charger_on_copy = ChgBox.Checked
                discharger_on_copy = DchgBox.Checked

            Case Cap_Demo_Button_Type.Charge_All_Cells, _
                 Cap_Demo_Button_Type.Discharge_All_Cells, _
                 Cap_Demo_Button_Type.Alternate_Chg_Dchg, _
                 Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg
                ' The demos that use the balancers to imbalance must lock matching off and prevent user control of the balancers

                ' Set up the GUI for not voltage matching
                temp_vmbox = VMBox.Checked
                VMBox.Checked = False
                VMTextBox.Text = ""
                voltage_match_checked_copy = temp_vmbox              ' remember the old setting so that it can be restored when demo is ended

                ' This demo controls the LTC3300s to force imbalance.  Do not let user turn on any balancing during demo.
                AlgoBtn.Enabled = False
                VMBox.Enabled = False
                balStartBtn.Enabled = False
                balStopBtn.Enabled = False

                ' Set up controls on main window to be appropriate for manual balancing, but lock out so that user can't mess with controls.
                D33001.Balance_Control_Manual_CheckBox.Checked = True
                D33001.Balance_Control_Manual_Write_Command_Button.Enabled = False
                D33001.Balance_Control_Manual_Execute_Button.Enabled = False
                D33001.Balance_Control_Manual_Suspend_Button.Enabled = False
                D33001.Balance_Control_Manual_Read_Command_Button.Enabled = False
                D33001.Balance_Control_Manual_Read_Status_Button.Enabled = False
                D33001.Balance_Control_Timed_CheckBox.Enabled = False
                D33001.Balance_Control_Manual_CheckBox.Enabled = False

                ' Remember the state of the charger before this button was pressed
                charger_on_copy = ChgBox.Checked
                discharger_on_copy = DchgBox.Checked

        End Select

        ' Remember the test started
        cap_demo_button_activated = demo_to_start

    End Sub

    ' This function will disable all of the cap demo and trade show buttons.
    Public Sub Disable_All_Cap_Demo_Buttons()

        ' Set the button for the selected op demo to be enabled and colored, set all other buttons appropriately locked out.
        For op_demo_num As Cap_Demo_Button_Type = 0 To Cap_Demo_Button_Type.Num_Cap_Demo_Buttons - 1
            Cap_Demo_Button_Data(op_demo_num).button.Enabled = False
        Next op_demo_num

        opDemoBtn.Enabled = False
    End Sub

    ' This function will enable all of the cap demo and trade show buttons.
    Public Sub Enable_All_Cap_Demo_Buttons()

        ' Set the button for the selected op demo to be enabled and colored, set all other buttons appropriately locked out.
        For op_demo_num As Cap_Demo_Button_Type = 0 To Cap_Demo_Button_Type.Num_Cap_Demo_Buttons - 1
            Cap_Demo_Button_Data(op_demo_num).button.Enabled = True
        Next op_demo_num

        opDemoBtn.Enabled = True
    End Sub

    ' Demo buttons alter the settings for the charger/discharger.  After we unpress the button, we wish to restore
    ' the settings for the charger/discharger as if the manual button had been pressed to restore the setting.
    Public Sub Restore_Charger_and_Discharger_Settings()

        ' temporary objects so that handler functions can be called directly
        Dim e As New System.EventArgs
        Dim sender As New System.Object

        If charger_on_copy = True Then
            ChgBtn_Click(sender, e)
        ElseIf discharger_on_copy = True Then
            DchgBtn_Click(sender, e)
        Else
            NoneBtn_Click(sender, e)
        End If


    End Sub
#End Region

#Region "Manual Charge/Discharge"
    Private Sub ChgBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ChgBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        ' Charge if there's not an OV condition.
        Dim board_num As Integer
        If D33001.Get_SystemOV(board_num) = True Then
            ' If OV present, do not start charging
            ' Popup a message about OV preventing charging
            If OVorUVPopup.Visible = False Then
                OVorUVPopup.Set_Text("Charging", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                OVorUVPopup.Show()
            End If
        Else
            ' Set up the buttons in Cap Demo Window while test is in progress
            Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Charge)

            D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "C")
        End If

    End Sub

    Private Sub DchgBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles DchgBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        ' Set up the buttons in Cap Demo Window while test is in progress
        Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Discharge)

        D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "D")

    End Sub

    Private Sub NoneBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles NoneBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        ' Set up the buttons in Cap Demo Window while test is in progress
        Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Stop_Chg_Dchg)

        D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "N")
    End Sub
#End Region

#Region "Demos"

    Private Sub ChgBalBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ChgBalBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
                Exit Sub
            End If
        Catch
        End Try

        If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Charging).button_deactivated_text Then
            ' Test not in progress, start it if there's not an UV/OV condition.
            Dim board_num As Integer
            If D33001.Get_SystemOVorUV(board_num) = True Then
                ' If OV or UV present, do not start the balancing
                ' Popup a message about OV or UV preventing balancing
                If OVorUVPopup.Visible = False Then
                    OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                    OVorUVPopup.Show()
                End If
            Else
                ' Set up the buttons in Cap Demo Window while test is in progress
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Balance_While_Charging)

                ' Send command to DC2100A to start charging.
                D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "C")

                ' Send command to DC2100A to compute balance times.
                D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_ALGORITHM_COMMAND + "W")

                ' Start balancing
                D33001.Timed_Balance_Start()
            End If

        Else
            ' Test already in progress, stop it.
            Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

            ' Restore the charger/discharger settings
            Restore_Charger_and_Discharger_Settings()

            ' Stop balancing
            D33001.Timed_Balance_Stop(False)

        End If

    End Sub

    Private Sub DchgBalBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles DchgBalBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
                Exit Sub
            End If
        Catch
        End Try

        Try
            If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Balance_While_Discharging).button_deactivated_text Then

                ' Test not in progress, start it if there's not an UV/OV condition.
                Dim board_num As Integer
                If D33001.Get_SystemOVorUV(board_num) = True Then
                    ' If OV or UV present, do not start the balancing
                    ' Popup a message about OV or UV preventing balancing
                    If OVorUVPopup.Visible = False Then
                        OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                        OVorUVPopup.Show()
                    End If
                Else
                    ' Set up the buttons in Cap Demo Window while test is in progress
                    Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Balance_While_Discharging)

                    ' Send command to DC2100A to start discharging.
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "D")

                    ' Send command to DC2100A to compute balance times.
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_ALGORITHM_COMMAND + "W")

                    ' Start balancing
                    D33001.Timed_Balance_Start()
                End If

            Else
                ' Test already in progress, stop it.
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

                ' Restore the charger/discharger settings
                Restore_Charger_and_Discharger_Settings()

                ' Stop balancing
                D33001.Timed_Balance_Stop(False)

            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub ChgAllBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ChgAllBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        Try
            If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Charge_All_Cells).button_deactivated_text Then
                ' Test not in progress, start it if there's not an UV/OV condition.
                Dim board_num As Integer
                If D33001.Get_SystemOVorUV(board_num) = True Then
                    ' If OV or UV present, do not start the balancing
                    ' Popup a message about OV or UV preventing balancing
                    If OVorUVPopup.Visible = False Then
                        OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                        OVorUVPopup.Show()
                    End If
                Else
                    ' Turn off the charge/discharge
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "N")

                    ' Set up the buttons in Cap Demo Window while test is in progress
                    Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Charge_All_Cells)

                    ' Set up balancers to all charge
                    For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                        D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, cell_num).Set_Write_Action(Cell_Balancer.Balance_Action.Charge)
                    Next cell_num

                    ' Update GUI and calculate CRC
                    For ic_num As Integer = 0 To D33001.NUM_LTC3300 - 1
                        D33001.LTC3300s(D33001.DC2100A_PIC_BOARD_NUM, ic_num).Update_Write_Command_GUI()
                    Next ic_num

                    ' Send commands to DC2100A and read back data
                    D33001.LTC3300_Write_Balance()
                    D33001.LTC3300_Execute()
                    D33001.readAllData()
                End If
            Else
                ' Test already in progress, stop it.
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

                ' Restore the charger/discharger settings
                Restore_Charger_and_Discharger_Settings()

                ' Suspend balancing.
                D33001.LTC3300_Suspend()
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub DchgAllBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles DchgAllBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        Try
            If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Discharge_All_Cells).button_deactivated_text Then
                ' Test not in progress, start it if there's not an UV/OV condition.
                Dim board_num As Integer
                If D33001.Get_SystemOVorUV(board_num) = True Then
                    ' If OV or UV present, do not start the balancing
                    ' Popup a message about OV or UV preventing balancing
                    If OVorUVPopup.Visible = False Then
                        OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                        OVorUVPopup.Show()
                    End If
                Else
                    ' Turn off the charge/discharge
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "N")

                    ' Set up the buttons in Cap Demo Window while test is in progress
                    Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Discharge_All_Cells)

                    ' Set up balancers to all discharge
                    For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                        D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, cell_num).Set_Write_Action(Cell_Balancer.Balance_Action.Discharge)
                    Next cell_num

                    ' Update GUI and calculate CRC
                    For ic_num As Integer = 0 To D33001.NUM_LTC3300 - 1
                        D33001.LTC3300s(D33001.DC2100A_PIC_BOARD_NUM, ic_num).Update_Write_Command_GUI()
                    Next ic_num

                    ' Send commands to DC2100A and read back data
                    D33001.LTC3300_Write_Balance()
                    D33001.LTC3300_Execute()
                    D33001.readAllData()
                End If
            Else
                ' Test already in progress, stop it.
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

                ' Restore the charger/discharger settings
                Restore_Charger_and_Discharger_Settings()

                ' Suspend balancing.
                D33001.LTC3300_Suspend()
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub AltChgBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles AltChgBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        Try
            If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Alternate_Chg_Dchg).button_deactivated_text Then
                ' Test not in progress, start it if there's not an UV/OV condition.
                Dim board_num As Integer
                If D33001.Get_SystemOVorUV(board_num) = True Then
                    ' If OV or UV present, do not start the balancing
                    ' Popup a message about OV or UV preventing balancing
                    If OVorUVPopup.Visible = False Then
                        OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                        OVorUVPopup.Show()
                    End If
                Else
                    ' Turn off the charge/discharge
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "N")

                    ' Set up the buttons in Cap Demo Window while test is in progress
                    Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Alternate_Chg_Dchg)

                    ' Set up balancers to alternate charge/discharge
                    For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                        Dim balance_action As Cell_Balancer.Balance_Action
                        If ((cell_num Mod 2) = 0) Then
                            balance_action = Cell_Balancer.Balance_Action.Charge
                        Else
                            balance_action = Cell_Balancer.Balance_Action.Discharge
                        End If
                        D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, cell_num).Set_Write_Action(balance_action)
                    Next cell_num

                    ' Update GUI and calculate CRC
                    For ic_num As Integer = 0 To D33001.NUM_LTC3300 - 1
                        D33001.LTC3300s(D33001.DC2100A_PIC_BOARD_NUM, ic_num).Update_Write_Command_GUI()
                    Next ic_num

                    ' Send commands to DC2100A and read back data
                    D33001.LTC3300_Write_Balance()
                    D33001.LTC3300_Execute()
                    D33001.readAllData()
                End If
            Else
                ' Test already in progress, stop it.
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

                ' Restore the charger/discharger settings
                Restore_Charger_and_Discharger_Settings()

                ' Suspend balancing.
                D33001.LTC3300_Suspend()
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Private Sub DemoChgBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles DemoChgBtn.Click

        Try  ' If this was a human pressing the button and a Trade Show Demo is in progress, stop it
            Dim this_button As Button = DirectCast(sender, Button)
            If (trade_show_demo_state <> Trade_Show_Demo_State_Type.Idle) Then
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch
        End Try

        Try
            If Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button.Text = Cap_Demo_Button_Data(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg).button_deactivated_text Then
                ' Test not in progress, start it if there's not an UV/OV condition.
                Dim board_num As Integer
                If D33001.Get_SystemOVorUV(board_num) = True Then
                    ' If OV or UV present, do not start the balancing
                    ' Popup a message about OV or UV preventing balancing
                    If OVorUVPopup.Visible = False Then
                        OVorUVPopup.Set_Text("Balancing", D33001.Get_OVUV_Condition_String(board_num), board_num, False)
                        OVorUVPopup.Show()
                    End If
                Else
                    ' Turn off the charge/discharge
                    D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_CAP_DEMO_COMMAND + "N")

                    ' Set up the buttons in Cap Demo Window while test is in progress
                    Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.Cell_7_Chg_Cell_6_Dchg)

                    ' Set up balancers to alternate charge/discharge
                    For cell_num As Integer = 0 To D33001.MAX_CELLS - 1
                        D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, cell_num).Set_Write_Action(Cell_Balancer.Balance_Action.None)
                    Next cell_num
                    D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, 5).Set_Write_Action(Cell_Balancer.Balance_Action.Discharge)
                    D33001.Cell_Balancers(D33001.DC2100A_PIC_BOARD_NUM, 6).Set_Write_Action(Cell_Balancer.Balance_Action.Charge)

                    ' Update GUI and calculate CRC
                    For ic_num As Integer = 0 To D33001.NUM_LTC3300 - 1
                        D33001.LTC3300s(D33001.DC2100A_PIC_BOARD_NUM, ic_num).Update_Write_Command_GUI()
                    Next ic_num

                    ' Send commands to DC2100A and read back data
                    D33001.LTC3300_Write_Balance()
                    D33001.LTC3300_Execute()
                    D33001.readAllData()
                End If
            Else
                ' Test already in progress, stop it.
                Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)

                ' Restore the charger/discharger settings
                Restore_Charger_and_Discharger_Settings()

                ' Suspend balancing.
                D33001.LTC3300_Suspend()
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Public Sub Update_Demos()
        Dim is_balancing_next As Boolean
        Dim nonzero_balance_time_next As Boolean
        Dim stack_charge_next As Single
        Dim balance_stopped_sent_next As Boolean

        Try

            ' Evaluate conditions used to mark timed balancing on graph and voltage match
            is_balancing_next = D33001.voltage_timestamp(D33001.selectedBoard).is_balancing
            If (D33001.Max_Balance_Time_Value <> 0) Then
                nonzero_balance_time_next = True
            Else
                nonzero_balance_time_next = False
            End If
            stack_charge_next = Calculate_Stack_Charge(D33001.vMin)
            balance_stopped_sent_next = False   ' most paths through code do not result in this flag being set, so assume clear for now

            If VMBox.Checked = False Then

                ' If not voltage matching, just place lines on the voltage samples where the balancing started and stopped
                If (is_balancing_last = False) And (is_balancing_next = True) Then
                    ' These are the first samples after we started timed balancing, make balance lines
                    Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Started, stack_charge_last.ToString(CHARGE_FORMAT) + "As", 1)
                ElseIf (is_balancing_last = True) And (is_balancing_next = False) Then
                    ' These are the first samples after we stopped timed balancing, make balance lines
                    Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Stopped, stack_charge_next.ToString(CHARGE_FORMAT) + "As")
                    balance_stopped_sent_next = True
                ElseIf (nonzero_balance_time_last = True) And (nonzero_balance_time_next = False) And (is_balancing_last = False) And (balance_stopped_sent_last = False) Then
                    ' Balancing was so quick that we didn't get any voltage samples while the balancers are on
                    Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Started, stack_charge_last.ToString(CHARGE_FORMAT) + "As", 1)
                    Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Stopped, stack_charge_next.ToString(CHARGE_FORMAT) + "As")
                    balance_stopped_sent_next = True
                End If
            Else

                ' If voltage matching, place lines on the voltage samples where the balancing started.
                ' When stopped, however, decide if we need to start again.
                If (is_balancing_last = False) And (is_balancing_next = True) Then
                    ' These are the first samples after we started timed balancing, make balance lines
                    Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Started_Voltage_Matching, stack_charge_last.ToString(CHARGE_FORMAT) + "As", 1)
                ElseIf (is_balancing_next = False) Then

                    ' If GUI wants to be Timed Balancing with Voltage Matching, but FW isn't currently balancing, decide if balancing should be started.
                    If (D33001.isTimedBalancing = True) Then

                        ' Calculate the voltage imbalance
                        Dim volt_diff As Single = (D33001.Stack_Summary_Data.Volt_Max - D33001.Stack_Summary_Data.Volt_Min)
                        Dim volt_diff_limit As Single = CSng(voltage_match_copy) / D33001.MV_PER_V

                        If (nonzero_balance_time_last = True) And (nonzero_balance_time_next = False) And (is_balancing_last = False) And (balance_stopped_sent_last = False) Then
                            ' Balancing was so quick that we didn't get any voltage samples while the balancers are on
                            Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Started_Voltage_Matching, stack_charge_last.ToString(CHARGE_FORMAT) + "As", 1)
                            Add_Balance_Line(CapBoardPopup.Balance_Line_Type.Balance_Stopped, stack_charge_next.ToString(CHARGE_FORMAT) + "As")
                            balance_stopped_sent_next = True
                        ElseIf (nonzero_balance_time_next = True) Then
                            ' If the FW has balance times loaded but isn't actively balancing, then the algorithm command must have calculated non-zero balance times
                            ' Start the balancing to voltage match
                            D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_TIMED_BALANCE_COMMAND + "B" + D33001.selectedBoard.ToString("X2"))

                            ' Show message to indicate that balancing is started due to voltage matching
                            D33001.NoteLabel.Text = "Balancing: " + volt_diff.ToString(D33001.VOLTAGE_FORMAT) + "V > " + volt_diff_limit.ToString(D33001.VOLTAGE_FORMAT)
                        Else
                            ' Check to see if voltages are matched.
                            If volt_diff > volt_diff_limit Then
                                ' If still not matched, try calculating new balance times
                                D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_ALGORITHM_COMMAND + "W")
                            ElseIf (is_balancing_last = True) Then
                                ' These are the first samples after we stopped timed balancing, make balance lines
                                Add_Balance_Line(Balance_Line_Type.Balance_Stopped, stack_charge_next.ToString(CHARGE_FORMAT) + "As")
                                balance_stopped_sent_next = True
                            End If

                        End If

                    End If

                End If

            End If

            ' Save conditions used to mark timed balancing on graph and voltage match so that they can be detected when they change the next time subroutine is executed.
            is_balancing_last = is_balancing_next
            nonzero_balance_time_last = nonzero_balance_time_next
            stack_charge_last = stack_charge_next
            balance_stopped_sent_last = balance_stopped_sent_next

        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

#End Region

#Region "Trade Show Demo"

    Private Sub opDemoBtn_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles opDemoBtn.Click
        Try
            If trade_show_demo_state = Trade_Show_Demo_State_Type.Idle Then
                ' Test not in progress, start it
                Trade_Show_Demo_State_Machine_Start()
            Else
                ' Test already in progress, stop it.
                Trade_Show_Demo_State_Machine_Stop()
            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    Public Sub Trade_Show_Demo_State_Machine_Start()

        ' temporary objects so that handler functions can be called directly
        Dim e As New System.EventArgs
        Dim sender As New System.Object

        ' Clear transition variables
        trade_show_demo_float_voltage_list.Clear()

        ' Set up graph display
        GraphOptionPopup.scopeMode = False
        GraphOptionPopup.snappedToFront = False

        ' Make control panel and graph display the demo results
        Enable_CheckBox.Checked = True
        D33001.autoReadBox.Checked = True

        ' Stop and clear any balancing that may be underway
        D33001.Timed_Balance_Stop(True)

        ' Save settings so that they can be restored after trade show demo
        vmin_copy = D33001.vMin
        vmax_copy = D33001.vMax

        'Set OV/UV so they're not interfering with the charging/discharging/balancing
        D33001.vMin = TRADE_SHOW_DEMO_MIN_VOLTAGE
        D33001.vMax = TRADE_SHOW_DEMO_MAX_VOLTAGE
        D33001.vMinBox.Text = D33001.vMin.ToString(D33001.VOLTAGE_LIMIT_FORMAT)
        D33001.vMaxBox.Text = D33001.vMax.ToString(D33001.VOLTAGE_LIMIT_FORMAT)
        D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_UVOV_THRESHOLDS_COMMAND + CInt(D33001.vMax / D33001.VOLTAGE_RESOLUTION).ToString("X4") + CInt(D33001.vMin / D33001.VOLTAGE_RESOLUTION).ToString("X4"))

        ' Start with the graph cleared
        ClearLogBtn_Click(sender, e)

        ' Make button show that test is in progress
        Trade_Show_Demo_Button_Data.button.Text = Trade_Show_Demo_Button_Data.button_activated_text
        Trade_Show_Demo_Button_Data.button.BackColor = ACTIVATED_COLOR

        ' Clear results from last test
        noBalTimeBox.Text = ""
        balTimeBox.Text = ""
        noBalChargeBox.Text = ""
        balChargeBox.Text = ""

        ' Display message about state transition
        D33001.NoteLabel.Text = "Trade Show Demo Started"

        ' Start trade show demo at charger float voltage and all cells balanced.
        ChgBtn_Click(sender, e)
        trade_show_demo_state = Trade_Show_Demo_State_Type.Init_Charging
        trade_show_demo_timer = 0
        trade_show_demo_timestamp_last = D33001.voltage_timestamp(D33001.selectedBoard).time
        trade_show_demo_display_time_start = 0

    End Sub

    Public Sub Trade_Show_Demo_State_Machine_Stop()

        ' temporary objects so that handler functions can be called directly
        Dim e As New System.EventArgs
        Dim sender As New System.Object

        ' If there was so much of a init period that the graph looks bad at the end of the trade show demo, adjust graph to display the trade show demo.
        If trade_show_demo_display_time_start <> 0 Then
            GraphOptionPopup.scopeRange = (CapGraph.ChartAreas(0).AxisX.Maximum - trade_show_demo_display_time_start)
            GraphOptionPopup.snappedToFront = True
            GraphOptionPopup.scopeMode = True
        End If

        ' Restore settings to those before the trade show demo was started
        D33001.vMin = vmin_copy
        D33001.vMax = vmax_copy
        D33001.vMinBox.Text = D33001.vMin.ToString(D33001.VOLTAGE_LIMIT_FORMAT)
        D33001.vMaxBox.Text = D33001.vMax.ToString(D33001.VOLTAGE_LIMIT_FORMAT)
        D33001.USB_Comm_List_Out.Add(D33001.USB_PARSER_UVOV_THRESHOLDS_COMMAND + CInt(D33001.vMax / D33001.VOLTAGE_RESOLUTION).ToString("X4") + CInt(D33001.vMin / D33001.VOLTAGE_RESOLUTION).ToString("X4"))

        ' Stop whatever test is in progress and turn off charger
        Update_GUI_Cap_Demo_Buttons(Cap_Demo_Button_Type.None_Selected)
        VMBox.Checked = False
        NoneBtn_Click(sender, e)

        ' Make button show that test is not in progress
        Trade_Show_Demo_Button_Data.button.Text = Trade_Show_Demo_Button_Data.button_deactivated_text
        Trade_Show_Demo_Button_Data.button.BackColor = DEACTIVATED_COLOR

        ' Display message about state transition
        D33001.NoteLabel.Text = "Trade Show Demo Complete"

        ' Stop state machine and return to Idle
        trade_show_demo_state = Trade_Show_Demo_State_Type.Idle
        trade_show_demo_timer = 0
    End Sub

    Public Sub Update_Trade_Show_Demo_State_Machine()

        ' temporary objects so that handler functions can be called directly
        Dim e As New System.EventArgs
        Dim sender As New System.Object

        Try

            ' If Idle, do nothing
            If trade_show_demo_state = Trade_Show_Demo_State_Type.Idle Then
                Exit Sub
            End If

            ' Update state machine timer
            Dim trade_show_demo_timestamp_next As Single = D33001.voltage_timestamp(D33001.selectedBoard).time
            Dim trade_show_demo_timestamp_step As Integer = (trade_show_demo_timestamp_next - trade_show_demo_timestamp_last) * D33001.MS_PER_SEC
            trade_show_demo_timestamp_last = trade_show_demo_timestamp_next
            trade_show_demo_timer += trade_show_demo_timestamp_step

            ' Update Floating Flag
            Dim Floating As Boolean = False        ' Start off assuming not floating
            If (D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Average < TRADE_SHOW_DEMO_HIGH_VOLTAGE) Then
                trade_show_demo_float_voltage_list.Clear()      ' Not floating when stack voltage too low
            Else
                trade_show_demo_float_voltage_list.Add(D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Sum)
                If trade_show_demo_float_voltage_list.Count > 1 Then
                    trade_show_demo_float_voltage_list.Sort()
                    If (trade_show_demo_float_voltage_list(trade_show_demo_float_voltage_list.Count - 1) - trade_show_demo_float_voltage_list(0)) > TRADE_SHOW_DEMO_FLOAT_VOLTAGE_DELTA Then
                        trade_show_demo_float_voltage_list.Clear()
                    Else
                        Floating = True             ' When stack voltage has not changed by enough, we are Floating
                    End If
                End If
            End If

            ' Update Balanced Flag
            Dim Balanced As Boolean
            If ((D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Max - D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Min) < TRADE_SHOW_DEMO_BALANCED_VOLTAGE_DELTA) Then
                Balanced = True                     ' When max and min voltages are close enough, we are Balanced
            Else
                Balanced = False
            End If

            ' Update Discharged flag
            Dim Discharged As Boolean
            If (D33001.system_state <> D33001.System_State_Type.Awake) Then
                Discharged = True                   ' When 6804s are not responding at all, we are Discharged.
            ElseIf (D33001.voltage_timestamp(D33001.selectedBoard).is_balancing = False) AndAlso
                   (D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Min < TRADE_SHOW_DEMO_LOW_VOLTAGE) Then
                Discharged = True                   ' If not balancing and the min cell is low enough, we are Discharged.
            Else
                Discharged = False
            End If

            ' Update Charged flag
            Dim Charged As Boolean
            If (D33001.voltage_timestamp(D33001.selectedBoard).is_balancing = False) AndAlso
               ((D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Max > trade_show_demo_high_voltage_measured)) Then
                Charged = True                      ' If not balancing and the max cell is high enough, we are Charged.
            Else
                Charged = False
            End If

            ' Evaluate state machine
            Select Case trade_show_demo_state
                Case Trade_Show_Demo_State_Type.Idle
                    ' Do Nothing - wait until state machine is started

                Case Trade_Show_Demo_State_Type.Init_Charging
                    ' Test if Volt_Min is valid, and significantly above the UV threshold
                    If Discharged = False Then
                        If trade_show_demo_timer >= TRADE_SHOW_DEMO_DISCHARGED_TIME Then

                            ' Start charging with balancing.
                            ChgBalBtn_Click(sender, e)

                            ' Display message about state transition
                            D33001.NoteLabel.Text = "Initializing to float voltage and balanced cells"
                            'addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 50, "")

                            ' Advance the state.
                            trade_show_demo_state = Trade_Show_Demo_State_Type.Init_Charging_With_Balancing
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Init_Charging_With_Balancing
                    If (Floating = True) AndAlso (Balanced = True) Then
                        If (trade_show_demo_timer >= TRADE_SHOW_DEMO_FLOAT_TIME) AndAlso
                           (trade_show_demo_timer >= TRADE_SHOW_DEMO_BALANCED_TIME) Then

                            ' Save actual high voltage for this cap demo system.
                            trade_show_demo_high_voltage_measured = D33001.Board_Summary_Data(D33001.selectedBoard).Volt_Average

                            ' Save actual high voltage for this cap demo system.
                            trade_show_demo_display_time_start = Math.Max(CapGraph.ChartAreas(0).AxisX.Maximum - 10, 0)

                            ' Stop charging with balancing.
                            ChgBalBtn_Click(sender, e)
                            NoneBtn_Click(sender, e)

                            ' Display messages about state transition
                            D33001.NoteLabel.Text = "Delay"
                            addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 4.5, 10, "Init at float" + ChrW(13) & ChrW(10) + "w/balanced cells...")

                            ' Advance the state.
                            trade_show_demo_state = Trade_Show_Demo_State_Type.Init_Charging_Delay
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Init_Charging_Delay
                    If (trade_show_demo_timer >= (TRADE_SHOW_DELAY_TIME - TRADE_SHOW_DEMO_FLOAT_TIME)) Then
                        ' Start timing the discharge time without balancing.
                        trade_show_demo_discharge_time_wo_balancing = 0

                        ' Start discharging without balancing.
                        DchgBtn_Click(sender, e)

                        ' Display messages about state transition
                        D33001.NoteLabel.Text = "Discharging without Balancing"
                        addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 50, "Discharge" + ChrW(13) & ChrW(10) + "w/o Balancing...")

                        ' Advance the state.
                        trade_show_demo_state = Trade_Show_Demo_State_Type.Discharge_Without_Balancing
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Discharge_Without_Balancing
                    ' Time the discharge without balancing, adjusting for the filtering of transition variables and the time spent in between states
                    Dim temp_time As Single
                    trade_show_demo_discharge_time_wo_balancing += trade_show_demo_timestamp_step
                    temp_time = Math.Max(trade_show_demo_discharge_time_wo_balancing - trade_show_demo_timer, 0)
                    noBalTimeBox.Text = (temp_time / D33001.MS_PER_SEC).ToString("##.0") + "s"

                    If (Discharged = True) Then
                        If trade_show_demo_timer >= TRADE_SHOW_DEMO_DISCHARGED_TIME Then

                            ' Adjust time without balancing to ignore the filtering.
                            trade_show_demo_discharge_time_wo_balancing -= trade_show_demo_timer

                            ' Start timing the discharge time with balancing.
                            trade_show_demo_discharge_time_w_balancing = 0

                            ' Start discharging with balancing.
                            DchgBalBtn_Click(sender, e)

                            ' Set the voltage matching setting for the trade show demo
                            VMTextBox.Text = (TRADE_SHOW_DEMO_BALANCED_VOLTAGE_DELTA * D33001.MV_PER_V).ToString()
                            VMTextBox_LostFocus(sender, e)

                            ' Display messages about state transition
                            D33001.NoteLabel.Text = "Discharging with Balancing"
                            addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 10, "Discharge" + ChrW(13) & ChrW(10) + "w/ Balancing...")

                            ' Advance the state.
                            trade_show_demo_state = Trade_Show_Demo_State_Type.Discharge_With_Balancing
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Discharge_With_Balancing
                    ' Time the discharge with balancing, adjusting for the filtering of transition variables and the time spent in between states
                    trade_show_demo_discharge_time_w_balancing += trade_show_demo_timestamp_step
                    Dim total_time As Integer = trade_show_demo_discharge_time_wo_balancing + Math.Max(trade_show_demo_discharge_time_w_balancing - TRADE_SHOW_DEMO_TIME_BETWEEN_STATES - trade_show_demo_timer, 0)
                    balTimeBox.Text = (total_time / D33001.MS_PER_SEC).ToString("##.0") + "s (+" + _
                                      ((100 * (total_time - trade_show_demo_discharge_time_wo_balancing) + (trade_show_demo_discharge_time_wo_balancing \ 2)) \ trade_show_demo_discharge_time_wo_balancing).ToString("##") + "%)"

                    If (Discharged = True) Then

                        If trade_show_demo_timer >= TRADE_SHOW_DEMO_DISCHARGED_TIME Then

                            ' Stop discharging with balancing.
                            DchgBalBtn_Click(sender, e)
                            NoneBtn_Click(sender, e)

                            ' Display messages about state transition
                            D33001.NoteLabel.Text = "Delay"
                            'addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 50, "")

                            ' Advance the state.
                            trade_show_demo_state = Trade_Show_Demo_State_Type.Discharge_Delay
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Discharge_Delay
                    If (trade_show_demo_timer >= TRADE_SHOW_DELAY_TIME) Then
                        ' Start charging without balancing.
                        ChgBtn_Click(sender, e)

                        ' Display messages about state transition
                        D33001.NoteLabel.Text = "Charging without Balancing"
                        addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 50, "Charge" + ChrW(13) & ChrW(10) + "w/o Balancing...")

                        ' Advance the state.
                        trade_show_demo_state = Trade_Show_Demo_State_Type.Charge_Without_Balancing
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Charge_Without_Balancing
                    ' Calculate the charge without balancing (can only get charge from the weakest cell, which is the one that reaches its float voltage first)
                    ' Lock calculation at the start of the timer, so that we don't add the extra volts added when trade_show_demo_timer = TRADE_SHOW_DEMO_CHARGED_TIME
                    If trade_show_demo_timer = trade_show_demo_timestamp_step Then
                        trade_show_demo_charge_wo_balancing = Calculate_Stack_Charge(TRADE_SHOW_DEMO_LOW_VOLTAGE) / D33001.MAX_CELLS
                        noBalChargeBox.Text = (trade_show_demo_charge_wo_balancing).ToString("##.0") + "As"
                    End If

                    If (Charged = True) Then
                        If trade_show_demo_timer >= TRADE_SHOW_DEMO_CHARGED_TIME Then

                            ' Start charging with balancing.
                            ChgBalBtn_Click(sender, e)

                            ' Display message about state transition
                            D33001.NoteLabel.Text = "Charging with Balancing"
                            addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 5, 10, "Charge" + ChrW(13) & ChrW(10) + "w/ Balancing...")

                            trade_show_demo_state = Trade_Show_Demo_State_Type.Charge_With_Balancing
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Charge_With_Balancing
                    ' Calculate the charge with balancing (can only get average charge from the entire stack)
                    trade_show_demo_charge_w_balancing = Calculate_Stack_Charge(TRADE_SHOW_DEMO_LOW_VOLTAGE) / D33001.MAX_CELLS
                    balChargeBox.Text = (trade_show_demo_charge_w_balancing).ToString("##.0") + "As (+" + _
                                        (100 * (trade_show_demo_charge_w_balancing - trade_show_demo_charge_wo_balancing) / trade_show_demo_charge_wo_balancing).ToString("##") + "%)"

                    If (Charged = True) AndAlso (Balanced = True) Then

                        If trade_show_demo_timer >= TRADE_SHOW_DEMO_CHARGED_TIME AndAlso
                           (trade_show_demo_timer >= TRADE_SHOW_DEMO_BALANCED_TIME) Then

                            ' Stop charging with balancing.
                            ChgBalBtn_Click(sender, e)
                            NoneBtn_Click(sender, e)

                            ' Display message about state transition
                            D33001.NoteLabel.Text = "End of Demo"
                            addOpDemoText(CapGraph.ChartAreas(0).AxisX.Maximum, 1.5, 50, "End of" + ChrW(13) & ChrW(10) + "Demo")

                            trade_show_demo_state = Trade_Show_Demo_State_Type.Charge_Delay
                            trade_show_demo_timer = 0
                        End If
                    Else
                        trade_show_demo_timer = 0
                    End If

                Case Trade_Show_Demo_State_Type.Charge_Delay
                    If (trade_show_demo_timer >= TRADE_SHOW_DELAY_TIME) Then
                        ' Turn off graph at end so that annotations don't look bad at the end.
                        Enable_CheckBox.Checked = False

                        ' At end of trade show demo, stop as if someone had cancelled it manually.
                        Trade_Show_Demo_State_Machine_Stop()
                    End If

                Case Else
                    ' If unknown state, stop as if someone had cancelled trade show demo manually.
                    Trade_Show_Demo_State_Machine_Stop()

            End Select

            ' Display debug data
            '            If D33001.DEBUG = True Then
            '            Debug_TextBox.Text = [Enum].GetName(GetType(Trade_Show_Demo_State_Type), trade_show_demo_state) + ", " + _
            '                                "Time Step = " + trade_show_demo_timestamp_step.ToString() + ", " + _
            '                                "Balanced = " + Balanced.ToString() + ", " + _
            '                                "Floating = " + Floating.ToString() + ", " + _
            '                                "Discharged = " + Discharged.ToString() + ", " + _
            '                                "Charged = " + Charged.ToString()
            '            End If
        Catch ex As Exception
            D33001.Handle_Exception(ex)
        End Try

    End Sub

    'Adds an annotation to the graph during the operational demo (yValue is from the top down...aka the greater the value the lower the text)
    'xValue input should be based on the current graph maximum, but you can go higher than the max and the graph will not have an error
    Private Sub addOpDemoText(ByVal xValue As Single, ByVal xValueOffset As Single, ByVal yValue As Single, ByVal text As String)
        Dim newText As DataVisualization.Charting.TextAnnotation = New DataVisualization.Charting.TextAnnotation
        newText.Name = [Enum].GetName(GetType(Trade_Show_Demo_State_Type), trade_show_demo_state)
        newText.Text = text
        CapGraph.Annotations.Add(newText)
        CapGraph.Annotations(newText.Name).ClipToChartArea = "ChartArea1"
        CapGraph.Annotations(newText.Name).Font = New System.Drawing.Font("Microsoft Sans Serif", 8.25!, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, CType(0, Byte))
        CapGraph.Annotations(newText.Name).AxisX = CapGraph.ChartAreas(0).AxisX
        CapGraph.Annotations(newText.Name).AnchorX = xValue
        CapGraph.Annotations(newText.Name).Alignment = System.Drawing.ContentAlignment.MiddleLeft
        CapGraph.Annotations(newText.Name).AnchorOffsetX = xValueOffset

        CapGraph.Annotations(newText.Name).ForeColor = Drawing.Color.DimGray
        CapGraph.Annotations(newText.Name).Y = yValue
    End Sub
#End Region

End Class