/*
 Linear Technology DC2100A Demonstration Board.
 This code contains the USB descriptors used by the DC2100A to identify itself to the connected PC.

 This code is used with a modified version of the CCS USB driver as the original driver did not support strings in RAM.

 http://www.linear.com/solutions/5126

 REVISION HISTORY
 $Revision: 542 $
 $Date: 2014-07-31 11:57:59 -0400 (Thu, 31 Jul 2014) $

 Copyright (c) 2013, Linear Technology Corp.(LTC)
 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, this
    list of conditions and the following disclaimer.
 2. Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 The views and conclusions contained in the software and documentation are those
 of the authors and should not be interpreted as representing official policies,
 either expressed or implied, of Linear Technology Corp.

*/

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Includes
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "Typedefs.h"
#include "DC2100A.h"
#include "USB_Descriptors.h"
#include "USB_Parser.h"
#include "USB.h"
#include "System.h"

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Definitions
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#define USB_DESCRIPTOR_LANGUAGE_STRING         {0x09, 0x04} // USB-IF LANGID defintion specifies 0x0409 as English (United States)
#define USB_DESCRIPTOR_LANGUAGE_STRING_SIZE    2            // sizeof(USB_DESCRIPTOR_LANGUAGE_STRING)

#define USB_DESCRIPTOR_STRING_HEADER_SIZE      2            // 2 bytes for string type identifier and string length

// Structure to track the characters in a string descriptor as they are sent to the USB driver.
// Note - this struct breaks if any strings are larger than 255 bytes, as the int16* passed in is incremented as an int16 by USB.c.
typedef struct
{
    unsigned int8 index;                                    // Index into string.
    unsigned int8 num;                                      // string number
} USB_DESCRIPTOR_STRING_INDEX_TYPE;

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Global Data
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Local Data
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const char usb_descriptor_string_language_data[USB_DESCRIPTOR_LANGUAGE_STRING_SIZE]     = USB_DESCRIPTOR_LANGUAGE_STRING;
const char usb_descriptor_string_manufacturer_data[DC2100A_COMPANY_STRING_SIZE]         = DC2100A_COMPANY_STRING;
const char usb_descriptor_string_product_data[DC2100A_IDSTRING_SIZE]                    = DC2100A_IDSTRING;
const char usb_descriptor_string_serial_number_data[DC2100A_SERIAL_NUM_SIZE]            = DC2100A_SERIAL_NUM_DEFAULT;

// Length of the strings, all unicode except for the language string, with 2 bytes added for USB string descriptor header
const char usb_descriptor_string_lengths[USB_DESCRIPTOR_NUM_STRINGS] = {                        \
        USB_DESCRIPTOR_LANGUAGE_STRING_SIZE + USB_DESCRIPTOR_STRING_HEADER_SIZE,                \
        DC2100A_COMPANY_STRING_SIZE * UNICODE_PER_ASCII + USB_DESCRIPTOR_STRING_HEADER_SIZE,    \
        DC2100A_IDSTRING_SIZE * UNICODE_PER_ASCII + USB_DESCRIPTOR_STRING_HEADER_SIZE,          \
        DC2100A_SERIAL_NUM_SIZE * UNICODE_PER_ASCII + USB_DESCRIPTOR_STRING_HEADER_SIZE};

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Local Prototypes
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Global Functions
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// Setup structure to track the characters in a string descriptor as they are sent to the USB driver.
// Return the length of the string, as the USB driver uses this as the limit for a loop.
unsigned int16 Usb_Descriptor_String_Setup(unsigned int8  string_num, unsigned int16* usb_mem_ptr)
{
    int8 string_length;
    USB_DESCRIPTOR_STRING_INDEX_TYPE* string_index_ptr;

    if(string_num < USB_DESCRIPTOR_NUM_STRINGS)
    {
        // use memory already reserved by USB driver to keep track of string descriptor being sent to the USB
        string_index_ptr = (USB_DESCRIPTOR_STRING_INDEX_TYPE*)usb_mem_ptr;

        string_index_ptr->num = string_num;
        string_index_ptr->index = 0;

        string_length = usb_descriptor_string_lengths[string_index_ptr->num];
    }
    else
    {
        string_length = 0;
    }

    return (int16) string_length;
}

// Gets one character from a string descriptor.
// Return the indexed character from the string.
char Usb_Descriptor_String_Get_Char(unsigned int16* usb_mem_ptr)
{
    char ret_char;
    USB_DESCRIPTOR_STRING_INDEX_TYPE* string_index_ptr;
    int8 adjusted_index;

    // use memory already reserved by USB driver to keep track of string descriptor being sent to the USB
    string_index_ptr = (USB_DESCRIPTOR_STRING_INDEX_TYPE*)usb_mem_ptr;

    if(string_index_ptr->index == 0)
    {
        ret_char = usb_descriptor_string_lengths[string_index_ptr->num];
    }
    else if(string_index_ptr->index == 1)
    {
        ret_char = USB_DESC_STRING_TYPE;
    }
    else
    {
        adjusted_index = string_index_ptr->index - USB_DESCRIPTOR_STRING_HEADER_SIZE;

        // Only string 0 is not Unicode
        if(string_index_ptr->num == USB_DESCRIPTOR_STRING_LANGUAGE)
        {
            ret_char = usb_descriptor_string_language_data[adjusted_index];
        }
        else
        {
            // All of the Unicode strings use 0 for their odd bytes
            if(string_index_ptr->index & 1)
            {
                ret_char = 0;
            }
            else
            {
                adjusted_index = adjusted_index / UNICODE_PER_ASCII;
                switch (string_index_ptr->num)
                {
                    case USB_DESCRIPTOR_STRING_MANUFACTURER:
                        ret_char = usb_descriptor_string_manufacturer_data[adjusted_index];
                        break;

                    case USB_DESCRIPTOR_STRING_PRODUCT:
                        {
                            if((System_Pic_Board_Mfg_Data_Valid == TRUE) && (adjusted_index < DC2100A_MODEL_NUM_SIZE))
                            {
                                ret_char = System_Pic_Board_Mfg_Data.model_num[adjusted_index];
                            }
                            else
                            {
                                ret_char = usb_descriptor_string_product_data[adjusted_index];
                            }
                        }
                        break;

                    case USB_DESCRIPTOR_STRING_SERIAL_NUMBER:
                        {
                            if(System_Pic_Board_Mfg_Data_Valid == TRUE)
                            {
                                ret_char = System_Pic_Board_Mfg_Data.serial_num[adjusted_index];
                            }
                            else
                            {
                                ret_char = usb_descriptor_string_serial_number_data[adjusted_index];
                            }
                        }
                        break;

                    default:
                        ret_char = 0;
                        break;
                }
            }
        }
    }

    return ret_char;
}

//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Local Functions
//-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

