/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      main.c
*
* Description:    Main file for the demo application
*
* Revision History:
*
*   01/14/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include "gModes.h"


#include <SI32_PBCFG_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
#include <si32_device.h>
#include "myCPU.h"
#include "myLCD0.h"
#include "mySensor.h"
#include "myPB.h"
#include "myTrace.h"


/* Pause button */
#define S4      0x0080    /* PB1.7 */


/*****************************************************************************/
/* PutTemperature                                                            */
/*****************************************************************************/

void PutTemperature( int Temperature )
{
   int  WholeDegree;
   int  DeciDegree;
   bool Negative = false;

   /* Clear the display */
   myLCD0_put_char('\r');
   myLCD0_put_char('\n');

   /* If the scale is Fahrenheit then do the conversion */
   if ( Scale == 'F' )
      Temperature = ((Temperature*9)/5) + 320;
   
   /* Normalize the temperature to a positive number */
   if ( Temperature < 0 )
   {
      Temperature = -Temperature;
      Negative = true;
   }

   /* Extract the whole and fractional components */
   WholeDegree = Temperature / 10;
   DeciDegree  = Temperature % 10;

   /* If the temperature is negative */
   if ( Negative )
   {
      myLCD0_put_char('-');
      if ( WholeDegree >= 10 )
      {
         if ( DeciDegree >= 5 )
            WholeDegree++;
         myLCD0_put_number( WholeDegree, 2 );
      }
      else /* Negative but greater than -10 */
      {
         myLCD0_put_number( WholeDegree, 1 );
         myLCD0_put_char('.');
         myLCD0_put_number( DeciDegree, 1 );
      }
   }
   /* If the temperature is a triple digit whole number */
   else if ( WholeDegree >= 100 )
   {
      if ( DeciDegree >= 5 )
         WholeDegree++;
      myLCD0_put_number( WholeDegree, 3 );
   }
   else /* Positive but less than a hundred */
   {
      myLCD0_put_number( WholeDegree, 2 );
      myLCD0_put_char('.');
      myLCD0_put_number( DeciDegree, 1 );
   }

   myLCD0_put_char('\'');
   myLCD0_put_char( Scale ); 
   
   if ( Thermistor )
      myLCD0_put_char('.');
}


/*****************************************************************************/
/* PutHumidity                                                               */
/*****************************************************************************/

void PutHumidity( int Humidity )
{
   int WholePercent;
   int DeciPercent;

   /* Extract the whole and fractional components */
   WholePercent = Humidity / 10;
   DeciPercent  = Humidity % 10;

   /* Round up to the nearest percent */   
   if ( DeciPercent >= 5 )
      WholePercent++;

   myLCD0_put_number( WholePercent, 3 );
   myLCD0_put_char('H');
}


//------------------------------------------------------------------------------
// My application.
//------------------------------------------------------------------------------
int main( void )
{
   int Temperature;
   int Humidity;
   int Result;
   
   /* Enter the default operating mode for this application */
   enter_default_mode_from_reset();
   Sensor_enter_default_mode_from_reset();
 
   /* Initialize the trace buffer */
   TraceReset();

   /* If the S4 button is pressed */
   if ( (SI32_PBSTD_A_read_pins(SI32_PBSTD_1) & S4) == 0 )
   {
      myLCD0_put_string("PAUSED");
      
      /* Pause until S4 is released */
      /* This gives the IDE or flash utility a chance to */
      /* get control before the MCU enters sleep mode    */
      while ( (SI32_PBSTD_A_read_pins(SI32_PBSTD_1) & S4) == 0 );
   }

   while ( true )
   {
      /* Wake up the sensor */
      Sensor_WakeUp( I2C0_BUS_1 );

      /* Read the temperature */
      if ( Thermistor )
         Result = Sensor_ReadThermistor( I2C0_BUS_1, &Temperature );
      else   
         Result = Sensor_ReadTemperature( I2C0_BUS_1, &Temperature );
      if ( Result == I2C0_STATUS_SUCCESS )
      {   
         /* Read the humidity */
         Result = Sensor_ReadHumidity( I2C0_BUS_1, Temperature, &Humidity );
         if ( Result == I2C0_STATUS_SUCCESS )
         {
            /* Display the temperature and humidity */
            PutTemperature( Temperature );
            PutHumidity( Humidity );
         }   
      }

      if ( Result != I2C0_STATUS_SUCCESS )
         myLCD0_put_string("ERROR");   

      /* Put the sensor back to sleep */
      Sensor_Sleep( I2C0_BUS_1 );

      /* Read the sensor twice a second */
      Delay( 500 );
   }
}

