/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Packet.cs
*
* Description:    The Packet class sends and received data from the Data Logger
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

using System;
using System.IO.Ports;
using System.Windows.Forms;


namespace Silabs.DataLogger
{
    public class Packet : ICloneable
    {
        /* Request */
        public const byte REQ_GET_ID                      = 1;
        public const byte REQ_GET_VERSION                 = 2;
        public const byte REQ_GET_STATUS                  = 3;
        public const byte REQ_GET_TEMPERATURE             = 4;
        public const byte REQ_GET_HUMIDITY                = 5;
        public const byte REQ_GET_TIME                    = 6;
        public const byte REQ_SET_TIME                    = 7;
        public const byte REQ_GET_TIME_ZONE               = 8;
        public const byte REQ_SET_TIME_ZONE               = 9;
        public const byte REQ_GET_SAMPLE_INTERVAL         = 10;
        public const byte REQ_SET_SAMPLE_INTERVAL         = 11;
        public const byte REQ_GET_HIGH_TEMP_THRESHOLD     = 12;
        public const byte REQ_SET_HIGH_TEMP_THRESHOLD     = 13;
        public const byte REQ_GET_LOW_TEMP_THRESHOLD      = 14;
        public const byte REQ_SET_LOW_TEMP_THRESHOLD      = 15;
        public const byte REQ_GET_HIGH_HUMIDITY_THRESHOLD = 16;
        public const byte REQ_SET_HIGH_HUMIDITY_THRESHOLD = 17;
        public const byte REQ_GET_LOW_HUMIDITY_THRESHOLD  = 18;
        public const byte REQ_SET_LOW_HUMIDITY_THRESHOLD  = 19;
        public const byte REQ_GET_LOGGING                 = 20;
        public const byte REQ_SET_LOGGING                 = 21;
        public const byte REQ_GET_SAMPLE_COUNT            = 22;
        public const byte REQ_GET_LOG                     = 23;
        public const byte REQ_ERASE_LOG                   = 24;
        public const byte REQ_SET_CONNECTED               = 25;

        /* The Receive method returns true for success */
        public const bool Success = true;
        public const bool Failure = false;

        /* Data field contains a parameter */
        public const int IsParameter = -1;

        /* Offsets of fields within a packet */
        private const int offsetSOP = 0;
        private const int offsetRequest = 1;
        private const int offsetDataLength = 2;
        private const int offsetData = 4;

        /* Size constants */
        private const int sizeHeader = 5;
        private const int sizeParameter = 4;
        private const int sizeSample = 8;
        private const int sizeInt = 4;
        private const int sizeMaxUShort = 65535;

        /* Start-of-Packet constant */
        private const byte SOP = 0xAC;

        /* Private data members */
        private byte[] Buffer;
        private byte[] Field;
        private PacketSample[] sample;

        /* Constructor - Default packet type is the parameter packet */
        public Packet() : this(IsParameter) { }

        /* Constructor */
        public Packet(int SampleCount)
        {
            int DataLength;

            if (SampleCount == IsParameter)
                DataLength = sizeParameter;
            else
                DataLength = sizeSample * SampleCount;

            /* Allocate the packet buffer */
            Buffer = new byte[sizeHeader + DataLength];

            /* Adjust DataLength to fit within a ushort */
            if (DataLength > sizeMaxUShort)
                DataLength = sizeMaxUShort;

            /* Fill in the packet header */
            Buffer[offsetSOP] = SOP;
            SetUShort((ushort)DataLength, offsetDataLength);

            /* If there are samples */
            if (SampleCount > 0)
            {
                /* Create a list of samples */
                sample = new PacketSample[SampleCount];
                for (int Index = 0; Index < SampleCount; Index++)
                    sample[Index] = new PacketSample(Buffer, offsetData, Index);
            }
            else Parameter = 0;

            /* Allocate miscellaneous stuff */
            Field = new byte[sizeInt];
        }

        /* Private Constructor - for cloning */
        private Packet(byte[] OldBuffer)
        {
            /* Allocate a new packet buffer */
            Buffer = new byte[OldBuffer.Length];

            /* Copy the contents of the old buffer to the new buffer */
            Array.Copy(OldBuffer, Buffer, Buffer.Length);

            /* Find the number of samples */
            int SampleCount = (Buffer.Length - sizeHeader) / sizeSample;

            /* If there are samples */
            if (SampleCount > 0)
            {
                /* Create a list of samples */
                sample = new PacketSample[SampleCount];
                for (int Index = 0; Index < SampleCount; Index++)
                    sample[Index] = new PacketSample(Buffer, offsetData, Index);
            }

            /* Allocate miscellaneous stuff */
            Field = new byte[sizeInt];
        }

        /* Clone method - with deep copy */
        public object Clone()
        {
            return new Packet(Buffer);
        }

        /* Receive method */
        public bool Receive(SerialPort SerialPort )
        {
           return Receive( SerialPort, null );
        }

        /* Receive method */
        public bool Receive(SerialPort SerialPort, ProgressBar progressBar )
        {
            int Count;
            int Offset;
            byte Sum = 0;

            /* Read until the Start-of-Packet character is found */
            Buffer[offsetSOP] = 0;
            while (Buffer[offsetSOP] != SOP)
            {
                Count = SerialPort.Read(Buffer, offsetSOP, 1);
                if (Count == 0) return Failure;
            }

            /* Read the Request and DataLength fields */
            for (Offset = offsetRequest; Offset < offsetData; Offset += Count)
            {
                Count = SerialPort.Read(Buffer, Offset, offsetData - Offset);
                if (Count == 0) return Failure;
            }

            /* Read the Data and Checksum fields */
            for (Offset = offsetData; Offset < Buffer.Length; Offset += Count)
            {
                if ( progressBar != null )
                   progressBar.Value = Offset*100 / Buffer.Length;

                Count = SerialPort.Read(Buffer, Offset, Buffer.Length - Offset);
                if (Count == 0)return Failure;
            }

            /* Calculate the checksum */
            for (Offset = offsetRequest; Offset < Buffer.Length; Offset++)
                Sum += Buffer[Offset];

            /* If the checksum is bad */
            if (Sum != 0)
                return Failure;

            return Success;
        }

        /* Send method */
        public void Send(SerialPort SerialPort)
        {
            int Offset;
            byte Sum = 0;

            /* The Checksum field is the last byte in the packet */
            int offsetChecksum = Buffer.Length - 1;

            /* Calculate the checksum */
            for (Offset = offsetRequest; Offset < offsetChecksum; Offset++)
                Sum += Buffer[Offset];

            /* Store the checksum in the Checksum field */
            Buffer[offsetChecksum] = (byte)((~Sum) + 1);

            /* Transmit the packet to the data logger */
            SerialPort.Write(Buffer, 0, Buffer.Length);
        }

        /* Request property */
        public byte Request
        {
            get { return Buffer[offsetRequest]; }
            set { Buffer[offsetRequest] = value; }
        }

        /* Parameter property */
        public int Parameter
        {
            get { return GetInt(offsetData); }
            set { SetInt(value, offsetData); }
        }

        /* Sample property */
        public PacketSample[] Sample
        {
            get { return sample; }
        }

        /* GetInt method */
        private int GetInt(int Offset)
        {
            for (int x = 0; x < sizeInt; x++)
                Field[3 - x] = Buffer[Offset + x];
            return BitConverter.ToInt32(Field, 0);
        }

        /* SetInt method */
        private void SetInt(int Value, int Offset)
        {
            byte[] Source = BitConverter.GetBytes(Value);
            for (int x = 0; x < sizeInt; x++)
                Buffer[Offset + x] = Source[3 - x];
        }

        /* GetUShort method */
        private ushort GetUShort(int Offset)
        {
            Field[0] = Buffer[Offset + 1];
            Field[1] = Buffer[Offset + 0];
            return BitConverter.ToUInt16(Field, 0);
        }

        /* SetUShort method */
        private void SetUShort(ushort Value, int Offset)
        {
            byte[] Source = BitConverter.GetBytes(Value);
            Buffer[Offset + 0] = Source[1];
            Buffer[Offset + 1] = Source[0];
        }
    }
}

