/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      DataLogger.cs
*
* Description:    The DataLogger class represents the Data Logger board
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

using System;
using System.IO;
using System.IO.Ports;
using System.Windows.Forms;


namespace Silabs.DataLogger
{
    public class DataLogger : IDisposable
    {
        /* Private constants */
        private const int SEC_PER_MINUTE = 60;
        private const int SEC_PER_HOUR = SEC_PER_MINUTE * 60;
        private const int SEC_PER_DAY = SEC_PER_HOUR * 24;

        /* Public constansts */
        public const int TIMER_MODE  =  16;
        public const int NO_TIMEZONE = -16;

        /* Private data members */
        private bool disposed;
        private SerialPort serialPort;
        private Packet packet;
        private Sample[] sample;

        /* Constructor - Use default COM port */
        public DataLogger() : this("COM6") { }

        /* Constructor */
        public DataLogger( string portName )
        {
            /* Resources are not yet disposed of */
            disposed = false;

            /* Create the serial port object */
            serialPort = new SerialPort( portName, 9600 );

            /* Set read timeout to 1 second */
            serialPort.ReadTimeout = 1000;

            /* Open the serial port */
            serialPort.Open();

            /* Wake up the data logger */
            serialPort.RtsEnable = true;

            /* Create a packet object */
            packet = new Packet();
            
            /* Tell the FW that we are connected */
            Connected = true;
        }

        /* Dispose of resources */
        public void Dispose()
        {
            if (!disposed)
            {
                /* Tell the FW that we are disconnected */
                Connected = false;
                
                /* Let the data logger sleep */
                serialPort.RtsEnable = false;

                /* Close the serial port */
                serialPort.Close();

                disposed = true;
            }
        }

        /* Deconstructor */
        ~DataLogger()
        {
            Dispose();
        }

        /* ID property */
        public int ID
        {
            get
            {
                packet.Request = Packet.REQ_GET_ID;
                packet.Parameter = 0;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            return packet.Parameter;
            }
        }

        /* Version property */
        public float Version
        {
            get
            {
                packet.Request = Packet.REQ_GET_VERSION;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return (float)(packet.Parameter >> 8) + ((float)(packet.Parameter & 0xFF)) / 10;
            }
        }

        /* Status property */         
        public int Status
        {
            get 
            { 
                packet.Request = Packet.REQ_GET_STATUS;
                packet.Send( serialPort );
                if ( packet.Receive(serialPort) != Packet.Success )
                    throw new IOException("Cannot receive the response packet");
                return packet.Parameter;
            }
        }

        /* Temperature property */         
        public float Temperature
        {
            get 
            { 
                packet.Request = Packet.REQ_GET_TEMPERATURE;
                packet.Send( serialPort );
                if ( packet.Receive(serialPort) != Packet.Success )
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter)/10;
            }
        }

        /* Humidity property */         
        public float Humidity
        {
            get 
            { 
                packet.Request = Packet.REQ_GET_HUMIDITY;
                packet.Send( serialPort );
                if ( packet.Receive(serialPort) != Packet.Success )
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter)/10;
            }
        }

        /* SecondsInYear method */
        private uint SecondsInYear(int Year)
        {
            if ((Year % 4) == 0)
                return 366 * SEC_PER_DAY;
            else
                return 365 * SEC_PER_DAY;
        }

        /* SecondsInMonth method */
        private uint SecondsInMonth(int Month, int Year)
        {
            if (Month == 2)
            {
                if ((Year % 4) == 0)
                    return 29 * SEC_PER_DAY;
                else
                    return 28 * SEC_PER_DAY;
            }
            else if (Month == 4 || Month == 6 || Month == 9 || Month == 11)
            {
                return 30 * SEC_PER_DAY;
            }
            else
            {
                return 31 * SEC_PER_DAY;
            }
        }

        /* Time property */
        public DateTime Time
        {
            get
            {
                if (Timezone == TIMER_MODE)
                    throw new IOException("Must be in clock mode to get the time");

                packet.Request = Packet.REQ_GET_TIME;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");

                uint SysTime = (uint)packet.Parameter;
                int Year;
                int Month;

                for (Year = 0; SysTime >= SecondsInYear(Year); Year++)
                    SysTime -= SecondsInYear(Year);

                for (Month = 1; SysTime >= SecondsInMonth(Month, Year); Month++)
                    SysTime -= SecondsInMonth(Month, Year);

                int Day = (int)(SysTime / SEC_PER_DAY) + 1; SysTime %= SEC_PER_DAY;
                int Hour = (int)(SysTime / SEC_PER_HOUR); SysTime %= SEC_PER_HOUR;
                int Minute = (int)(SysTime / SEC_PER_MINUTE); SysTime %= SEC_PER_MINUTE;
                int Second = (int)SysTime;

                return new DateTime(2000 + Year, Month, Day, Hour, Minute, Second);
            }
            set
            {
                uint SysTime = 0;
                int x;

                for (x = 2000; x < value.Year; x++)
                    SysTime += SecondsInYear(x);

                for (x = 1; x < value.Month; x++)
                    SysTime += SecondsInMonth(x, value.Year);

                SysTime += (uint)(value.Day - 1) * SEC_PER_DAY;
                SysTime += (uint)value.Hour * SEC_PER_HOUR;
                SysTime += (uint)value.Minute * SEC_PER_MINUTE;
                SysTime += (uint)value.Second;

                packet.Request = Packet.REQ_SET_TIME;
                packet.Parameter = (int)SysTime;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");

                TimeZone localZone = TimeZone.CurrentTimeZone;
                TimeSpan localOffset = localZone.GetUtcOffset(value);
                Timezone = localOffset.Hours;
            }
        }

        /* Timer property */
        public TimeSpan Timer
        {
            get
            {
                if (Timezone != TIMER_MODE)
                    throw new IOException("Must be in timer mode to get the timer");

                packet.Request = Packet.REQ_GET_TIME;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");

                int Timer = (int)packet.Parameter;
                int Days = Timer / SEC_PER_DAY; Timer %= SEC_PER_DAY;
                int Hours = Timer / SEC_PER_HOUR; Timer %= SEC_PER_HOUR;
                int Minutes = Timer / SEC_PER_MINUTE; Timer %= SEC_PER_MINUTE;
                int Seconds = Timer;

                return new TimeSpan(Days, Hours, Minutes, Seconds);
            }
            set
            {
                int Timer = 0;

                Timer += value.Days * SEC_PER_DAY;
                Timer += value.Hours * SEC_PER_HOUR;
                Timer += value.Minutes * SEC_PER_MINUTE;
                Timer += value.Seconds;

                packet.Request = Packet.REQ_SET_TIME;
                packet.Parameter = Timer;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");

                Timezone = TIMER_MODE;
            }
        }

        /* Timezone property */
        public int Timezone
        {
            get
            {
                packet.Request = Packet.REQ_GET_TIME_ZONE;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return packet.Parameter;
            }
            set
            {
                packet.Request = Packet.REQ_SET_TIME_ZONE;
                packet.Parameter = value;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* SampleInterval property */
        public int SampleInterval
        {
            get
            {
                packet.Request = Packet.REQ_GET_SAMPLE_INTERVAL;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return packet.Parameter;
            }
            set
            {
                packet.Request = Packet.REQ_SET_SAMPLE_INTERVAL;
                packet.Parameter = value;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* HighTemperatureThreshold property */
        public float HighTemperatureThreshold
        {
            get
            {
                packet.Request = Packet.REQ_GET_HIGH_TEMP_THRESHOLD;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter) / 10;
            }
            set
            {
                packet.Request = Packet.REQ_SET_HIGH_TEMP_THRESHOLD;
                packet.Parameter = (int)Math.Round(value * 10);
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* LowTemperatureThreshold property */
        public float LowTemperatureThreshold
        {
            get
            {
                packet.Request = Packet.REQ_GET_LOW_TEMP_THRESHOLD;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter) / 10;
            }
            set
            {
                packet.Request = Packet.REQ_SET_LOW_TEMP_THRESHOLD;
                packet.Parameter = (int)Math.Round(value * 10);
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* HighHumidityThreshold property */
        public float HighHumidityThreshold
        {
            get
            {
                packet.Request = Packet.REQ_GET_HIGH_HUMIDITY_THRESHOLD;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter) / 10;
            }
            set
            {
                packet.Request = Packet.REQ_SET_HIGH_HUMIDITY_THRESHOLD;
                packet.Parameter = (int)Math.Round(value * 10);
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* LowHumidityThreshold property */
        public float LowHumidityThreshold
        {
            get
            {
                packet.Request = Packet.REQ_GET_LOW_HUMIDITY_THRESHOLD;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return ((float)packet.Parameter) / 10;
            }
            set
            {
                packet.Request = Packet.REQ_SET_LOW_HUMIDITY_THRESHOLD;
                packet.Parameter = (int)Math.Round(value * 10);
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* Logging property */
        public bool Logging
        {
            get
            {
                packet.Request = Packet.REQ_GET_LOGGING;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                if (packet.Parameter == 0)
                    return false;
                else
                    return true;
            }
            set
            {
                packet.Request = Packet.REQ_SET_LOGGING;
                if (value)
                    packet.Parameter = 1;
                else
                    packet.Parameter = 0;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
            }
        }

        /* SampleCount property */
        public int SampleCount
        {
            get
            {
                packet.Request = Packet.REQ_GET_SAMPLE_COUNT;
                packet.Send(serialPort);
                if (packet.Receive(serialPort) != Packet.Success)
                    throw new IOException("Cannot receive the response packet");
                return packet.Parameter;
            }
        }

        /* Connected property */
        public bool Connected
        {
            set
            {
                if ( packet != null )
                {
                    try
                    {
                        packet.Request = Packet.REQ_SET_CONNECTED;
                        if (value)
                            packet.Parameter = 1;
                        else
                            packet.Parameter = 0;
                        packet.Send(serialPort);
                        if (packet.Receive(serialPort) != Packet.Success)
                            throw new IOException("Cannot receive the response packet");
                    }
                    catch {}    
                }    
            }
        }

        /* Sample property */
        public Sample[] Sample
        {
            get { return sample; }
        }

        /* GetLog method */
        public void GetLog( ProgressBar progressBar )
        {
            /* Check if logging is enabled */
            bool LoggingEnabled = Logging;

            /* If logging is enabled then temporarily disable it */
            if (LoggingEnabled)
                Logging = false;

            /* Create a packet to receive the log */
            Packet logPacket = new Packet(SampleCount);

            /* Send the request to get the log */
            packet.Request = Packet.REQ_GET_LOG;
            packet.Send(serialPort);

            /* Receive the log packet */
            if (logPacket.Receive(serialPort,progressBar) != Packet.Success)
                throw new IOException("Cannot receive the log packet");

            /* If logging was enabled then re-enable it */
            if (LoggingEnabled)
                Logging = true;

            /* Get the timezone */
            int timezone = Timezone;

            /* Create an array of samples */
            sample = new Sample[logPacket.Sample.Length];
            for (int x = 0; x < sample.Length; x++)
                sample[x] = new Sample(logPacket.Sample[x], timezone);
        }
      
        /* SaveLog method */
        public void SaveLog( string Filename )
        {
            FileStream   stream = new FileStream(Filename, FileMode.Create );
            StreamWriter writer = new StreamWriter( stream );

            if ( Timezone == TIMER_MODE )
            {
                /* Write a header to the first line of the file */
                writer.WriteLine("Timerstamp,Temperature,Humidity");

                /* Write all the samples to the file, one per line */      
                foreach ( Sample s in sample )
                    writer.WriteLine("{0},{1:F1},{2:F1}", s.TimerStamp, 
                        s.Temperature, s.Humidity );
            }
            else /* Clock mode */
            {
                /* Write a header to the first line of the file */
                writer.WriteLine("Timestamp,Temperature,Humidity");

                /* Write all the samples to the file, one per line */      
                foreach ( Sample s in sample )
                    writer.WriteLine("{0},{1:F1},{2:F1}", s.TimeStamp, 
                        s.Temperature, s.Humidity );
            }      

            writer.Close();
            stream.Close();
        }
      
        /* RestoreLog method */
        public void RestoreLog( string Filename )
        {
            FileStream   stream = new FileStream(Filename, FileMode.Open );
            StreamReader reader = new StreamReader( stream );

            /* Read the header from the file */
            string header = reader.ReadLine();

            int timezone;

            /* Validate the header */
            if ( header.Equals("Timestamp,Temperature,Humidity") )
                timezone = 0;
            else if ( header.Equals("Timerstamp,Temperature,Humidity") )
                timezone = TIMER_MODE;   
            else throw new IOException(
                String.Format("File '{0}' does not contain the correct header",Filename) );

            /* Count the number of samples in the file */
            int count = 0;
            while ( reader.ReadLine() != null ) 
                count++;

            /* Go back to the second line of the file */
            stream.Position = 0;
            reader.DiscardBufferedData();   
            reader.ReadLine();

            /* There are three values per line */
            string[] values;

            /* Create an array of samples */   
            sample = new Sample[count];
            for ( int x=0; x<count; x++ )
            {
                values = reader.ReadLine().Split(',');
                if ( timezone == TIMER_MODE )
                {
                    sample[x] = new Sample( TimeSpan.Parse(values[0]),
                        Single.Parse(values[1]), Single.Parse(values[2]) );
                }      
                else /* Clock mode */
                {      
                    sample[x] = new Sample( DateTime.Parse(values[0]),
                        Single.Parse(values[1]), Single.Parse(values[2]) );
                }      
            }   

            reader.Close();
            stream.Close();
        }

        /* EraseLog method */
        public void EraseLog()
        {
            packet.Request = Packet.REQ_ERASE_LOG;
            packet.Send(serialPort);
            if (packet.Receive(serialPort) != Packet.Success)
                throw new IOException("Cannot receive the response packet");
        }
    }
}

