/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myUART0.c
*
* Description:    Serial port driver
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <SI32_UART_B_Type.h>
#include <si32_device.h>
#include "main.h"
#include "myCpu.h"
#include "myUART0.h"


#define TX_BUFF_SIZE  16
#define RX_BUFF_SIZE  16

/* The most significant bit of TCNT indicates if the transmit FIFO is full */
#define SI32_UART_B_FIFOCN_TX_FIFO_FULL  0x00040000

/* Macro to check if the transmit FIFO is not full */
#define SI32_UART_B_is_tx_fifo_not_full(basePointer) \
   ((bool)!(basePointer->FIFOCN.U32 & SI32_UART_B_FIFOCN_TX_FIFO_FULL))

/* Macro to check if the receive FIFO is not empty */
#define SI32_UART_B_is_rx_fifo_not_empty(basePointer) \
   ((bool)(basePointer->FIFOCN.U32 & SI32_UART_B_FIFOCN_RCNT_MASK))

/* Transmit and receive buffers */
uint8_t TxBuff[TX_BUFF_SIZE];
uint8_t RxBuff[RX_BUFF_SIZE];

/* Pointers into the transmit and receive buffers */
uint8_t *pTxIn  = TxBuff;
uint8_t *pTxOut = TxBuff;
uint8_t *pRxIn  = RxBuff;
uint8_t *pRxOut = RxBuff;

/* Error counters */
int RxLost    = 0;
int RxTimeout = 0;

/* Forward references */
uint8_t *IncTx( uint8_t *pTx );
uint8_t *IncRx( uint8_t *pRx );


/*****************************************************************************/
/* UART0_Send                                                                */
/*****************************************************************************/

int UART0_Send( uint8_t *pData, int DataSize )
{
   uint8_t *pTemp;
   int DataSent;
   
   /* For each byte in the caller's buffer */
   for ( DataSent=0; DataSent<DataSize; DataSent++ )
   {
      /* Process events while the transmit buffer is full */
      pTemp = IncTx( pTxIn );
      while ( pTemp == pTxOut )
         ProcessEvents();
      
      /* Copy the byte from the caller's buffer to the transmit buffer */
      *pTxIn = *pData++;
      pTxIn  = pTemp;

      /* Make sure that the tx data request interrupt is enabled */
      SI32_UART_B_enable_tx_data_request_interrupt( SI32_UART_0 );
   }  

   /* Return the number of bytes copied from the caller's buffer */
   return DataSent;      
}


/*****************************************************************************/
/* UART0_Receive                                                             */
/*****************************************************************************/

int UART0_Receive( uint8_t *pData, int DataSize, bool Timeout )
{
   uint32_t ReceiveTime;
   int DataReceived;
   
   /* Note when the receive was started */
   ReceiveTime = get_msTicks();

   /* While the receive buffer is empty */
   while ( pRxIn == pRxOut )
   {
      /* If timeout is enabled and no data was received within the last millisecond */
      if ( Timeout && (ElapsedTime(ReceiveTime)>2) )
      {
         RxTimeout++;
         return 0;
      }

      /* Process events while waiting for a byte to be received */
      ProcessEvents();
   }

   /* While the caller's buffer is not full */
   for ( DataReceived=0; DataReceived<DataSize; DataReceived++ )
   {
      /* If there is another byte in the receive buffer */
      if ( pRxOut != pRxIn )
      {
         /* Copy the byte from the receive buffer to the caller's buffer */
         *pData++ = *pRxOut;
         pRxOut = IncRx( pRxOut );

         /* Make sure that the rx data request interrupt is enabled */
         SI32_UART_B_enable_rx_data_request_interrupt( SI32_UART_0 );
      }
      else /* The receive buffer is now empty */
      {
         /* Exit the loop */
         break;
      }
   }

   /* Return the number of bytes copied to the caller's buffer */
   return DataReceived;
}


/*****************************************************************************/
/* UART0_tx_data_req_handler                                                 */
/*****************************************************************************/

void UART0_tx_data_req_handler( void )
{
   /* While the transmit buffer is not empty and the transmit FIFO is not full */
   while ( (pTxOut != pTxIn) && SI32_UART_B_is_tx_fifo_not_full(SI32_UART_0) )
   {
      /* Move the data from the transmit buffer to the transmit FIFO */
      SI32_UART_B_write_data_u8( SI32_UART_0, *pTxOut );
      pTxOut = IncTx( pTxOut );
   }

   /* If the transmit buffer is empty then disable the tx data req interrupt */
   if ( pTxOut == pTxIn )
      SI32_UART_B_disable_tx_data_request_interrupt( SI32_UART_0 );
}


/*****************************************************************************/
/* UART0_rx_data_req_handler                                                 */
/*****************************************************************************/

void UART0_rx_data_req_handler( void )
{
   uint8_t *pTemp;

   /* Temporarily increment pRxIn */
   pTemp = IncRx( pRxIn );

   /* While the recive buffer is not full and the receive FIFO is not empty */
   while ( (pTemp != pRxOut) && SI32_UART_B_is_rx_fifo_not_empty(SI32_UART_0) )
   {
      /* Move the data from the receive FIFO to the receive buffer */
      *pRxIn = SI32_UART_B_read_data_u8( SI32_UART_0 );
      
      /* Increment pRxIn */
      pRxIn = pTemp;
      pTemp = IncRx( pRxIn );
   }

   /* If the receive buffer is full then disable the rx data req interrupt */
   if ( pTemp == pRxOut )
      SI32_UART_B_disable_rx_data_request_interrupt( SI32_UART_0 );
}


/*****************************************************************************/
/* UART0_rx_error_handler                                                    */
/*****************************************************************************/

void UART0_rx_error_handler( void )
{
   /* Count the number of lost receive bytes */
   RxLost++;
   
   /* Clear the receive error interrupt */
   SI32_UART_B_clear_rx_overrun_error_interrupt( SI32_UART_0 );
}


/*****************************************************************************/
/* IncTx                                                                     */
/*****************************************************************************/

uint8_t *IncTx( uint8_t *pTx )
{
   /* Increment the transmit pointer with wraparound */
   if ( ++pTx == &TxBuff[TX_BUFF_SIZE] )
      pTx = TxBuff;
   return pTx;   
}


/*****************************************************************************/
/* IncRx                                                                     */
/*****************************************************************************/

uint8_t *IncRx( uint8_t *pRx )
{
   /* Increment the receive pointer with wraparound */
   if ( ++pRx == &RxBuff[RX_BUFF_SIZE] )
      pRx = RxBuff;
   return pRx;   
}


