/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myPacket.c
*
* Description:    Send and receive packets using the Data Logger Protocol
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include "myUART0.h"
#include "myLog.h"
#include "myPacket.h"


/* Start-of-Packet character */
#define SOP  0xAC


/*****************************************************************************/
/* Packet_Receive                                                            */
/*****************************************************************************/

int Packet_Receive( PACKET *pPacket )
{
   uint8_t *pByte;
   uint8_t *pLimit;
   uint8_t  Sum;
   int      Count;
   
   /* Clear the start field */
   pPacket->Start = 0;

   /* Receive until a Start-Of-Packet character is found */   
   while ( pPacket->Start != SOP )
   {
      /* Don't timeout when trying to receive the SOP */
      Count = UART0_Receive( &pPacket->Start, 1, false );
      if ( Count == 0 ) return PACKET_FAILURE;
   } 

   /* Receive the Request and DataLength fields */
   for ( pByte=&pPacket->Request,pLimit=pByte+3; pByte<pLimit; pByte+=Count )
   {
      Count = UART0_Receive( pByte, pLimit-pByte, true );
      if ( Count == 0 ) return PACKET_FAILURE;
   }

   /* Change the endianess of the DataLength field */
   pPacket->DataLength = Swap16( pPacket->DataLength );

   /* Receive the Data and Checksum fields */
   for ( pByte=pLimit,pLimit+=pPacket->DataLength+1; pByte<pLimit; pByte+=Count )
   {
      Count = UART0_Receive( pByte, pLimit-pByte, true );
      if ( Count == 0 ) return PACKET_FAILURE;
   }

   /* Change the endianess of the Paremeter field */
   pPacket->Parameter = Swap32( pPacket->Parameter );

   /* Calculate the checksum */
   for ( pByte=&pPacket->Request,Sum=0; pByte<pLimit; pByte++ )
      Sum += *pByte;      

   /* Validate the checksum */
   if ( Sum == 0 )
      return PACKET_SUCCESS;
   else
      return PACKET_FAILURE;   
}


/*****************************************************************************/
/* Packet_Send                                                               */
/*****************************************************************************/

int Packet_Send( PACKET *pPacket )
{
   uint8_t *pByte;
   uint8_t  Sum;

   /* Change the endianess of the DataLength field */
   pPacket->DataLength = Swap16( pPacket->DataLength );

   /* Change the endianess of the Paremeter field */
   pPacket->Parameter = Swap32( pPacket->Parameter );
   
   /* Calculate the checksum */      
   for ( pByte=&pPacket->Request,Sum=0; pByte<&pPacket->Checksum; pByte++ )
      Sum += *pByte;      

   /* Store the checksum in the packet */
   pPacket->Checksum = (~Sum) + 1;

   /* Send the packet */   
   UART0_Send( (uint8_t*)pPacket, 9 );

   return PACKET_SUCCESS;
}


/*****************************************************************************/
/* Packet_SendLog                                                            */
/*****************************************************************************/

int Packet_SendLog( PACKET *pPacket )
{
   uint8_t *pData;
   uint8_t *pDataLimit;
   uint8_t  Sum;
   
   /* Store the size of the log in the packet */
   if ( SampleCount == MAX_SAMPLE_COUNT )
      pPacket->DataLength = 65535;
   else   
      pPacket->DataLength = SampleCount * SAMPLE_SIZE;

   /* Change the endianess of the DataLength field */
   pPacket->DataLength = Swap16( pPacket->DataLength );

   /* Point just beyond the end of the packet header */
   pDataLimit = (uint8_t*)&pPacket->Parameter;

   /* Calculate the checksum across the header */      
   for ( pData=&pPacket->Request,Sum=0; pData<pDataLimit; pData++ )
      Sum += *pData;      

   /* Point just beyond the end of the log data */
   pDataLimit = (uint8_t*)pLog;

   /* Calculate the checkum across the log data */
   for ( pData=(uint8_t*)LOG_ADDRESS; pData<pDataLimit; pData++ )
      Sum += *pData;      

   /* Store the checksum in the packet */
   pPacket->Checksum = (~Sum) + 1;

   /* Send the packet header */
   UART0_Send( (uint8_t*)pPacket, 4 );

   /* Point to the beginning of log data */
   pData = (uint8_t*)LOG_ADDRESS;

   /* Send the log data (this may take a while) */
   UART0_Send( pData, pDataLimit-pData );

   /* Send the checksum */
   UART0_Send( &pPacket->Checksum, 1 );

   return PACKET_SUCCESS;
}

