/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myLog.c
*
* Description:    Save humidity and temperature samples in flash memory
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_FLASHCTRL_A_Type.h>
#include <SI32_VMON_B_Type.h>
#include <SI32_RSTSRC_A_Type.h>
#include "myLog.h"


/* Sample */
typedef struct
{
   uint32_t Timestamp;
   int16_t  Temperature;
   int16_t  Humidity;
} SAMPLE;

/* Where the next sample will be written */
uint16_t *pLog;

/* The number of samples in the log */
int SampleCount;



/*****************************************************************************/
/* Log_Init                                                                  */
/*****************************************************************************/

uint32_t Log_Init( void )
{
   SAMPLE *pSample = (SAMPLE*)LOG_ADDRESS;

   /* Ensure that VMON is enabled and is a reset source */
   SI32_VMON_B_enable_vbat_supply_monitor( SI32_VMON_0 );
   SI32_RSTSRC_A_enable_vdd_monitor_reset_source( SI32_RSTSRC_0 );

   /* Count the number of samples in the log */
   for ( SampleCount=0; SampleCount<MAX_SAMPLE_COUNT; pSample++,SampleCount++ )
      if ( pSample->Timestamp == 0xFFFFFFFF ) break;

   /* This is where the next sample will be written */
   pLog = (uint16_t*)pSample;

   /* If the log is not empty then return the timestamp from the last sample */
   if ( SampleCount > 0 )
      return Swap32((pSample-1)->Timestamp);   
   else
      return 0;
}


/*****************************************************************************/
/* Log_WriteData                                                             */
/*****************************************************************************/

void Log_WriteData( uint16_t *pData, int DataCount )
{
   int x;
   
   for ( x=0; x<DataCount; x++ )
   {
      /* Write the data to the flash */
      SI32_FLASHCTRL_A_write_wraddr( SI32_FLASHCTRL_0, (uint32_t)pLog++ );
      SI32_FLASHCTRL_A_write_wrdata( SI32_FLASHCTRL_0, *pData++ );
   }
}


/*****************************************************************************/
/* Log_WriteSample                                                           */
/*****************************************************************************/

void Log_WriteSample( uint32_t Timestamp, int Temperature, int Humidity )
{
   uint16_t timestamp[2];
   uint16_t temperature;
   uint16_t humidity;
   
   if ( SampleCount < MAX_SAMPLE_COUNT )
   {
      /* Change the endianess of the sample data */
      timestamp[0] = Swap16( Timestamp>>16 );
      timestamp[1] = Swap16( Timestamp     );
      temperature  = Swap16( Temperature   );
      humidity     = Swap16( Humidity      );

      __disable_irq();

      /* Unlock the flash for multiple writes */
      SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0xA5 );
      SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0xF2 );
      
      /* Write the sample to flash */
      Log_WriteData( timestamp,    2 ); 
      Log_WriteData( &temperature, 1 );
      Log_WriteData( &humidity,    1 );

      /* Relock the flash */
      SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0x5A );

      __enable_irq();
      
      SampleCount++;
   }
}


/*****************************************************************************/
/* Log_Erase                                                                 */
/*****************************************************************************/

void Log_Erase( void )
{
   uint16_t *pPage;

   /* Enter flash erase mode */
   SI32_FLASHCTRL_A_enter_flash_erase_mode( SI32_FLASHCTRL_0 );

   __disable_irq();

   /* Unlock the flash for multiple erases */
   SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0xA5 );
   SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0xF2 );

   /* Erase the written pages of the log */
   for ( pPage=LOG_ADDRESS; pPage<pLog; pPage+=PAGE_SIZE )
   {
      SI32_FLASHCTRL_A_write_wraddr( SI32_FLASHCTRL_0, (uint32_t)pPage );
      SI32_FLASHCTRL_A_write_wrdata( SI32_FLASHCTRL_0, 0 );
   }

   /* Relock the flash */
   SI32_FLASHCTRL_A_write_flash_key( SI32_FLASHCTRL_0, 0x5A );

   __enable_irq();

   /* Exit flash erase mode */
   SI32_FLASHCTRL_A_exit_flash_erase_mode( SI32_FLASHCTRL_0 );

   /* The log is now empty */
   SampleCount = 0;

   /* Restart at the begining of the log */
   pLog = LOG_ADDRESS;
}   


/*****************************************************************************/
/* Swap16                                                                    */
/*****************************************************************************/

uint16_t Swap16( uint16_t HalfWord )
{
   return (HalfWord>>8)|(HalfWord<<8);
}


/*****************************************************************************/
/* Swap32                                                                    */
/*****************************************************************************/

uint32_t Swap32( uint32_t Word )
{
   return ((uint32_t)Swap16(Word>>16))|((uint32_t)(Swap16(Word))<<16);
}


