/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      UART.c
*
* Description:    Send and receive data over the UART
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "Tick.h"
#include "UART.h"


#define TX_BUFF_SIZE  16
#define RX_BUFF_SIZE  16 

/* Transmit and receive buffers */
SEGMENT_VARIABLE( TxBuff[TX_BUFF_SIZE], U8, SEG_PDATA );
SEGMENT_VARIABLE( RxBuff[RX_BUFF_SIZE], U8, SEG_PDATA );

/* Pointers into the transmit and receive buffers */
VARIABLE_SEGMENT_POINTER( pTxIn,  U8, SEG_PDATA );
VARIABLE_SEGMENT_POINTER( pTxOut, U8, SEG_PDATA );
VARIABLE_SEGMENT_POINTER( pRxIn,  U8, SEG_PDATA );
VARIABLE_SEGMENT_POINTER( pRxOut, U8, SEG_PDATA );

/* Status flags */
U8 TxIdle = TRUE;

/* Error counters */
U16 RxLost    = 0;
U16 RxTimeout = 0;

U16 ReceiveTime;


/*****************************************************************************/
/* UART_Init                                                                 */
/*****************************************************************************/

void UART_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Enable UART on the crossbar */
   XBR0 |= 0x01;
   XBR2  = 0x40;

   /* Make sure that the UART pins are not skipped */
   P0SKIP &= 0xCF;
   
   /* Configure UART */
   SCON0 = 0x10;

   /* Timer 1 uses SYSCLK */
//   CKCON |= 0x08;

   /* Configure timer 1 for UART bit rate (57600) */
   TCON |= 0x40;
   TMOD |= 0x20;
// TH1   = 0x52;   /* 57600 */
   TH1   = 0xA9;   /*  9600 */

   /* Initialize the buffer pointers */
   pTxIn  = TxBuff;
   pTxOut = TxBuff;
   pRxIn  = RxBuff;
   pRxOut = RxBuff;
   
   /* Enable UART interrupts */
   ES0 = 1;
}


/*****************************************************************************/
/* IncTx                                                                     */
/*****************************************************************************/

VARIABLE_SEGMENT_POINTER( IncTx( VARIABLE_SEGMENT_POINTER(pTx,U8,SEG_PDATA) ) reentrant, U8, SEG_PDATA )
{
   /* Increment the transmit pointer with wraparound */
   if ( ++pTx == &TxBuff[TX_BUFF_SIZE] )
      pTx = TxBuff;
   return pTx;   
}


/*****************************************************************************/
/* IncRx                                                                     */
/*****************************************************************************/

VARIABLE_SEGMENT_POINTER( IncRx( VARIABLE_SEGMENT_POINTER(pRx,U8,SEG_PDATA) ) reentrant, U8, SEG_PDATA )
{
   /* Increment the receive pointer with wraparound */
   if ( ++pRx == &RxBuff[RX_BUFF_SIZE] )
      pRx = RxBuff;
   return pRx;   
}


/*****************************************************************************/
/* UART_Send                                                                 */
/*****************************************************************************/

U16 UART_Send( U8 *pData, U16 DataSize )
{
   VARIABLE_SEGMENT_POINTER( pTemp, U8, SEG_PDATA );
   U16 DataSent;
   
   /* For each byte in the caller's buffer */
   for ( DataSent=0; DataSent<DataSize; DataSent++ )
   {
      /* Process events while the transmit buffer is full */
      pTemp = IncTx( pTxIn );
      while ( pTemp == pTxOut )
         ProcessEvents();
      
      /* Copy the byte into the transmit buffer and increment pointers */
      *pTxIn = *pData++;
      ES0 = 0;
      pTxIn = pTemp;
      ES0 = 1;

      /* If the UART transmitter is idle */
      if ( TxIdle )
      {
         /* Transmit the first byte and increment the pTxOut pointer */
         ES0 = 0;
         SBUF0  = *pTxOut;
         pTxOut = IncTx( pTxOut );
         TxIdle = FALSE;
         ES0 = 1;
      }
   }  

   /* Return the number of bytes copied from the caller's buffer */
   return DataSent;      
}


/*****************************************************************************/
/* UART_Receive                                                              */
/*****************************************************************************/

U8 UART_Receive( U8 *pData, U8 DataSize, U8 Timeout )
{
   U8 DataReceived;
   
   /* Note when the receive was started */
   ReceiveTime = TickCount();

   /* While the receive buffer is empty */
   while ( pRxIn == pRxOut )
   {
      /* If timeout is enabled and no data was received within the last millisecond */
      if ( Timeout && (ElapsedTime(ReceiveTime)>2) )
      {
         RxTimeout++;
         return 0;
      }

      /* Process events while waiting for a byte to be received */
      ProcessEvents();
   }
   
   /* While the caller's buffer is not full */
   for ( DataReceived=0; DataReceived<DataSize; DataReceived++ )
   {
      /* If there is another byte in the receive buffer */
      if ( pRxOut != pRxIn )
      {
         /* Copy the byte to the caller's buffer and increment pointers */
         *pData++ = *pRxOut;
         ES0 = 0;
         pRxOut = IncRx( pRxOut );
         ES0 = 1;
      }
      else /* The receive buffer is now empty */
      {
         /* Exit the loop */
         break;
      }
   }

   /* Return the number of bytes copied to the caller's buffer */
   return DataReceived;
}


/*****************************************************************************/
/* UART_ISR                                                                  */
/*****************************************************************************/

INTERRUPT( UART_ISR, INTERRUPT_UART0 )
{
   VARIABLE_SEGMENT_POINTER( pTemp, U8, SEG_PDATA );
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;

   /* While there is a transmit or receive interrupt pending */
   while ( SCON0 & 0x03 )
   {
      /* If a byte has been received */
      if ( RI0 )
      {
         /* Reset the receive interrupt */
         RI0 = 0;
         
         /* Copy the byte to the receive buffer */
         *pRxIn = SBUF0;
         
         /* If the receive buffer is not full then increment the pRxIn pointer */
         pTemp = IncRx( pRxIn );
         if ( pTemp != pRxOut )
            pRxIn = pTemp;
         else
            RxLost++;   
      }

      /* If a byte has been transmitted */
      if ( TI0 )
      {
         /* Reset the transmit interrupt */
         TI0 = 0;
         
         /* If the transmit buffer is not empty */
         if ( pTxOut != pTxIn )
         {
            /* Transmit the next byte and increment the pTxOut pointer */
            SBUF0  = *pTxOut;
            pTxOut = IncTx( pTxOut );
         }
         else TxIdle = TRUE;
      }
   }

   SFRPAGE = OldPage;
}

