/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Sensor_Si7005.c
*
* Description:    Read the temperature and humidity from a Si7005 device
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "Tick.h"
#include "I2C.h"
#include "Sensor.h"


/* I2C slave address of Si70xx */
#define SI70XX_ADDR                     0x41

/* Commands */
#define CMD_MEASURE_HUMIDITY_HOLD       0xE5
#define CMD_MEASURE_HUMIDITY_NO_HOLD    0xF5
#define CMD_MEASURE_TEMPERATURE_HOLD    0xE3
#define CMD_MEASURE_TEMPERATURE_NO_HOLD 0xF3
#define CMD_MEASURE_THERMISTOR_HOLD     0xEE
#define CMD_READ_PREVIOUS_TEMPERATURE   0xE0
#define CMD_RESET                       0xFE
#define CMD_WRITE_REGISTER_1            0xE6
#define CMD_READ_REGISTER_1             0xE7
#define CMD_WRITE_REGISTER_2            0x50
#define CMD_READ_REGISTER_2             0x10
#define CMD_WRITE_REGISTER_3            0x51
#define CMD_READ_REGISTER_3             0x11
#define CMD_WRITE_COEFFICIENT           0xC5
#define CMD_READ_COEFFICIENT            0x84

/* User Register 1 */
#define REG1_RESOLUTION_MASK            0x81
#define REG1_RESOLUTION_H12_T14         0x00
#define REG1_RESOLUTION_H08_T12         0x01
#define REG1_RESOLUTION_H10_T13         0x80
#define REG1_RESOLUTION_H11_T11         0x81
#define REG1_LOW_VOLTAGE                0x40
#define REG1_ENABLE_HEATER              0x04

/* User Register 2 */
#define REG2_VOUT                       0x01
#define REG2_VREF_VDD                   0x02
#define REG2_VIN_BUFFERED               0x04
#define REG2_RESERVED                   0x08
#define REG2_FAST_CONVERSION            0x10
#define REG2_MODE_CORRECTION            0x20
#define REG2_MODE_NO_HOLD               0x40

/* Device Identification */
#define ID_SAMPLE                       0xFF
#define ID_SI7013                       0x0D
#define ID_SI7020                       0x14
#define ID_SI7021                       0x15


/*****************************************************************************/
/* Sensor_Init                                                               */
/*****************************************************************************/

void Sensor_Init( void )
{
   /* Nothing to do for the Si70xx */
}


/*****************************************************************************/
/* Sensor_WakeUp                                                             */
/*****************************************************************************/

void Sensor_WakeUp( U8 Bus )
{
   /* The Si70xx automatically wakes up when you give it a command */
   Bus = I2C_BUS_1;
}


/*****************************************************************************/
/* Sensor_Sleep                                                              */
/*****************************************************************************/

void Sensor_Sleep( U8 Bus )
{
   /* The Si70xx automatically sleeps between commands */
   Bus = I2C_BUS_1;
}


/*****************************************************************************/
/* Sensor_Measure                                                            */
/*****************************************************************************/

S8 Sensor_Measure( U8 Bus, U8 Command, U16 *Value )
{
   U8 Data[2];
   S8 Result;

   /* Measure the humidity or temperature value */
   Result = I2C_ReadData( Bus, SI70XX_ADDR, Command, Data, 2 );
   if ( Result != SUCCESS ) return Result;

   /* Swap the bytes and clear the status bits */
   *Value = ((Data[0]*256) + Data[1]) & ~3;
   
   return SUCCESS;
}


/*****************************************************************************/
/* Sensor_ReadTemperature                                                    */
/*                                                                           */
/* Temperature is returned in deci-degrees Celsius.                          */
/* For example: 24.7 degrees Celsius is returned as 247.                     */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadTemperature( U8 Bus, S16 *Temperature )
{
   U16 Value;
   S8  Result;

   /* Measure the temperature */
   Result = Sensor_Measure( Bus, CMD_MEASURE_TEMPERATURE_HOLD, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the temperature to deci-degree Celsius */
   *Temperature = ((((S32)Value)*1757)>>16) - 469;

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ReadHumidity                                                       */
/*                                                                           */
/* Humidity is returned in deci-percent relative humidity.                   */
/* For example: 48.2 percent relative humidity is returned as 482.           */
/*                                                                           */
/*****************************************************************************/

S8 Sensor_ReadHumidity( U8 Bus, S16 Temperature, S16 *Humidity )
{
   U16 Value;
   S32 Decipercent;
   S8  Result;
   
   /* Keep the compiler happy */
   Temperature = 0;

   /* Measure the humidity */
   Result = Sensor_Measure( Bus, CMD_MEASURE_HUMIDITY_HOLD, &Value );
   if ( Result != SUCCESS ) return Result;

   /* Convert the humidity to deci-percent */
   Decipercent = ((((S32)Value)*625)>>15) - 60;

   /* Limit the humidity to valid values */
   if ( Decipercent < 0 )
      *Humidity = 0;
   else if ( Decipercent > 1000 )
      *Humidity = 1000;
   else
      *Humidity = Decipercent;      

   return SUCCESS;   
}


/*****************************************************************************/
/* Sensor_ID                                                                 */
/*****************************************************************************/

U8 Sensor_ID( U8 Bus )
{
   U8 Data[6];
   S8 Result;

   /* Bogus calls to avoid uncalled segments */
   if ( Bus == 10 )
   {
      I2C_ReadByte(  Bus, SI70XX_ADDR, 0, Data );
      I2C_WriteByte( Bus, SI70XX_ADDR, 0, 0    );
   }

   /* Send two command bytes and receive back 6 data bytes */
   Result = I2C_ReadDataWrite2( Bus, SI70XX_ADDR, 0xFC, 0xC9, Data, 6 );
   if ( Result == SUCCESS )
      return Data[0];  /* The data first byte contains the ID */
   else
      return 0;        /* Error */
}

