/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      PortMatch.c
*
* Description:    Handle changes in external signals
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "RTC.h"
#include "PortMatch.h"


/* Timestamps for debouncing */
SEGMENT_VARIABLE( LastTime, U32, SEG_PDATA );
SEGMENT_VARIABLE( ThisTime, U32, SEG_PDATA );


/*****************************************************************************/
/* PortMatch_Init                                                            */
/*****************************************************************************/

void PortMatch_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Generate an interrupt when a button or RTS goes low */
   P0MASK = SW1_EVENT|SW2_EVENT|SW3_EVENT|RTS_EVENT;
   P0MAT  = SW1_EVENT|SW2_EVENT|SW3_EVENT|RTS_EVENT;

   /* Enable the Port Match interrupt */
   EIE2 |= 0x02;
}


/*****************************************************************************/
/* PortMatch_ISR                                                             */
/*****************************************************************************/

INTERRUPT( PortMatch_ISR, INTERRUPT_PORT_MATCH )
{
   U8 OldPage = SFRPAGE;
   U8 Mask;
   U8 Event;
   
   SFRPAGE = LEGACY_PAGE;

   /* Get the value of the RTC timer */
   ThisTime = RTC_ReadTimer();

   /* Debounce by ignoring interrupts that happen within 20 ms */
   if ( RTC_ElapsedTime(&LastTime,&ThisTime) > 660 )
   {
      /* For each button */
      for ( Mask=SW1_EVENT; Mask<SW4_EVENT; Mask<<=1 )
      {
         /* Get the state of the button */
         Event = Mask & P0;
         
         /* If the button has changed state */
         if ( Event != (Mask&P0MAT) )
         {
            /* Change the match for the button */
            P0MAT ^= Mask;
            
            /* If the button is pressed then set the SWx event */
            if ( Event == 0 )
               Events |= Mask;   
         }
      }
   }

   /* Save the RTC timer value */
   LastTime = ThisTime;

   /* Get the state of RTS */
   Event = RTS_EVENT & P0;
   
   /* If RTS has changed */
   if ( Event != (RTS_EVENT&P0MAT) )
   {
      /* Change the match for RTS */
      P0MAT ^= RTS_EVENT;

      /* If RTS is low */
      if ( Event == 0 )
      {
         /* Set the RTS event */
         Events |= RTS_EVENT;
         
         /* Turn on LED4 to show connected to PC */
         LED4 = 0;
      }   
      else /* RTS is high */
      {
         /* Clear the RTS event */
         Events &= ~RTS_EVENT;
         
         /* Turn off LED4 to show disconnected from PC */   
         LED4 = 1;
      }   

      /* The purpose of the RTS event is to block the MCU from sleeping */
   }   
   
   SFRPAGE = OldPage;
}

