/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      I2C.c
*
* Description:    Use the I2C protocol to read and write registers on a device
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "Tick.h"
#include "I2C.h"


/* Maximum time to wait for a transfer to complete */
#define MAX_XFER_TIME   300  /* ms */

/* Maximum number of data bytes that can be read or written */
#define MAX_XFER_LENGTH  8

/* Number of I2C buses */
#define I2C_BUS_COUNT  2

/* Transfer request */
typedef struct
{
   U8 Type;
   U8 Status;
   U8 Bus;
   U8 Address;
   U8 Length;   /* 0,1,2,3,4,5,6,7,8 */
   U8 Data[MAX_XFER_LENGTH];
} XFER;

/* Transfer type */
#define XFER_TYPE_WRITE          0x10
#define XFER_TYPE_READ           0x20
#define XFER_TYPE_WRITE2         0x40
#define XFER_TYPE_WRITE3         0x80
#define XFER_TYPE_WRITE_READ    (XFER_TYPE_WRITE|XFER_TYPE_READ)
#define XFER_TYPE_WRITE2_READ   (XFER_TYPE_WRITE2|XFER_TYPE_READ)
#define XFER_TYPE_WRITE3_READ   (XFER_TYPE_WRITE3|XFER_TYPE_READ)

/* Transfer status */
#define XFER_STATUS_NONE         0x00
#define XFER_STATUS_SUCCESS      0x01
#define XFER_STATUS_ADDR_NAK     0x02
#define XFER_STATUS_DATA_NAK     0x03
#define XFER_STATUS_TIMEOUT      0x04
#define XFER_STATUS_ARBLOST      0x05
#define XFER_STATUS_BAD_LENGTH   0x06
#define XFER_STATUS_BAD_MODE     0x07
#define XFER_STATUS_BAD_STATE    0x08

/* Prototypes */
void I2C_Transfer( void );


/* Global variables */
SEGMENT_VARIABLE( Transfer, XFER, SEG_IDATA );
U8   DataCount;
U8   CurrentBus = I2C_BUS_1;
U16  TransferStartTime;


/*****************************************************************************/
/* I2C_Init                                                                  */
/*****************************************************************************/

void I2C_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Enable SMBus on the crossbar */
   XBR0 |= 0x04;
   XBR2  = 0x40;

   /* Move SMBus to P1.5 and P1.6 for bus 1 */
   P0SKIP = 0xFF;
   P1SKIP = 0x1F;

   /* Configure SMBus */
   SMB0CF = 0xDC;

   /* Timer 0 uses SYSCLK */
   CKCON |= 0x04;

   /* Configure timer 0 for SMBus bit rate (200 kHz) */
   TCON |= 0x10;
   TMOD |= 0x02;
   TH0   = 0xDF;
   
   /* Configure timer 3 for SCL low timeout (25ms) */
   TMR3CN  = 0x04;
   TMR3RLH = 0x5D; 
   TMR3RLL = 0x3D;
   
   /* Enable SMBus and timer 3 interrupts */
   EIE1 |= 0x81;
}
   

/*****************************************************************************/
/* I2C_ReadData                                                              */
/*****************************************************************************/

S8 I2C_ReadData( U8 Bus, U8 Slave, U8 Register, U8 *Data, U8 DataLength )
{
   U8 x;
   
   /* Make sure the bus number is good */
   if ( Bus >= I2C_BUS_COUNT )
      return FAILURE;

   /* Make sure the DataLength is good */
   if ( DataLength > MAX_XFER_LENGTH )
      return FAILURE;
   
   /* Write the register address and read the register data */
   Transfer.Type    = XFER_TYPE_WRITE_READ;
   Transfer.Bus     = Bus;
   Transfer.Address = Slave;
   Transfer.Length  = DataLength;
   Transfer.Data[0] = Register;
   I2C_Transfer();
   if ( Transfer.Status != XFER_STATUS_SUCCESS )
      return FAILURE;

   /* Return the register data */
   for ( x=0; x<DataLength; x++ )
      *Data++ = Transfer.Data[x];   
   
   return SUCCESS;
}


/*****************************************************************************/
/* I2C_ReadDataWrite2                                                        */
/*****************************************************************************/

S8 I2C_ReadDataWrite2( U8 Bus, U8 Slave, U8 Cmd0, U8 Cmd1, U8 *Data, U8 DataLength )
{
   U8 x;

   /* Make sure the bus number is good */
   if ( Bus >= I2C_BUS_COUNT )
      return FAILURE;

   /* Make sure the DataLength is good */
   if ( DataLength > MAX_XFER_LENGTH )
      return FAILURE;

   /* Write the register address and read the register data */
   Transfer.Type    = XFER_TYPE_WRITE2_READ;
   Transfer.Bus     = Bus;
   Transfer.Address = Slave;
   Transfer.Length  = DataLength;
   Transfer.Data[0] = Cmd0;
   Transfer.Data[1] = Cmd1;
   I2C_Transfer();
   if ( Transfer.Status != XFER_STATUS_SUCCESS )
      return FAILURE;

   /* Return the register data */
   for ( x=0; x<DataLength; x++ )
      *Data++ = Transfer.Data[x];   
   
   return SUCCESS;
}


/*****************************************************************************/
/* I2C_WriteData                                                             */
/*****************************************************************************/

S8 I2C_WriteData( U8 Bus, U8 Slave, U8 Register, U8 *Data, U8 DataLength )
{
   U8 x;

   /* Make sure the bus number is good */
   if ( Bus >= I2C_BUS_COUNT )
      return FAILURE;

   /* Make sure the DataLength is good */
   if ( DataLength >= MAX_XFER_LENGTH )
      return FAILURE;

   /* Write data to the specified register */
   Transfer.Type    = XFER_TYPE_WRITE;
   Transfer.Bus     = Bus;
   Transfer.Address = Slave;
   Transfer.Length  = DataLength+1;
   Transfer.Data[0] = Register;
   for ( x=1; x<=DataLength; x++ )
      Transfer.Data[x] = *Data++;
   I2C_Transfer();
   if ( Transfer.Status != XFER_STATUS_SUCCESS )
      return FAILURE;

   return SUCCESS;
}


/*****************************************************************************/
/* I2C_ReadByte                                                              */
/*****************************************************************************/

S8 I2C_ReadByte( U8 Bus, U8 Slave, U8 Register, U8 *Byte )
{
   return I2C_ReadData( Bus, Slave, Register, Byte, 1 );
}


/*****************************************************************************/
/* I2C_WriteByte                                                             */
/*****************************************************************************/

S8 I2C_WriteByte( U8 Bus, U8 Slave, U8 Register, U8 Byte )
{
   return I2C_WriteData( Bus, Slave, Register, &Byte, 1 );
}


/*****************************************************************************/
/* I2C_Reset                                                                 */
/*****************************************************************************/

void I2C_Reset( void )
{
   /* Disable I2C */
   SMB0CF &= 0x7F;
   
   /* Enable I2C */
   SMB0CF |= 0x80;
}


/*****************************************************************************/
/* I2C_Transfer                                                              */
/*****************************************************************************/

void I2C_Transfer( void )
{
   /* Move SMBus to the specified bus */
   if ( Transfer.Bus != CurrentBus )
   {
      if ( Transfer.Bus == I2C_BUS_1 )
         P1SKIP = 0x1F;         /* Bus 1: P1.5 and P1.6 */
      else   
         P1SKIP = 0x00;         /* Bus 2: P1.0 and P1.1 */
   
      CurrentBus = Transfer.Bus;
   }

   /* The transfer has not started yet */
   Transfer.Status = XFER_STATUS_NONE;

   /* No data has been transfered yet */   
   DataCount = 0;

   /* Start the transfer */
   STA = 1;

   /* Note when the transfer started */   
   TransferStartTime = TickCount();
   
   /* Block until the transfer is finished or timed out */
   while ( Transfer.Status == XFER_STATUS_NONE )
   {
      if ( ElapsedTime(TransferStartTime) > MAX_XFER_TIME )
      {
         /* Reset the I2C block */
         I2C_Reset();

         /* Transfer timed out */
         Transfer.Status = XFER_STATUS_TIMEOUT;
         break;
      }
   }
}


/*****************************************************************************/
/* I2C_ISR                                                                   */
/*****************************************************************************/

INTERRUPT( I2C_ISR, INTERRUPT_SMBUS0 )
{
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;

   if ( ARBLOST )
   {
      /* Arbitration was lost */
      Transfer.Status = XFER_STATUS_ARBLOST;
   }
   else if ( MASTER )
   {
      /* If a start was generated */
      if ( STA )
      {
         /* Clear the start flag */
         STA = 0;

         /* Transmit the address and read/write bit */            
         if ( Transfer.Type & (XFER_TYPE_WRITE | XFER_TYPE_WRITE2| XFER_TYPE_WRITE3 ) )
            SMB0DAT = Transfer.Address<<1;
         else
            SMB0DAT = (Transfer.Address<<1) + 1;
      }
      else if ( TXMODE ) /* An address or data byte was transmitted */
      {
         /* If the address or data byte was acknowledged */
         if ( ACK )
         {
            switch ( Transfer.Type )
            {
               case XFER_TYPE_WRITE:
                  /* If there is more data to transmit */
                  if ( DataCount < Transfer.Length )
                  {
                     /* Transmit the next data byte */
                     SMB0DAT = Transfer.Data[DataCount++];
                  }   
                  else /* The write transfer is finished */
                  {
                     /* Generate a stop */
                     STO = 1;

                     /* Write transfer is successful */
                     Transfer.Status = XFER_STATUS_SUCCESS;
                  }
                  break;
               case XFER_TYPE_READ:
                  /* If reading zero bytes */
                  if ( Transfer.Length == 0 )
                  {
                     /* Generate a stop */
                     STO = 1;
                     
                     /* Read transfer of zero bytes is successful */
                     Transfer.Status = XFER_STATUS_SUCCESS;
                  }
                  /* else receive a byte */
                  break;
               case XFER_TYPE_WRITE_READ:
                  /* If the data byte has not been transmitted */
                  if ( DataCount == 0 )
                  {
                     /* Transmit the data byte */
                     SMB0DAT = Transfer.Data[DataCount++];
                  }   
                  else /* The write transfer is finished */
                  {
                     /* Start the read transfer */
                     Transfer.Type = XFER_TYPE_READ;
                     
                     /* Reset the data counter */
                     DataCount = 0;
                     
                     /* Generate a restart */
                     STA = 1;
                  }
                  break;      
               case XFER_TYPE_WRITE2_READ:
                  /* If the data has not been transmitted */
                  if ( DataCount < 2 )
                  {
                     /* Transmit the data byte */
                     SMB0DAT = Transfer.Data[DataCount++];
                  }   
                  else /* The write transfer is finished */
                  {
                     /* Start the read transfer */
                     Transfer.Type = XFER_TYPE_READ;
                     
                     /* Reset the data counter */
                     DataCount = 0;
                     
                     /* Generate a restart */
                     STA = 1;
                  }
                  break;  
               case XFER_TYPE_WRITE3_READ:
                  /* If the data has not been transmitted */
                  if ( DataCount < 3 )
                  {
                     /* Transmit the data byte */
                     SMB0DAT = Transfer.Data[DataCount++];
                  }   
                  else /* The write transfer is finished */
                  {
                     /* Start the read transfer */
                     Transfer.Type = XFER_TYPE_READ;
                     
                     /* Reset the data counter */
                     DataCount = 0;
                     
                     /* Generate a restart */
                     STA = 1;
                  }
                  break;    
            }
         }
         else /* The address or data byte was not acknowledged */
         {
            /* If no data bytes were transferred */
            if ( DataCount == 0 )
            {
               /* If the address of a read was not acknowledged */
               if ( Transfer.Type == XFER_TYPE_READ )
               {
                  /* Try again */
                  STA = 1;
               }
               else /* The address of a write was not acknowledged */
               {
                  Transfer.Status = XFER_STATUS_ADDR_NAK;

                  /* Abort the transfer */
                  STO = 1;
               }
            }   
            else /* A data byte was not acknowledged */
            {
               Transfer.Status = XFER_STATUS_DATA_NAK;

               /* Abort the transfer */
               STO = 1;
            }   
         }
      }
      else if ( ACKRQ ) /* A data byte was received */
      {
         /* Get the received data byte */
         Transfer.Data[DataCount++] = SMB0DAT;

         /* If more data is needed */
         if ( DataCount < Transfer.Length )
         {
            /* Tell the slave to send another byte */
            ACK = 1;
         }
         else /* The read transfer is finished */
         {
            /* Tell the slave that we are done */
            ACK = 0;

            /* Generate a stop */
            STO = 1;

            /* Read transfer is successful */
            Transfer.Status = XFER_STATUS_SUCCESS;
         }
      }
      else /* Invalid state */
      {
         /* Abort the transfer */
         STO = 1;
         Transfer.Status = XFER_STATUS_BAD_STATE;
      }
   }
   else /* In slave mode */
   {
      /* Reset the I2C block */
      I2C_Reset();
      Transfer.Status = XFER_STATUS_BAD_MODE;
   } 

   /* Clear the I2C interrupt */
   SI = 0;

   SFRPAGE = OldPage;
}


/*****************************************************************************/
/* SCL_Low_Timeout_ISR                                                       */
/*****************************************************************************/

INTERRUPT( SCL_Low_Timeout_ISR, INTERRUPT_TIMER3 )
{
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;

   /* Clear the timer 3 interrupt */
   TMR3CN &= 0x7F;

   /* Reset the I2C block */
   I2C_Reset();

   SFRPAGE = OldPage;
}

