// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------

/*!
 * \file
 * \brief ON Semiconductor motor driver control library header file
 *        for Arduino micro.
 *
 *        Available motor driver is LV8702V.
 */

#ifndef TEST_MOTOR_CONTROL_LIB_H
#define TEST_MOTOR_CONTROL_LIB_H

#include <Arduino.h>

#ifndef ArduinoMock_h
#include <SPI.h>
#endif

/*!
 *  \class motorControlLibraryBase
 *  \brief Motor Control Livrary base class.
 *
 *         This class is defined common members, functions and etc.
 */
class motorControlLibraryBase
{
  public:
    static const int SUCCESS  = 0;   //!< success value define
    static const int FAILURE  = 1;   //!< failure value define
    static const int RESTRICT = 2;   //!< prohibition operate define
    static const uint8_t RX   = 0;   //!< Arduino micro RX pin number define
    static const uint8_t TX   = 1;   //!< Arduino micro TX pin number define
    static const uint8_t D2   = 2;   //!< Arduino micro Degital pin number define
    static const uint8_t D3   = 3;   //!< Arduino micro Degital pin number define
    static const uint8_t D4   = 4;   //!< Arduino micro Degital pin number define
    static const uint8_t D5   = 5;   //!< Arduino micro Degital pin number define
    static const uint8_t D6   = 6;   //!< Arduino micro Degital pin number define
    static const uint8_t D7   = 7;   //!< Arduino micro Degital pin number define
    static const uint8_t IO8  = 8;   //!< Arduino micro I/O pin number define
    static const uint8_t IO9  = 9;   //!< Arduino micro I/O pin number define
    static const uint8_t IO10 = 10;  //!< Arduino micro I/O pin number define
    static const uint8_t IO11 = 11;  //!< Arduino micro I/O pin number define
    static const uint8_t IO12 = 12;  //!< Arduino micro I/O pin number define
    static const uint8_t IO13 = 13;  //!< Arduino micro I/O pin number define
    static const uint8_t MISO = 14;  //!< Arduino micro MISO pin number define
    static const uint8_t SCK  = 15;  //!< Arduino micro SCK  pin number define
    static const uint8_t MOSI = 16;  //!< Arduino micro MOSI pin number define
    static const uint8_t SS   = 17;  //!< Arduino micro SS   pin number define
    static const uint8_t A0   = 18;  //!< Arduino micro Analog pin number define
    static const uint8_t A1   = 19;  //!< Arduino micro Analog pin number define
    static const uint8_t A2   = 20;  //!< Arduino micro Analog pin number define
    static const uint8_t A3   = 21;  //!< Arduino micro Analog pin number define
    static const uint8_t A4   = 22;  //!< Arduino micro Analog pin number define
    static const uint8_t A5   = 23;  //!< Arduino micro Analog pin number define
    // byte dutyRateToValue(int rate);  //!< convert duty rate to duty value
};

/*!
 *  \class GuiSerialInterface
 *  \brief Parse serial message from debug GUI.
 *
 *         This class is defined common members, functions and etc.
 */
class GuiSerialInterface
{
  public:
    static const uint16_t DEFAULT_BAUDRATE = 19200; //!< Default serial baud rate

    void guiSerialRead();
    virtual int guiSerialParse(char *serialRecvStr) = 0; // abstract
    virtual void timeoutPol();
    virtual ~GuiSerialInterface()
    {
      // Don't remove!
    }

  protected:
    void _serialReadArduino(char *serialRecvStr, const int bufferSize);

  private:
    static const uint8_t _SMES_BUFFER_SIZE = 64; //!< Data buffler size from serial recieved
    static const uint32_t POLCOUNTMAX   = 3000;
    uint32_t polCounter;
};

/*!
 *  \class Lib_LV8702V
 *  \brief Motor Control Livrary class for LV8702V driver.
 *
 *         This class is defined LV8702V driver control APIs.
 */
class Lib_LV8702V: public motorControlLibraryBase, public GuiSerialInterface
{
  public:
    virtual ~Lib_LV8702V() {} // Don't remove!
    int  initLib();
    int  setChipEnable(byte select);
	int  setReset(byte select);
	int  setMaxCurrent(float adpVoltage, float adpCurrent, float mtrCurrent, float mtrResistance);
    int  setRefVoltage(float vref);
    int  setStepAngle(float angle);
    int  motorRotationDeg(float freq, float deg, byte direction, byte excitation);
    int  motorRotationTime(float freq, float time, byte direction, byte excitation);
    int  motorRotationStep(float freq, float step, byte direction, byte excitation);
    void motorRotationStop();
    void motorRotationFree();
    int  setEfficiency(byte efficiency, byte driveMargin, byte boostup);
    int  readAdc();
	uint32_t  readDriveStatus();
	void clrDstCount();
    virtual int guiSerialParse(char* type);
    float timerFire();

  protected:
    bool              _isRotation     = false;
    float             _currentMax     = 2.72F;
    float             _stepAngle      = STEP_ANGLE_MIN;
	byte              _reset          = RESET_OFF;
	byte              _direction      = DIRECTION_CW;
	byte              _excitation     = EXCITATION_FULL;
    float             _stepFreq       = 0.0F;
    float             _nowFreq        = 0.0F;
    byte              _phaseFlag      = 0;
    uint32_t          _targetStep     = 0;
    uint32_t          _nowStep        = 0;
	volatile uint8_t  _dstVal1        = HIGH;
	volatile uint8_t  _dstVal2        = HIGH;
	volatile uint32_t _dstCnt1        = DST_COUNT_MIN;
	volatile uint32_t _dstCnt2        = DST_COUNT_MIN;
	volatile uint8_t  _stepCnt        = STEP_COUNT_INITIAL;
	volatile uint8_t  _stepCntLimit   = STEP_COUNT_LIMIT_FULL;
	volatile uint8_t  _stepCntErr     = STEP_COUNT_NON_ERROR;
	volatile uint8_t  _stepCntErrMask = STEP_COUNT_ERROR_MASK;
	bool              _fallingEdge    = true;
    void timeoutPol();

    int   _setOutputEnable(byte select);
    int   _setDirection(byte select);
    int   _setExcitation(byte select);

  private:
    static const uint8_t   CHIP_ENABLE_OFF          = 0;
    static const uint8_t   CHIP_ENABLE_ON           = 1;
    static const uint8_t   OUTPUT_DISABLE           = 0;
    static const uint8_t   OUTPUT_ENABLE            = 1;
    static const uint8_t   RESET_OFF                = 0;
    static const uint8_t   RESET_ON                 = 1;
    static constexpr float ADAPTER_VOLTAGE_MIN      = 9.0F;
    static constexpr float ADAPTER_VOLTAGE_MAX      = 32.0F;
    static constexpr float ADAPTER_CURRET_MIN       = 0.0F;
    static constexpr float ADAPTER_CURRET_MAX       = 10.0F;
    static constexpr float MOTOR_CURRET_MIN         = 0.1F;
    static constexpr float MOTOR_CURRET_MAX         = 2.5F;
    static constexpr float MOTOR_RESISTANCE_MIN     = 0.1F;
    static constexpr float MOTOR_RESISTANCE_MAX     = 500.0F;
    static constexpr float VREF_MIN                 = 0.0F;
    static constexpr float VREF_MAX                 = 3.00F;
    static constexpr float REFERENCE_VREF           = 0.33F;
    static constexpr float STEP_ANGLE_MIN           = 0.01F;
    static constexpr float STEP_ANGLE_MAX           = 360.00F;
    static constexpr float FREQ_MIN                 = 1.0F;
    static constexpr float FREQ_MAX                 = 4800.0F;
    static constexpr float ANGLE_MIN                = 0.00F;
    static constexpr float ANGLE_MAX                = 16777215.00F;
    static constexpr float TIME_MIN                 = 0;
    static constexpr float TIME_MAX                 = 65535;
    static constexpr float STEP_MIN                 = 0;
    static constexpr float STEP_MAX                 = 16777215;
    static const uint8_t   DIRECTION_CW             = 0;
    static const uint8_t   DIRECTION_CCW            = 1;
    static const uint8_t   EXCITATION_FULL          = 0;
    static const uint8_t   EXCITATION_HALF100       = 1;
    static const uint8_t   EXCITATION_HALF70        = 2;
    static const uint8_t   EXCITATION_QUARTER       = 3;
    static const uint8_t   EFFICIENCY_NORMAL        = 0;
    static const uint8_t   EFFICIENCY_HIGH          = 1;
    static const uint8_t   DRIVE_MARGIN_S           = 0;
    static const uint8_t   DRIVE_MARGIN_M           = 1;
    static const uint8_t   DRIVE_MARGIN_L           = 2;
    static const uint8_t   BOOST_UP_MIN             = 0;
    static const uint8_t   BOOST_UP_LOW             = 1;
    static const uint8_t   BOOST_UP_HIGH            = 2;
    static const uint8_t   BOOST_UP_MAX             = 3;
    static const uint8_t   DST_COUNT_MIN            = 0;
    static const uint8_t   DST_COUNT_MAX            = 255;
    static const uint8_t   MONI_INITIAL             = 0;
    static const uint8_t   MONI_NON_INITIAL         = 1;
    static const uint8_t   STEP_COUNT_INITIAL       = 0;
    static const uint8_t   STEP_COUNT_LIMIT_FULL    = 4;
    static const uint8_t   STEP_COUNT_LIMIT_HALF    = 8;
    static const uint8_t   STEP_COUNT_LIMIT_QUARTER = 16;
    static const uint8_t   STEP_COUNT_NON_ERROR     = 0;
    static const uint8_t   STEP_COUNT_ERROR         = 1;
    static const uint8_t   STEP_COUNT_ERROR_UNMASK  = 0;
    static const uint8_t   STEP_COUNT_ERROR_MASK    = 2;

    // Serial message identiffer types
    static const byte SRMES_GET_ID                  = 0x03;
    static const byte SRMES_POLLING_ID              = 0x04;
    static const byte SRMES_SET_CHIP_ENABLE         = 0x41;
    static const byte SRMES_SET_RESET               = 0x43;
    static const byte SRMES_SET_MAX_CURRENT         = 0x44;
    static const byte SRMES_SET_REF_VOLTAGE         = 0x45;
    static const byte SRMES_STEP_ANGLE              = 0x46;
    static const byte SRMES_ROTATION_ANGLE          = 0x51;
    static const byte SRMES_ROTATION_TIME           = 0x52;
    static const byte SRMES_ROTATION_STEP           = 0x53;
    static const byte SRMES_ROTATION_STOP           = 0x54;
    static const byte SRMES_ROTATION_FREE           = 0x55;
    static const byte SRMES_SET_EFFICIENCY          = 0x61;
    static const byte SRMES_READ_ADC                = 0x64;
    static const byte SRMES_READ_DRIVESTATUS        = 0x65;
    static const byte SRMES_CLR_DST_COUNT           = 0x68;
    // Identiffer types of serial message response
    static const byte SRMES_RES_MAX_CURRENT         = 0x31;
    static const byte SRMES_RES_READ_ADC            = 0x32;
    static const byte SRMES_RES_READ_DRIVESTATUS    = 0x33;

    // Segment definition of serial message
    typedef struct
    {
      byte identiffer;
      byte select;
    } SrMesDivSetChipEnable;
    typedef struct
    {
      byte identiffer;
      byte select;
    } SrMesDivSetReset;
    typedef struct
    {
      byte identiffer;
      byte adpVoltage[2];
      byte adpCurrent[2];
      byte mtrCurrent[2];
      byte mtrResistance[2];
    } SrMesDivSetMaxCurrent;
    typedef struct
    {
      byte identiffer;
      byte vref[2];
    } SrMesDivSetRefVoltage;
    typedef struct
    {
      byte identiffer;
      byte angle[2];
    } SrMesDivSetStepAngle;
    typedef struct
    {
      byte identiffer;
      byte freq[2];
      byte angle[4];
      byte direction;
      byte excitation;
    } SrMesDivRotationDeg;
    typedef struct
    {
      byte identiffer;
      byte freq[2];
      byte time[2];
      byte direction;
      byte excitation;
    } SrMesDivRotationTime;
    typedef struct
    {
      byte identiffer;
      byte freq[2];
      byte step[4];
      byte direction;
      byte excitation;
    } SrMesDivRotationStep;
    typedef struct
    {
      byte identiffer;
      byte efficiency;
      byte driveMargin;
      byte boostup;
    } SrMesDivSetEfficiency;
    typedef struct
    {
      byte identiffer;
      byte recvCurrent[2];
    } SrMesDivRecvMaxCurrent;
    typedef struct
    {
      byte identiffer;
      byte recvAdc[2];
    } SrMesDivRecvReadAdc;
    typedef struct
    {
      byte identiffer;
      byte recvDriveStatus[4];
    } SrMesDivRecvReadDriveStatus;
};

#endif
