// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------

#include "LV8702_Lib.h"
#include <stdio.h>

#define ID "LV8702_Ver.1.0\n"

//Development Version: 2.5.0

// --------------------------------------------------------------------
// GuiSerialInterface common class
// --------------------------------------------------------------------
/*!
 *  \fn     GuiSerialInterface::guiSerialRead()
 *  \brief  Read serial messages and execeute that values.
 *
 *          If serial message recieved then parse and execute.
 *          (That minnimum onecharacter or more characters.)
 *          Will be calling abstruct method (guiSerialParse) of motor class.
 *
 *  \param  None.
 *  \return void
 */
void GuiSerialInterface::guiSerialRead()
{
  char *serialRecvStr = new char[_SMES_BUFFER_SIZE] {};
  _serialReadArduino(serialRecvStr, _SMES_BUFFER_SIZE);

  if (serialRecvStr[0] != '\0')
  {
    polCounter = millis();
    // Call to abstruct method.
    guiSerialParse(serialRecvStr);
  }
  else{
    if(millis() - polCounter > POLCOUNTMAX){
      timeoutPol();
      polCounter = millis();
    }
  }

  delete[] serialRecvStr;
}

/*!
 *  \fn     GuiSerialInterface::_serialReadArduino()
 *  \brief  Read serial message into the buffer from computer.
 *
 *  \param  *serialRecvStr   Buffer for received message.
 *  \param  bufferSize       Read buffer size.(max 63)
 *  \return void
 */
void GuiSerialInterface::_serialReadArduino(char *serialRecvStr, const int bufferSize)
{
  int i = 0;

  while (Serial.available())
  {
    char receptChar = Serial.read();
    serialRecvStr[i++] = receptChar;

    // Interrupted if buffer was overflowed.
    if (i >= (bufferSize - 1))
    {
      serialRecvStr[i] = '\0';
      return;
    }
  }
}

// ================================
// Lib_LV8702V
// ================================
/*!
 *  \fn     Lib_LV8702V::initLib()
 *  \brief  Set to initial parameters of the Lib_LV8702V motor driver.
 *
 *          Below set the parameters.
 *           - setChipEnable(0);
 *           - setReset(0);
 *           - setRefVoltage(0.33);
 *           - setStepAngle(0.01)
 *           - setEfficiency(0, 0, 0);
 *           - _setOutputEnable(1);
 *           - _setDirection(0);
 *           - _setExcitation(0);
 *
 *  \param  None.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::initLib()
{
  /* TIMER4
   *Set TCCR4B Cs 3,2,1,0 -> 0001
   * for D6,D13  PWMF 31.373[kHz]
   */
  TCCR4B &= B11110000;
  TCCR4B |= B00000001;

  // Initial setting of each pin.
  digitalWrite(D2,LOW);
  digitalWrite(D3,LOW);
  digitalWrite(D4,LOW);
  digitalWrite(D5,LOW);
  digitalWrite(D6,LOW);
  digitalWrite(D7,LOW);
  digitalWrite(IO8,LOW);
  digitalWrite(IO9,LOW);
  digitalWrite(IO10,LOW);
  digitalWrite(IO11,LOW);
  digitalWrite(IO12,LOW);
  digitalWrite(IO13,LOW);
  digitalWrite(MOSI,LOW);

  // set pin(D2, D3) set drive margin
  pinMode(D2, OUTPUT);
  pinMode(D3, OUTPUT);

  // set pin(D4) set high efficiency mode
  pinMode(D4, OUTPUT);

  // set pin(D5) set direction
  pinMode(D5, OUTPUT);

  // set pin(D6) reference voltage 0 V (duty 0)
  pinMode(D6, OUTPUT);

  // set pin(D7) reset
  pinMode(D7, OUTPUT);

  // set pin(IO8, IO9) set excitation
  pinMode(IO8, OUTPUT);
  pinMode(IO9, OUTPUT);

  // set pin(IO10) set output enable
  pinMode(IO10, OUTPUT);

  // set pin(IO11) clock pulse
  pinMode(IO11, OUTPUT);

  // set pin(IO12, MOSI) boost up Settings
  pinMode(IO12, OUTPUT);
  pinMode(MOSI, OUTPUT);

  // set pin(IO13) mode output and set POWER_OFF
  pinMode(IO13, OUTPUT);

  // Initial setting of each function.
  int result = SUCCESS;
  result |= setChipEnable(CHIP_ENABLE_OFF);
  result |= setReset(RESET_OFF);
  result |= setRefVoltage(REFERENCE_VREF);
  result |= setStepAngle(STEP_ANGLE_MIN);
  result |= setEfficiency(EFFICIENCY_NORMAL, DRIVE_MARGIN_S, BOOST_UP_MIN);
  result |= _setOutputEnable(OUTPUT_ENABLE);
  result |= _setDirection(DIRECTION_CW);
  result |= _setExcitation(EXCITATION_FULL);
  
  if(SUCCESS == result)
  {
    return SUCCESS;
  }
  else
  {
    // Impossible pattern.
    return FAILURE;
  }
}

/*!
 *  \fn     Lib_LV8702V::setChipEnable(byte select)
 *  \brief  Set enable of driver chip.
 *
 *          If set to 0 is output at LOW from Arduino IO13 pin.
 *          If set to 1 is output at HIGH from Arduino IO13 pin.
 *          That is input into ST pin of driver chip.
 *          If that input is HIGH, driver chip is enabled.
 *
 *  \param  select           0x00(0) is Power off.
 *                           0x01(1) is Power on.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setChipEnable(byte select)
{
  if (select == CHIP_ENABLE_OFF)
  {
    digitalWrite(IO13, LOW);
    _isRotation = false;
    _phaseFlag = 0;
    _nowStep = 0;
  	_stepCnt = STEP_COUNT_INITIAL;
  }
  else if (select == CHIP_ENABLE_ON)
  {
    digitalWrite(IO13, HIGH);
  }
  else
  {
    return FAILURE;
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::setReset(byte select)
 *  \brief   Set of reset parameters.
 *
 *          If set to 0 is output at LOW from Arduino D7 pin.
 *          If set to 1 is output at HIGH from Arduino D7 pin.
 *          That is input into RST pin of driver chip.
 *          If that input is HIGH, The excitation position of the output 
 *          is forcibly set to the initial position, and the MONI output is turned on.
 *
 *  \param  reset            Set of reset mode.
 *                           0x00(0) is normal.
 *                           0x01(1) is reset enable.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setReset(byte select)
{
  // Check parameters.
  if (RESET_OFF > select || RESET_ON < select)
  {
    return FAILURE;
  }

  // Set parameters.
  if(RESET_OFF == select)
  {
  	_reset = RESET_OFF;
    digitalWrite(D7, LOW);
  }
  else
  {
  	_reset= RESET_ON;
  	_stepCnt = STEP_COUNT_INITIAL;
    digitalWrite(D7, HIGH);
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::setMaxCurrent(float adpVoltage, float adpCurrent, float mtrCurrent, float mtrResistance)
 *  \brief  Set the output motor current limit value.
 *
 *  \param  adpVoltage    Set the supply voltage. Range is 9 to 32.
 *  \param  adpCurrent    Set the max supply current. Range is 0 to 10.
 *  \param  mtrCurrent    Set the motor rated current. Range is 0.1 to 2.5.
 *  \param  mtrResistance Set the motor winding resistance. Range is 0.1 to 500.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setMaxCurrent(float adpVoltage, float adpCurrent, float mtrCurrent, float mtrResistance)
{
  // check parameters.
  if ((adpVoltage < ADAPTER_VOLTAGE_MIN) || (adpVoltage > ADAPTER_VOLTAGE_MAX))
  {
    return FAILURE;
  }
  if ((adpCurrent < ADAPTER_CURRET_MIN) || (adpCurrent > ADAPTER_CURRET_MAX))
  {
    return FAILURE;
  }
  if ((mtrCurrent < MOTOR_CURRET_MIN) || (mtrCurrent > MOTOR_CURRET_MAX))
  {
    return FAILURE;
  }
  if ((mtrResistance < MOTOR_RESISTANCE_MIN) || (mtrResistance > MOTOR_RESISTANCE_MAX))
  {
    return FAILURE;
  }

  // set parameters.
  adpCurrent /= 2;
  if (mtrCurrent > adpCurrent)
  {
  	if (adpCurrent > (adpVoltage/mtrResistance))
  	{
  	  _currentMax = adpVoltage/mtrResistance;
  	}
  	else
  	{
  	  _currentMax = adpCurrent;
  	}
  }
  else
  {
  	if (mtrCurrent > (adpVoltage/mtrResistance))
  	{
  	  _currentMax = adpVoltage/mtrResistance;
  	}
  	else
  	{
  	  _currentMax = mtrCurrent;
  	}
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::setRefVoltage(float vref)
 *  \brief  Set the control voltage(CTL) of output ratio(Duty).
 *
 *          Designation of analog output voltage(PWM/31.373[kHz]) from Arduino D6 pin.
 *          That pulse is input to VREF pin of driver chip.
 *
 *  \param  vref             Vref range is 0x00(0) to 0x03(3).
 *                           0x00(0) is mean to set duty 0, and 0x03(3) is mean to set duty 153.
 *  \return 0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setRefVoltage(float vref)
{
  if ((vref < VREF_MIN) || (vref > VREF_MAX))
  {
   return FAILURE;
  }
  byte dutyVal = vref / 5 * 255;
  analogWrite(D6, dutyVal);
  return SUCCESS;
}

/*
 *  \fn     Lib_LV8702V::setStepAngle(float angle)
 *  \brief  Set the step angle.
 *
 *  \param  angle           angle range is 0.01 to 360
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setStepAngle(float angle)
{
  if ((angle < STEP_ANGLE_MIN) || (angle > STEP_ANGLE_MAX))
  {
      return FAILURE;
  }
  _stepAngle = angle;
  return SUCCESS;
}

/*!
 *  \fn      Lib_LV8702V::motorRotationDeg(float freq, float deg, byte direction, byte excitation)
 *  \brief   Motor rotation with specified angle.
 *
 *  \param  freq             Specify driving frequency.
 *                           0x01(1)-0x12C0(4800)
 *
 *  \param  deg              Specify rotation angle.
 *                           0x00(infinity)
 *                           0x01(1)-0xFFFFFF(16777215)
 *
 *  \param  direction        Specify rotation direction.
 *                           0x00(0) Clock Wise.
 *                           0x01(1) Counter Clock Wise.
 *
 *  \param  excitation       Set of excitation.
 *                           0x00(0) is full step.
 *                           0x01(1) is half step.(supply torque is 100%)
 *                           0x02(2) is half step.(supply torque is 70%)
 *                           0x03(3) is 1/4 step.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::motorRotationDeg(float freq, float deg, byte direction, byte excitation)
{
  byte result = SUCCESS;
  // check parameters.
  if (FREQ_MIN > freq || FREQ_MAX < freq)
  {
    return FAILURE;
  }
  if (ANGLE_MIN > deg || ANGLE_MAX < deg)
  {
    return FAILURE;
  }
  if (DIRECTION_CW > direction || DIRECTION_CCW < direction)
  {
    return FAILURE;
  }
  if (EXCITATION_FULL > excitation || EXCITATION_QUARTER < excitation)
  {
    return FAILURE;
  }

  _setOutputEnable(OUTPUT_ENABLE);
  _nowFreq = freq;
  _stepFreq = _nowFreq * 2;
  _nowStep = 0;
  if (EXCITATION_FULL == excitation)
  {
    _targetStep = uint32_t(deg / _stepAngle);
  }
  else if (EXCITATION_QUARTER == excitation)
  {
    _targetStep = uint32_t(deg / _stepAngle * 4);
  }
  else
  {
    _targetStep = uint32_t(deg / _stepAngle * 2);
  }
  result |= _setDirection(direction);
  result |= _setExcitation(excitation);
  if (!_isRotation) _fallingEdge = false;
  _isRotation = true;
  return result;
}

/*!
 *  \fn      Lib_LV8702V::motorRotationTime(float freq, float time, byte direction, byte excitation)
 *  \brief   Motor rotation with specified time.
 *
 *  \param  freq             Specify driving frequency.
 *                           0x01(1)-0x12C0(4800)
 *
 *  \param  time             Specify rotation time.
 *                           0x00(infinity)
 *                           0x01(1)-0xFFFF(65535)
 *
 *  \param  direction        Specify rotation direction.
 *                           0x00(0) Clock Wise.
 *                           0x01(1) Counter Clock Wise.
 *
 *  \param  excitation       Set of excitation.
 *                           0x00(0) is full step.
 *                           0x01(1) is half step.(supply torque is 100%)
 *                           0x02(2) is half step.(supply torque is 70%)
 *                           0x03(3) is 1/4 step.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::motorRotationTime(float freq, float time, byte direction, byte excitation)
{
  byte result = SUCCESS;
  // check parameters.
  if (FREQ_MIN > freq || FREQ_MAX < freq)
  {
    return FAILURE;
  }
  if (TIME_MIN > time || TIME_MAX < time)
  {
    return FAILURE;
  }
  if (DIRECTION_CW > direction || DIRECTION_CCW < direction)
  {
    return FAILURE;
  }
  if (EXCITATION_FULL > excitation || EXCITATION_QUARTER < excitation)
  {
    return FAILURE;
  }

  _setOutputEnable(OUTPUT_ENABLE);
  _nowFreq = freq;
  _stepFreq = _nowFreq * 2;
  _nowStep = 0;
  _targetStep = uint32_t(time * _nowFreq);
  result |= _setDirection(direction);
  result |= _setExcitation(excitation);
  if (!_isRotation) _fallingEdge = false;
  _isRotation = true;
  return result;
}

/*!
 *  \fn      Lib_LV8702V::motorRotationStep(float freq, float step, byte direction, byte excitation)
 *  \brief   Motor rotation with specified step.
 *
 *  \param  freq             Specify driving frequency.
 *                           0x01(1)-0x12C0(4800)
 *
 *  \param  step             Specify rotation step.
 *                           0x00(infinity)
 *                           0x01(1)-0xFFFFFF(16777215)
 *
 *  \param  direction        Specify rotation direction.
 *                           0x00(0) Clock Wise.
 *                           0x01(1) Counter Clock Wise.
 *
 *  \param  excitation       Set of excitation.
 *                           0x00(0) is full step.
 *                           0x01(1) is half step.(supply torque is 100%)
 *                           0x02(2) is half step.(supply torque is 70%)
 *                           0x03(3) is 1/4 step.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::motorRotationStep(float freq, float step, byte direction, byte excitation)
{
  byte result = SUCCESS;
  // check parameters.
  if (FREQ_MIN > freq || FREQ_MAX < freq)
  {
    return FAILURE;
  }
  if (STEP_MIN > step || STEP_MAX < step)
  {
    return FAILURE;
  }
  if (DIRECTION_CW > direction || DIRECTION_CCW < direction)
  {
    return FAILURE;
  }
  if (EXCITATION_FULL > excitation || EXCITATION_QUARTER < excitation)
  {
    return FAILURE;
  }

  _setOutputEnable(OUTPUT_ENABLE);
  _nowFreq = freq;
  _stepFreq = _nowFreq * 2;
  _nowStep = 0;
  _targetStep = (uint32_t)step;
  result |= _setDirection(direction);
  result |= _setExcitation(excitation);
  if (!_isRotation) _fallingEdge = false;
  _isRotation = true;
  return result;
}

/*
 *  \fn     Lib_LV8702V::motorRotationStop()
 *  \brief  Stop the motor rotation.(excitation state)
 *
 *  \param  None.
 *  \return None.
 */
void Lib_LV8702V::motorRotationStop()
{
  _isRotation = false;
  _phaseFlag = 0;
  _nowStep = 0;
}

/*
 *  \fn     Lib_LV8702V::motorRotationFree()
 *  \brief  Stop the motor rotation.(free run)
 *
 *  \param  None.
 *  \return None.
 */
void Lib_LV8702V::motorRotationFree()
{
  _setOutputEnable(OUTPUT_DISABLE);
  _isRotation = false;
  _phaseFlag = 0;
  _nowStep = 0;
}

/*!
 *  \fn     Lib_LV8702V::setEfficiency(byte efficiency, byte driveMargin, byte boostup)
 *  \brief   Set of high efficiency drive parameters.
 *
 *  \param  efficiency       Set of high efficiency mode.
 *                           0x00(0) is normal mode.
 *                           0x01(1) is high efficiency mode.
 *
 *  \param  driveMargin      Set of drive margin.
 *                           0x00(0) is margin small.
 *                           0x01(1) is margin middle.
 *                           0x02(2) is margin large.
 *
 *  \param  boostup          Set a level of boost-up.
 *                           0x00(0) is boostup min.
 *                           0x01(1) is boostup low.
 *                           0x02(2) is boostup high.
 *                           0x03(3) is boostup max.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::setEfficiency(byte efficiency, byte driveMargin, byte boostup)
{
  // Check parameters.
  if (EFFICIENCY_NORMAL > efficiency || EFFICIENCY_HIGH < efficiency)
  {
    return FAILURE;
  }
  if (DRIVE_MARGIN_S > driveMargin || DRIVE_MARGIN_L < driveMargin)
  {
    return FAILURE;
  }
  if (BOOST_UP_MIN > boostup || BOOST_UP_MAX < boostup)
  {
    return FAILURE;
  }

  // Set parameters.
  if(EFFICIENCY_NORMAL == efficiency)
  {
    digitalWrite(D4, LOW);
  }
  else
  {
    digitalWrite(D4, HIGH);
  }

  if(DRIVE_MARGIN_S == driveMargin)
  {
    digitalWrite(D2, LOW);
    digitalWrite(D3, LOW);
  }
  else if(DRIVE_MARGIN_M == driveMargin)
  {
    digitalWrite(D2, HIGH);
    digitalWrite(D3, LOW);
  }
  else
  {
    digitalWrite(D2, LOW);
    digitalWrite(D3, HIGH);
  }

  if(BOOST_UP_MIN == boostup)
  {
    digitalWrite(MOSI, LOW);
    digitalWrite(IO12, LOW);
  }
  else if(BOOST_UP_LOW == boostup)
  {
    digitalWrite(MOSI, HIGH);
    digitalWrite(IO12, LOW);
  }
  else if(BOOST_UP_HIGH == boostup)
  {
    digitalWrite(MOSI, LOW);
    digitalWrite(IO12, HIGH);
  }
  else
  {
    digitalWrite(MOSI, HIGH);
    digitalWrite(IO12, HIGH);
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::readAdc()
 *  \brief  Read the analog pin (A1:VREF) from Arduino.
 *
 *  \return Analog voltage value from selected arduino pin.
 *          (Success case)0(0V) - 1023(5V)
 *          (Failure case)65535
 */
int Lib_LV8702V::readAdc()
{
  return analogRead(A1);
}

/*!
 *  \fn     Lib_LV8702V::readDriveStatus()
 *  \brief  Read the driving status.
 *
 *          Below the configuration of the read value is shown.
 *          - Bit31 - 20 Error count of DST1 pin 
 *          - Bit19 - 8 Error count of DST2 pin 
 *          - Bit7 - 5 Reserved 
 *          - Bit4 Step count error
 *          - Bit3 - 2 Reserved
 *          - Bit1 Real time reading of DST1 pin
 *          - Bit0 Real time reading of DST2 pin
 *
 *  \param  None.
 *  \return 0x00000000(0) - 0x0FF0FF33(267452211)
 */
uint32_t Lib_LV8702V::readDriveStatus() {
  uint32_t result = 0;
  uint8_t  sstVal = PINF & _BV(PF0);
  result |= ((_dstCnt1 & 0xFFF) << 20);
  result |= ((_dstCnt2 & 0xFFF) << 8);

  if ((PINF & _BV(PF4)) == LOW && (PINF & _BV(PF5)) == LOW)
  {
  	_dstVal1 = 0;
  	_dstVal2 = 0;
  }
  else
  {
    result |= (_dstVal1 & 0x1) << 1;
    result |= _dstVal2 & 0x1;
  }
  return result | (sstVal << 5) | (_stepCntErr << 4);
}

/*!
 *  \fn     Lib_LV8702V::clrDstCount()
 *  \brief  Initialize DST1(A3),2(A2) pin error count and step count error.
 *
 *  \param  None.
 *  \return The result of executing the API.
 *          0 is success.
 */
void Lib_LV8702V::clrDstCount()
{
  _dstCnt1 = DST_COUNT_MIN;
  _dstCnt2 = DST_COUNT_MIN;
  _stepCntErr = STEP_COUNT_NON_ERROR;
  return;
}

/*!
 *  \fn     Lib_LV8702V::_setOutputEnable(byte select)
 *  \brief  Local method.
 */
int Lib_LV8702V::_setOutputEnable(byte select)
{
  // Power OFF
  if (select == OUTPUT_DISABLE)
  {
    digitalWrite(IO10, HIGH);
  }
  // Power ON
  else if (select == OUTPUT_ENABLE)
  {
    digitalWrite(IO10, LOW);
  }
  // Param(select) is out of the range [0 - 1]
  else
  {
    return FAILURE;
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::_setDirection()
 *  \brief  Local method.
 */
int Lib_LV8702V::_setDirection(byte select)
{
  // Mask when direction changes.
  if (_direction != select)
  {
    _stepCntErrMask = STEP_COUNT_ERROR_MASK;
  }
  if(DIRECTION_CW == select)
  {
    digitalWrite(D5, LOW);
  }
  else if(DIRECTION_CCW == select)
  {
    digitalWrite(D5, HIGH);
  }
  else
  {
    return FAILURE;
  }
  _direction = select;
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8702V::_setExcitation()
 *  \brief  Local method.
 */

int Lib_LV8702V::_setExcitation(byte select)
{
  // Mask when excitation changes.
  if (_excitation != select)
  {
    _stepCntErrMask = STEP_COUNT_ERROR_MASK;
  }
  if(EXCITATION_FULL == select)
  {
    digitalWrite(IO9, LOW);
    digitalWrite(IO8, LOW);
    _stepCntLimit = STEP_COUNT_LIMIT_FULL;
  }
  else if(EXCITATION_HALF100 == select)
  {
    digitalWrite(IO9, HIGH);
    digitalWrite(IO8, HIGH);
    _stepCntLimit = STEP_COUNT_LIMIT_HALF;
  }
  else if(EXCITATION_HALF70 == select)
  {
    digitalWrite(IO9, HIGH);
    digitalWrite(IO8, LOW);
    _stepCntLimit = STEP_COUNT_LIMIT_HALF;
  }
  else if(EXCITATION_QUARTER == select)
  {
    digitalWrite(IO9, LOW);
    digitalWrite(IO8, HIGH);
    _stepCntLimit = STEP_COUNT_LIMIT_QUARTER;
  }
  else
  {
    return FAILURE;
  }
  _excitation = select;
  return SUCCESS;
}

/*!+
 *  \fn     Lib_LV8702V::timerFire()
 *  \brief  Local method.
 */
float Lib_LV8702V::timerFire()
{
  unsigned long time_Start, time_End;
  uint8_t moniVal,dstTiming;
  if (_isRotation) {
    if (_targetStep != 0) {
      if (_nowStep > _targetStep)
      {
        motorRotationStop();
        return 20;
      }
    }
    time_Start = micros();
    if (_phaseFlag)
    {
      PORTB |= _BV(PB7); // STEP HIGH
    }
    else
    {
      PORTB &= ~_BV(PB7); // STEP LOW
      if (_fallingEdge)
      {
        // Update the DST error count
        dstTiming = (_stepCntLimit >> 2) - 1;
        if ((_stepCnt & dstTiming) == 0x00)
        {
          _dstVal1 = (PINF & _BV(PF4)) >> 4;  // Read A3(DST1) pin
          _dstVal2 = (PINF & _BV(PF5)) >> 5;  // Read A2(DST2) pin
          if (_dstVal1 | _dstVal2 == HIGH)
          {
            if (_dstCnt1 < DST_COUNT_MAX)
            {
              if (_dstVal1 == LOW) _dstCnt1++;
            }
            if (_dstCnt2 < DST_COUNT_MAX)
            {
              if (_dstVal2 == LOW) _dstCnt2++;
            }
          }
        }
        // Update the step count
        if (_reset == RESET_ON)
        {
          _stepCnt = STEP_COUNT_INITIAL;
        }
        else
        {
          moniVal = (PINF & _BV(PF1)) >> 1;  // Read A4(MONI) pin
          if (moniVal == MONI_INITIAL)
          {
            if (_stepCnt != _stepCntLimit - 1)
            {
              if (_stepCntErrMask == STEP_COUNT_ERROR_UNMASK) _stepCntErr = STEP_COUNT_ERROR;
              else                                            _stepCntErrMask--;
            }
            _stepCnt = STEP_COUNT_INITIAL;
          }
          else
          {
            _stepCnt++;
          }
        }
      }
      else
      {
        _fallingEdge = true;
      }
      _nowStep++;
    }
    _phaseFlag ^= 1;

    time_End = micros();
    long tmp = ((1000000.0F / _stepFreq)-(time_End - time_Start));

    return tmp;
  }
  else
  {
    return 20;
  }
}

/*!
 *  \fn     Lib_LV8702V::guiSerialParse(char *serialRecvStr)
 *  \brief  Run the API from the serial message.
 *
 *          It parses the serial message to be sent from the test GUI.
 *          From the message, and run the corresponding API.
 *          This method is a virtual method implementation of
 *          a common class of GuiSerialInterface.
 *
 *  \param  *serialRecvStr    Received serial messages from GUI.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8702V::guiSerialParse(char *serialRecvStr)
{
  // Implement for serial code execution.
  switch (serialRecvStr[0])
  {
    case SRMES_GET_ID:
      Serial.print(ID);
      return SUCCESS;
    case SRMES_POLLING_ID:
      return SUCCESS;
    case SRMES_SET_CHIP_ENABLE:
    {
      SrMesDivSetChipEnable chipValue = *(SrMesDivSetChipEnable *)serialRecvStr;
      return setChipEnable(chipValue.select);
    }
    case SRMES_SET_RESET:
    {
      SrMesDivSetReset resetValue = *(SrMesDivSetReset *)serialRecvStr;
      return setReset(resetValue.select);
    }
    case SRMES_SET_MAX_CURRENT:
    {
      SrMesDivSetMaxCurrent maxCurValue = *(SrMesDivSetMaxCurrent *)serialRecvStr;
      float adpVoltage    = float(((maxCurValue.adpVoltage[1] << 8) | maxCurValue.adpVoltage[0]) / 10.0F);
      float adpCurrent    = float(((maxCurValue.adpCurrent[1] << 8) | maxCurValue.adpCurrent[0]) / 10.0F);
      float mtrCurrent    = float(((maxCurValue.mtrCurrent[1] << 8) | maxCurValue.mtrCurrent[0]) / 10.0F);
      float mtrResistance = float(((maxCurValue.mtrResistance[1] << 8) | maxCurValue.mtrResistance[0]) / 10.0F);
      setMaxCurrent(adpVoltage, adpCurrent, mtrCurrent, mtrResistance);
      
      SrMesDivRecvMaxCurrent recvCurValue;
      recvCurValue.identiffer = SRMES_RES_MAX_CURRENT;
      uint16_t currentMax = _currentMax * 100;
        
      recvCurValue.recvCurrent[0] = (byte)currentMax;
      recvCurValue.recvCurrent[1] = (byte)(currentMax >> 8);
    
      Serial.write(recvCurValue.identiffer);
      Serial.write(recvCurValue.recvCurrent[1]);
      Serial.write(recvCurValue.recvCurrent[0]);
      Serial.write("\n");
      delay(1);
      return SUCCESS;
    }
    case SRMES_SET_REF_VOLTAGE:
    {
      SrMesDivSetRefVoltage vrefValue = *(SrMesDivSetRefVoltage *)serialRecvStr;
      float vref2 = float(((vrefValue.vref[1] << 8) | vrefValue.vref[0]) / 100.0F);
      return setRefVoltage(vref2);
    }
    case SRMES_STEP_ANGLE:
    {
      SrMesDivSetStepAngle stepAngle = *(SrMesDivSetStepAngle *)serialRecvStr;
      float tmp = ((stepAngle.angle[1] << 8) | stepAngle.angle[0]) / 100.0F;
      setStepAngle(tmp);
      return SUCCESS;
    }
    case SRMES_ROTATION_ANGLE:
    {
      SrMesDivRotationDeg rotationAngle = *(SrMesDivRotationDeg *)serialRecvStr;
      float freq = (((uint16_t)(rotationAngle.freq[1] << 8) + rotationAngle.freq[0])) / 10.0F;
        
      int32_t tmp = 0x00000000;
      tmp |= ((int32_t)rotationAngle.angle[3] << 24) & 0xFF000000;
      tmp |= ((int32_t)rotationAngle.angle[2] << 16) & 0x00FF0000;
      tmp |= (rotationAngle.angle[1] << 8) & 0x0000FF00;
      tmp |= (rotationAngle.angle[0]) & 0x000000FF;
      float angle  = tmp / 100.0F;
      return motorRotationDeg(freq, angle, rotationAngle.direction, rotationAngle.excitation);
    }
    case SRMES_ROTATION_TIME:
    {
      SrMesDivRotationTime rotationTime = *(SrMesDivRotationTime *)serialRecvStr;
      float freq    = ((uint16_t)((rotationTime.freq[1] << 8) + rotationTime.freq[0])) / 10.0F;
      float time = (rotationTime.time[1] << 8) + rotationTime.time[0];
      return motorRotationTime(freq, time, rotationTime.direction, rotationTime.excitation);
    }
    case SRMES_ROTATION_STEP:
    {
      SrMesDivRotationStep rotationStep = *(SrMesDivRotationStep *)serialRecvStr;
      float freq    = ((uint16_t)((rotationStep.freq[1] << 8) + rotationStep.freq[0])) / 10.0F;
        
        int32_t tmp = 0x00000000;
        tmp |= ((int32_t)rotationStep.step[3] << 24) & 0xFF000000;
        tmp |= ((int32_t)rotationStep.step[2] << 16) & 0x00FF0000;
        tmp |= (rotationStep.step[1] << 8) & 0x0000FF00;
        tmp |= (rotationStep.step[0]) & 0x000000FF;
    	float step = tmp;
        
      return motorRotationStep(freq, step, rotationStep.direction, rotationStep.excitation);
    }
    case SRMES_ROTATION_STOP:
    {
        motorRotationStop();
        return SUCCESS;
    }
    case SRMES_ROTATION_FREE:
    {
        motorRotationFree();
        return SUCCESS;
    }
    case SRMES_SET_EFFICIENCY:
    {
      SrMesDivSetEfficiency efficiencyValue = *(SrMesDivSetEfficiency *)serialRecvStr;
      return setEfficiency(efficiencyValue.efficiency, efficiencyValue.driveMargin, efficiencyValue.boostup);
    }
    case SRMES_READ_ADC:
    {
      int result = readAdc();
      SrMesDivRecvReadAdc recvAdcValue;
      recvAdcValue.identiffer = SRMES_RES_READ_ADC;
      recvAdcValue.recvAdc[0] = (byte)result;
      recvAdcValue.recvAdc[1] = (byte)(result >> 8);
      
      Serial.write(recvAdcValue.identiffer);
      Serial.write(recvAdcValue.recvAdc[1]);
      Serial.write(recvAdcValue.recvAdc[0]);
      Serial.write("\n");
      return SUCCESS;
    }
    case SRMES_READ_DRIVESTATUS:
    {
      uint32_t result = readDriveStatus();
      SrMesDivRecvReadDriveStatus recvDriveStatusValue;
      recvDriveStatusValue.identiffer = SRMES_RES_READ_DRIVESTATUS;
      recvDriveStatusValue.recvDriveStatus[0] = (byte)result;
      recvDriveStatusValue.recvDriveStatus[1] = (byte)(result >> 8);
      recvDriveStatusValue.recvDriveStatus[2] = (byte)(result >> 16);
      recvDriveStatusValue.recvDriveStatus[3] = (byte)(result >> 24);
      
      Serial.write(recvDriveStatusValue.identiffer);
      Serial.write(recvDriveStatusValue.recvDriveStatus[3]);
      Serial.write(recvDriveStatusValue.recvDriveStatus[2]);
      Serial.write(recvDriveStatusValue.recvDriveStatus[1]);
      Serial.write(recvDriveStatusValue.recvDriveStatus[0]);
      Serial.write("\n");
      return SUCCESS;
    }
    case SRMES_CLR_DST_COUNT:
    {
      clrDstCount();
      return SUCCESS;
    }
    default:
    {
    }
  }
  return FAILURE;
}

/*!
 *  \fn     Lib_LV8702V::timeoutPol()
 *  \brief  Local method.
 */
void Lib_LV8702V::timeoutPol(){
    motorRotationFree();
    _setOutputEnable(OUTPUT_ENABLE);
}
