// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------

 /*!
  * \file
  * \brief ON Semiconductor motor driver control library header file
  *        for Arduino micro.
  *
  *        Available motor driver is LV8548.
  */

#ifndef LV8548STEP_LIB_H
#define LV8548STEP_LIB_H

#include <Arduino.h>


  //#define LV8714TA_INTERRUPT TIMER3_OVF_vect

  /*!
   *  \class motorControlLibraryBase
   *  \brief Motor Control Livrary base class.
   *
   *         This class is defined common members, functions and etc.
   */
class motorControlLibraryBase
{
public:
	static const int SUCCESS = 0;   //!< success value define
	static const int FAILURE = 1;   //!< failure value define
	static const int RESTRICT = 2;   //!< prohibition operate define
	static const uint8_t RX = 0;   //!< Arduino micro RX pin number define
	static const uint8_t TX = 1;   //!< Arduino micro TX pin number define
	static const uint8_t D2 = 2;   //!< Arduino micro Degital pin number define
	static const uint8_t D3 = 3;   //!< Arduino micro Degital pin number define
	static const uint8_t D4 = 4;   //!< Arduino micro Degital pin number define
	static const uint8_t D5 = 5;   //!< Arduino micro Degital pin number define
	static const uint8_t D6 = 6;   //!< Arduino micro Degital pin number define
	static const uint8_t D7 = 7;   //!< Arduino micro Degital pin number define
	static const uint8_t IO8 = 8;   //!< Arduino micro I/O pin number define
	static const uint8_t IO9 = 9;   //!< Arduino micro I/O pin number define
	static const uint8_t IO10 = 10;  //!< Arduino micro I/O pin number define
	static const uint8_t IO11 = 11;  //!< Arduino micro I/O pin number define
	static const uint8_t IO12 = 12;  //!< Arduino micro I/O pin number define
	static const uint8_t IO13 = 13;  //!< Arduino micro I/O pin number define
	static const uint8_t MISO = 14;  //!< Arduino micro MISO pin number define
	static const uint8_t SCK = 15;  //!< Arduino micro SCK  pin number define
	static const uint8_t MOSI = 16;  //!< Arduino micro MOSI pin number define
	static const uint8_t SS = 17;  //!< Arduino micro SS   pin number define
	static const uint8_t A0 = 18;  //!< Arduino micro Analog pin number define
	static const uint8_t A1 = 19;  //!< Arduino micro Analog pin number define
	static const uint8_t A2 = 20;  //!< Arduino micro Analog pin number define
	static const uint8_t A3 = 21;  //!< Arduino micro Analog pin number define
	static const uint8_t A4 = 22;  //!< Arduino micro Analog pin number define
	static const uint8_t A5 = 23;  //!< Arduino micro Analog pin number define
};

/*!
 *  \class GuiSerialInterface
 *  \brief Parse serial message from debug GUI.
 *
 *         This class is defined common members, functions and etc.
 */
class GuiSerialInterface
{
public:
	static const uint16_t DEFAULT_BAUDRATE = 19200; //!< Default serial baud rate

	void guiSerialRead();
	virtual int guiSerialParse(char *serialRecvStr) = 0; // abstract
	virtual void timeoutPol();
	virtual ~GuiSerialInterface()
	{
		// Don't remove!
	}

protected:
	void _serialReadArduino(char *serialRecvStr, const int bufferSize);

private:
	static const uint8_t _SMES_BUFFER_SIZE = 64; //!< Data buffler size from serial recieved
	static const uint32_t POLCOUNTMAX   = 3000;
	uint32_t polCounter;
};

/*!
 *  \class Lib_LV8548
 *  \brief Motor Control Livrary class for LV8548 driver.
 *
 *         This class is defined LV8121V driver control APIs.
 */
class Lib_LV8548Step : public motorControlLibraryBase, public GuiSerialInterface
{
public:
	virtual ~Lib_LV8548Step() {} // Don't remove!
	int initLib();
	int setStepAngle(float deg);
	int motorRotationDeg(float freq, float deg, byte cwccw, byte exc);
	int motorRotationTime(float freq, uint16_t time, byte cwccw, byte exc);
	int motorRotationStep(float freq, uint16_t step, byte cwccw, byte exc);
	void motorRotationStop();
	void motorRotationFree();
	virtual int guiSerialParse(char *type);
	void timerFire(long tm);
	void displayAllforTest();
protected:
	byte inPin[4];
	float StepDeg = ANGLE_MIN;
	float StepFrequency = FREQ_MIN;
	byte Excitation = FULLSTEP;
	byte CwCcw = ROTATION_CW;
	uint32_t Target_Step = 0;
	float TimerCounter = 0;
	int8_t PhaseCounter = 0;
	uint32_t Now_Step = 0;
	bool isRotation = false;
	int setPinMode(byte pin);
	float freqChange(float freq, byte exp);
	void timeoutPol();

private:
	static const uint8_t ROTATION_CW = 0;
	static const uint8_t ROTATION_CCW = 1;
	static const uint8_t FULLSTEP = 0;
	static const uint8_t HALFSTEP = 1;
	static const uint16_t FREQ_MIN = 1;
	static const uint16_t FREQ_MAX = 4800;
	static const uint16_t DEG_MIN = 0;
	static const uint16_t DEG_MAX = 65535;
	static const uint16_t TIME_MIN = 0;
	static const uint16_t TIME_MAX = 65535;
	static const uint16_t STEP_MIN = 0;
	static const uint16_t STEP_MAX = 65535;
	static const uint16_t ANGLE_MIN = 0.01;
	static const uint16_t ANGLE_MAX = 360;

	// Serial message identiffer types
	static const byte SRMES_GET_ID = 0x03;		   // 3
	static const byte SRMES_POLLING_ID  = 0x04;         //  4
	static const byte SRMES_STEP_ANGLE = 0x69;     // 'i'
	static const byte SRMES_ROTATION_ANGLE = 0x6A; // 'j'
	static const byte SRMES_ROTATION_TIME = 0x6B;  // 'k'
	static const byte SRMES_ROTATION_STEP = 0x6C;  // 'l'
	static const byte SRMES_ROTATION_STOP = 0x6E;  // 'n'
	static const byte SRMES_ROTATION_FREE = 0x6F;  // 'o'

	static const byte PWM_FREQ_DividingRatio_1 = 0;//1/1 62.5kHz
	static const byte PWM_FREQ_DividingRatio_8 = 1;//1/8 7.813kHz
	static const byte PWM_FREQ_DividingRatio_64 = 2;//1/64 0.977kHz
	static const byte PWM_FREQ_DividingRatio_256 = 3;//1/256 0.244kHz
	static const byte PWM_FREQ_DividingRatio_1024 = 4;//1/1024 0.061kHz

	// Segment definition of serial message
	typedef struct
	{
		byte identiffer;
		byte angle[2];
	} SrMesDivSetStepAngle;

	typedef struct
	{
		byte identiffer;
		byte frequency[2];
		byte deg[2];
		byte rotation;
		byte exp;
	} SrMesDivRotationDeg;

	typedef struct
	{
		byte identiffer;
		byte frequency[2];
		byte time[2];
		byte rotation;
		byte exp;
	} SrMesDivRotationTime;

	typedef struct
	{
		byte identiffer;
		byte frequency[2];
		byte step[2];
		byte rotation;
		byte exp;
	} SrMesDivRotationStep;

	typedef struct
	{
		byte identiffer;
	} SrMesDivRotationStop;

	typedef struct
	{
		byte identiffer;
	} SrMesDivRotationFree;
};

#endif
