// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------

 /****** Revision History ******/


 /******************************/
#include "LV8548_STEP_Lib.h"

#define ID "LV8548Step_Ver.2.0.0\n"
// --------------------------------------------------------------------
// GuiSerialInterface common class
// --------------------------------------------------------------------
/*!
 *  \fn     GuiSerialInterface::guiSerialRead()
 *  \brief  Read serial messages and execeute that values.
 *
 *          If serial message recieved then parse and execute.
 *          (That minnimum onecharacter or more characters.)
 *          Will be calling abstruct method (guiSerialParse) of motor class.
 *
 *  \return void
 */
void GuiSerialInterface::guiSerialRead()
{
	char *serialRecvStr = new char[_SMES_BUFFER_SIZE] {};
	_serialReadArduino(serialRecvStr, _SMES_BUFFER_SIZE);

	if (serialRecvStr[0] != '\0')
	{
		polCounter = millis();
		// Call to abstruct method.
		guiSerialParse(serialRecvStr);
	}
	else{
		if(millis() - polCounter > POLCOUNTMAX){
			timeoutPol();
			polCounter = millis();
		}
	}
	delete[] serialRecvStr;
}

/*!
 *  \fn     GuiSerialInterface::_serialReadArduino()
 *  \brief  Read serial message into the buffer from computer.
 *
 *  \param  *serialRecvStr   Buffer for received message.
 *  \param  bufferSize       Read buffer size.(max 63)
 *  \return void
 */
void GuiSerialInterface::_serialReadArduino(char *serialRecvStr, const int bufferSize)
{
	int i = 0;

	while (Serial.available())
	{
		char receptChar = Serial.read();
		serialRecvStr[i++] = receptChar;

		// Interrupted if buffer was overflowed.
		if (i >= (bufferSize - 1))
		{
			serialRecvStr[i] = '\0';
			return;
		}
	}
}

byte FullstepForward[4][4] = { {HIGH,LOW,HIGH,LOW},{LOW,HIGH,HIGH,LOW},{LOW,HIGH,LOW,HIGH},{HIGH,LOW,LOW,HIGH} };
byte HalfstepForward[8][4] = {{HIGH,LOW,LOW,LOW},{HIGH,LOW,HIGH,LOW},{LOW,LOW,HIGH,LOW},
								{LOW,HIGH,HIGH,LOW},{LOW,HIGH,LOW,LOW},{LOW,HIGH,LOW,HIGH},{LOW,LOW,LOW,HIGH},{HIGH,LOW,LOW,HIGH}};

// --------------------------------------------------------------------
// LV8548 API class
// 
/*!
 *  \fn     Lib_LV8548::initLib()
 *  \brief  Initialize Aruduino.
 *
 *  \param  None.
 *  \return The result of executing Initialize.
 *          0 is success.
 *          1 is failure.
 */

int Lib_LV8548Step::initLib()
{
	inPin[0] = D3;
	inPin[1] = D5;
	inPin[2] = IO10;
	inPin[3] = IO11;
	//set PinMode
	for (int i = 0; i < 4; i++) {
		setPinMode(inPin[i]);
	}
	return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548Step::setPinMode()
 *  \brief  Local method.
 */
int Lib_LV8548Step::setPinMode(byte pin)
{
	pinMode(pin, OUTPUT);
}

/*!
 *  \fn     Lib_LV8548Step::motorRotationDeg(float freq, float deg, byte cwccw, byte exc)
 *  \brief  Frequency, rotation degree, rotation direction, excitation.
 *
 *  \param  freq            Specify driving frequency.
 *                          0x0A(10)-0x960(2400)
 *
 *          deg             Specify rotation angle.
 *                          0x00(infinity)
 *                          0x01(1)-0xFFFF(65535)
 *
 *          cwccw           Specify rotation direction.
 *                          0x00(0) Clock Wise.
 *                          0x01(1) Counter Clock Wise.
 *
 *          exc             Specify excitation mode.
 *                          0x00(0) FULLSTEP.
 *                          0x01(1) HALFSTEP.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548Step::motorRotationDeg(float freq, float deg, byte cwccw, byte exc)
{
	byte tmp;
	if ((freq < FREQ_MIN) || (freq > FREQ_MAX))
	{
		return FAILURE;
	}
	if ((deg < DEG_MIN) || (deg > DEG_MAX))
	{
		return FAILURE;
	}
	if ((cwccw < ROTATION_CW) || (cwccw > ROTATION_CCW))
	{
		return FAILURE;
	}
	if ((exc < FULLSTEP) || (exc > HALFSTEP))
	{
		return FAILURE;
	}
	if (Excitation != exc) PhaseCounter = 0;
	if (CwCcw != cwccw)
	{
		if(cwccw == ROTATION_CW){
			PhaseCounter++; //ʒuɖ߂
			if(PhaseCounter >= (exc == FULLSTEP ? 4 : 8)){
				PhaseCounter = 0;
			}
			
			PhaseCounter++; //ЂƂi߂
			if(PhaseCounter >= (exc == FULLSTEP ? 4 : 8)){
				PhaseCounter = 0;
			}
		}
		else{
			PhaseCounter--; //ʒuɖ߂
			if(PhaseCounter < 0){
				PhaseCounter = (exc == FULLSTEP ? 3 : 7);
			}
			PhaseCounter--;//ЂƂi߂
			if(PhaseCounter < 0){
				PhaseCounter = (exc == FULLSTEP ? 3 : 7);
			}
		}
	}
	StepFrequency = freqChange(freq, exc);
	Target_Step = deg / StepDeg * (exc == FULLSTEP ? 1 : 2);
	CwCcw = cwccw;
	Excitation = exc;
	Now_Step = 0;
	//TimerCounter = (1000000.0F / StepFrequency);
	//TimerCounter = 0;
	isRotation = true;
	return SUCCESS;
}
/*!
 *  \fn     Lib_LV8548Step::motorRotationTime(float freq, uint16_t time, byte cwccw, byte exc)
 *  \brief  Frequency, rotation time, rotation direction, excitation.
 *
 *  \param  freq            Specify driving frequency.
 *                          0x0A(10)-0x960(2400)
 *
 *          deg             Specify rotation time.
 *                          0x00(infinity)
 *                          0x01(1)-0xFFFF(65535)
 *
 *          cwccw           Specify rotation direction.
 *                          0x00(0) Clock Wise.
 *                          0x01(1) Counter Clock Wise.
 *
 *          exc             Specify excitation mode.
 *                          0x00(0) FULLSTEP.
 *                          0x01(1) HALFSTEP.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548Step::motorRotationTime(float freq, uint16_t time, byte cwccw, byte exc)
{
	if ((freq < FREQ_MIN) || (freq > FREQ_MAX))
	{
		return FAILURE;
	}
	if ((time < TIME_MIN) || (time > TIME_MAX))
	{
		return FAILURE;
	}
	if ((cwccw < ROTATION_CW) || (cwccw > ROTATION_CCW))
	{
		return FAILURE;
	}
	if ((exc < FULLSTEP) || (exc > HALFSTEP))
	{
		return FAILURE;
	}
	if (Excitation != exc) PhaseCounter = 0;
	motorRotationStop();
	
	uint32_t temp1 = time;
	uint32_t temp2 = freq;
	StepFrequency = (freqChange(freq, exc));
	
	uint32_t j = temp1 * temp2;

	Target_Step = j;
	CwCcw = cwccw;
	Excitation = exc;
	Now_Step = 0;
	

	
	
	isRotation = true;
	return SUCCESS;
}
/*!
 *  \fn     Lib_LV8548Step::motorRotationStep(float freq, uint16_t step, byte cwccw, byte exc)
 *  \brief  Frequency, rotation step, rotation direction, excitation.
 *
 *  \param  freq            Specify driving frequency.
 *                          0x0A(10)-0x960(2400)
 *
 *          deg             Specify rotation step.
 *                          0x00(infinity)
 *                          0x01(1)-0xFFFF(65535)
 *
 *          cwccw           Specify rotation direction.
 *                          0x00(0) Clock Wise.
 *                          0x01(1) Counter Clock Wise.
 *
 *          exc             Specify excitation mode.
 *                          0x00(0) FULLSTEP.
 *                          0x01(1) HALFSTEP.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548Step::motorRotationStep(float freq, uint16_t step, byte cwccw, byte exc)
{
	byte tmp;
	if ((freq < FREQ_MIN) || (freq > FREQ_MAX))
	{
		return FAILURE;
	}
	if ((step < STEP_MIN) || (step > STEP_MAX))
	{
		return FAILURE;
	}
	if ((cwccw < ROTATION_CW) || (cwccw > ROTATION_CCW))
	{
		return FAILURE;
	}
	if ((exc < FULLSTEP) || (exc > HALFSTEP))
	{
		return FAILURE;
	}
	if (Excitation != exc) PhaseCounter = 0;
	
	if (CwCcw != cwccw)
	{
		if(cwccw == ROTATION_CW){
			PhaseCounter++; //ʒuɖ߂
			if(PhaseCounter >= (exc == FULLSTEP ? 4 : 8)){
				PhaseCounter = 0;
			}
			
			PhaseCounter++; //ЂƂi߂
			if(PhaseCounter >= (exc == FULLSTEP ? 4 : 8)){
				PhaseCounter = 0;
			}
		}
		else{
			PhaseCounter--; //ʒuɖ߂
			if(PhaseCounter < 0){
				PhaseCounter = (exc == FULLSTEP ? 3 : 7);
			}
			PhaseCounter--;//ЂƂi߂
			if(PhaseCounter < 0){
				PhaseCounter = (exc == FULLSTEP ? 3 : 7);
			}
		}
	}
	
	StepFrequency = freqChange(freq, exc);
	Target_Step = step;
	CwCcw = cwccw;
	Excitation = exc;
	Now_Step = 0;
	isRotation = true;
	return SUCCESS;
}

/*
 *  \fn     Lib_LV8548Step::motorRotationStop()
 *  \brief  Stop the motor rotation.(excitation state)
 *
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548Step::motorRotationStop()
{
	isRotation = false;
	TimerCounter = 0.0F;
	//PhaseCounter = 0;
	Now_Step = 0;
}

/*
 *  \fn     Lib_LV8548Step::motorRotationFree()
 *  \brief  Stop the motor rotation.(free run)
 *
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548Step::motorRotationFree()
{
	isRotation = false;
	TimerCounter = 0.0F;
	PhaseCounter = 0;
	Now_Step = 0;
	for (int i = 0; i < 4; i++)
	{
		digitalWrite(inPin[i], LOW);
	}
}
/*
 *  \fn     Lib_LV8548Step::setStepAngle(float deg)
 *  \brief  Set the step angle.
 *
 *  \param  deg             Set step angle.
 *                          0.01-360.0
 *
 *  \return The result of executing Initialize.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548Step::setStepAngle(float deg)
{
	if ((deg < ANGLE_MIN) || (deg > ANGLE_MAX))
	{
		return FAILURE;
	}
	StepDeg = deg;
	return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548Step::freqChange(float freq, byte exp)
 *  \brief  Local method.
 */
float Lib_LV8548Step::freqChange(float freq, byte exp)
{
	return freq;

	if (exp == FULLSTEP) {
		return freq * 4.0F;
	}
	else {
		return freq * 8.0F;
	}

}

/*!
 *  \fn     Lib_LV8548Step::timerFire()
 *  \brief  Output rectangular wave with timer interrupt.
 *  \param  None.
 *  \return None.
 */
 
 void Lib_LV8548Step::timerFire(long timer)
{
	byte tmp;
	unsigned long time_millisStart, time_millisEnd;
	TimerCounter += (float)timer;//s
	if (isRotation) {

		if (TimerCounter >= (1000000.0F / (StepFrequency))) {
		time_millisStart = micros();
			
			if (Target_Step != 0) {
				if (Now_Step >= ((short)Target_Step))
				{
					isRotation = false;
					Now_Step = 0;
					TimerCounter = 0.0F;
					return;
				}
			}
			for (int i = 0; i < 4; i++) {
				if (Excitation == FULLSTEP) {
					digitalWrite(inPin[i], FullstepForward[PhaseCounter][i]);
				}
				else {
					digitalWrite(inPin[i], HalfstepForward[PhaseCounter][i]);
				}
			}
			if(CwCcw == ROTATION_CW){
				PhaseCounter++;
				tmp = (Excitation == FULLSTEP ? 4 : 8);
				if (PhaseCounter >= tmp) {
					PhaseCounter = 0;
				}
			}
			else{
				PhaseCounter--;
				tmp = (Excitation == FULLSTEP ? 4 : 8);
				if (PhaseCounter <= -1) {
					PhaseCounter = tmp-1;
				}
			}
			Now_Step++;
			time_millisEnd = micros();
			TimerCounter = 0.0F;
			//TimerCounter = (float)(time_millisEnd - time_millisStart);
		}
	}
}

/*!
 *  \fn     Lib_LV8548DC::serialParse(char *serialRecvStr)
 *  \brief  Run the API from the serial message.
 *
 *          It parses the serial message to be sent from the test GUI.
 *          From the message, and run the corresponding API.
 *          This method is a virtual method implementation of
 *          a common class of GuiSerialInterface.
 *
 *  \param  *serialRecvStr    Received serial messages from GUI.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548Step::guiSerialParse(char *serialRecvStr)
{
	// Implement for serial code execution.
	switch (serialRecvStr[0])
	{
	case SRMES_GET_ID:	// 3
		Serial.print(ID);
	      return SUCCESS;
     case SRMES_POLLING_ID:  // 4
      return SUCCESS;
	case SRMES_STEP_ANGLE:
	{
		SrMesDivSetStepAngle stepAngle = *(SrMesDivSetStepAngle *)serialRecvStr;
		float tmp = ((uint16_t)((stepAngle.angle[1] << 8) + stepAngle.angle[0])) / 100.0F;
		setStepAngle(tmp);
		return SUCCESS;
	}
	case SRMES_ROTATION_ANGLE:
	{
		SrMesDivRotationDeg rotationAngle = *(SrMesDivRotationDeg *)serialRecvStr;
		float freq = ((uint16_t)((rotationAngle.frequency[1] << 8) + rotationAngle.frequency[0])) / 10.0F;
		float tmp = ((uint16_t)((rotationAngle.deg[1] << 8) + rotationAngle.deg[0]))  / 100.0F;
		motorRotationDeg(freq, tmp, rotationAngle.rotation, rotationAngle.exp);
		return SUCCESS;
	}
	case SRMES_ROTATION_TIME:
	{
		SrMesDivRotationTime rotationTime = *(SrMesDivRotationTime *)serialRecvStr;
		float freq = ((uint16_t)((rotationTime.frequency[1] << 8) + rotationTime.frequency[0])) / 10.0F;
		uint16_t tmp = (uint16_t)((rotationTime.time[1] << 8) + rotationTime.time[0]) / 10.0F;
		motorRotationTime(freq, tmp, rotationTime.rotation, rotationTime.exp);
		return SUCCESS;
	}
	case SRMES_ROTATION_STEP:
	{
		SrMesDivRotationStep rotationStep = *(SrMesDivRotationStep *)serialRecvStr;
		float freq = ((uint16_t)((rotationStep.frequency[1] << 8) + rotationStep.frequency[0])) / 10.0F;
		uint16_t tmp = (uint16_t)((rotationStep.step[1] << 8) + rotationStep.step[0]) / 10.0F;
		motorRotationStep(freq, tmp, rotationStep.rotation, rotationStep.exp);
		return SUCCESS;
	}
	case SRMES_ROTATION_STOP:
	{
		motorRotationStop();
		return SUCCESS;
	}
	case SRMES_ROTATION_FREE:
	{
		motorRotationFree();
		return SUCCESS;
	}
	default:
	{
		Serial.print("unknown command");
	}
	}
	return FAILURE;
}

/*!
 *  \fn     Lib_LV8548Step::timeoutPol()
 *  \brief  Stop the motor with polling timeout.
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548Step::timeoutPol(){
	motorRotationFree();
}

/*!
 *  \fn     Lib_LV8548Step::displayAllforTest()
 *  \brief  Display the parameter for API test.
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548Step::displayAllforTest() {
	char buf[256];
	sprintf(buf, "*******************************************\n");
	Serial.write(buf);
	dtostrf(StepDeg, 5, 2, buf);
	Serial.write("StepDeg:");
	Serial.write(buf);
	Serial.write("\n");
	dtostrf(StepFrequency, 7, 2, buf);
	Serial.write("StepFrequency:");
	Serial.write(buf);
	Serial.write("\n");
	sprintf(buf, "Target_Step: %u\n", Target_Step);
	Serial.write(buf);
	sprintf(buf, "CwCcw: %d\n", CwCcw);
	Serial.write(buf);
	sprintf(buf, "Excitation: %d\n", Excitation);
	Serial.write(buf);
	sprintf(buf, "isRotation: %d\n", isRotation);
	Serial.write(buf);
	sprintf(buf, "PhaseCounter: %d\n", PhaseCounter);
	Serial.write(buf);
	dtostrf(Now_Step, 5, 2, buf);
	Serial.write("Now_Step:");
	Serial.write(buf);
	Serial.write("\n");
	sprintf(buf, "*******************************************\n");
	Serial.write(buf);
}
