// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------

/*!
 * \file
 * \brief ON Semiconductor motor driver control library header file
 *        for Arduino micro.
 *
 *        Available motor driver is LV8548.
 */

#ifndef LV8548_LIB_H
#define LV8548_LIB_H

#include <Arduino.h>

#ifndef ArduinoMock_h
#include <SPI.h>
#endif


/*!
 *  \class motorControlLibraryBase
 *  \brief Motor Control Livrary base class.
 *
 *         This class is defined common members, functions and etc.
 */
class motorControlLibraryBase
{
  public:
    static const int SUCCESS  = 0;   //!< success value define
    static const int FAILURE  = 1;   //!< failure value define
    static const int RESTRICT = 2;   //!< prohibition operate define
    static const uint8_t RX   = 0;   //!< Arduino micro RX pin number define
    static const uint8_t TX   = 1;   //!< Arduino micro TX pin number define
    static const uint8_t D2   = 2;   //!< Arduino micro Degital pin number define
    static const uint8_t D3   = 3;   //!< Arduino micro Degital pin number define
    static const uint8_t D4   = 4;   //!< Arduino micro Degital pin number define
    static const uint8_t D5   = 5;   //!< Arduino micro Degital pin number define
    static const uint8_t D6   = 6;   //!< Arduino micro Degital pin number define
    static const uint8_t D7   = 7;   //!< Arduino micro Degital pin number define
    static const uint8_t IO8  = 8;   //!< Arduino micro I/O pin number define
    static const uint8_t IO9  = 9;   //!< Arduino micro I/O pin number define
    static const uint8_t IO10 = 10;  //!< Arduino micro I/O pin number define
    static const uint8_t IO11 = 11;  //!< Arduino micro I/O pin number define
    static const uint8_t IO12 = 12;  //!< Arduino micro I/O pin number define
    static const uint8_t IO13 = 13;  //!< Arduino micro I/O pin number define
    static const uint8_t MISO = 14;  //!< Arduino micro MISO pin number define
    static const uint8_t SCK  = 15;  //!< Arduino micro SCK  pin number define
    static const uint8_t MOSI = 16;  //!< Arduino micro MOSI pin number define
    static const uint8_t SS   = 17;  //!< Arduino micro SS   pin number define
    static const uint8_t A0   = 18;  //!< Arduino micro Analog pin number define
    static const uint8_t A1   = 19;  //!< Arduino micro Analog pin number define
    static const uint8_t A2   = 20;  //!< Arduino micro Analog pin number define
    static const uint8_t A3   = 21;  //!< Arduino micro Analog pin number define
    static const uint8_t A4   = 22;  //!< Arduino micro Analog pin number define
    static const uint8_t A5   = 23;  //!< Arduino micro Analog pin number define
};

/*!
 *  \class GuiSerialInterface
 *  \brief Parse serial message from debug GUI.
 *
 *         This class is defined common members, functions and etc.
 */
class GuiSerialInterface
{
  public:
    static const uint16_t DEFAULT_BAUDRATE = 19200; //!< Default serial baud rate

    void guiSerialRead();
    virtual int guiSerialParse(char *serialRecvStr) = 0; // abstract
	virtual void timeoutPol();
    virtual ~GuiSerialInterface()
    {
      // Don't remove!
    }

  protected:
    void _serialReadArduino(char *serialRecvStr, const int bufferSize);
	static const byte PWM_FREQ_DividingRatio_8    = 0;  //division ratio 1/8 7.813kHz
    static const byte PWM_FREQ_DividingRatio_64   = 1;  //division ratio 1/64 0.977kHz
    static const byte PWM_FREQ_DividingRatio_256  = 2;  //division ratio 1/256 0.244kHz
    static const byte PWM_FREQ_DividingRatio_1024 = 3;  //division ratio 1/1024 0.061kHz
	uint8_t pwmFreq = 1;

  private:
    static const uint8_t _SMES_BUFFER_SIZE = 64; //!< Data buffler size from serial recieved
	static const uint32_t POLCOUNTMAX   = 3000;
	uint32_t polCounter;


};

/*!
 *  \class Lib_LV8548
 *  \brief Motor Control Library class for LV8548 driver.
 *
 *         This class is defined LV8548 driver control APIs.
 */
class Lib_LV8548DC: public motorControlLibraryBase, public GuiSerialInterface
{
  public:
    virtual ~Lib_LV8548DC() {} // Don't remove!
    int initLib();
    int initLibPortSet(byte in1,byte in2,byte in3,byte in4);
    int setPWMFrequency(byte hz);
    int setCtlVoltage(byte motorNo,byte duty);
    int setRotation(byte motorNo,byte select);
    int setStartFlag(byte motorNo,byte select);
    virtual int guiSerialParse(char *type);
    void displayAllforTest();
  protected:
    uint8_t motor1RotateMode = ROTATION_FW_BK;
    uint8_t motor2RotateMode = ROTATION_FW_BK;
    float motor1Duty = DUTY_MIN;
    float motor2Duty = DUTY_MIN;
    uint8_t motor1StartFlag = FLAG_OFF;
    uint8_t motor2StartFlag = FLAG_OFF;
    uint8_t inPin[4];
    byte dutyRateToValue(int rate);
    byte dutyRateToReverseValue(byte rate);
	void timeoutPol();
    int motorRotation(byte motorNo);
    int setTCCR0();
    int setTCCR1();
    int setTCCR3();
    int setTCCR4();
    int setPinMode(byte pin);

  private:
    static const uint8_t DUTY_MIN       = 0;     //!< duty rate 0% = 0
    static const uint8_t DUTY_MAX       = 100;   //!< duty rate 100% = 255
    static const uint8_t FLAG_OFF       = 0;     //!< set start flag off.
    static const uint8_t FLAG_ON        = 1;     //!< set start flag on.
    static const uint8_t FLAG_BRAKE     = 2;     //!< set start flag brake.
    static const uint8_t ROTATION_FW_BK = 1;     //!< set foward & brake rotation.
    static const uint8_t ROTATION_FW_SB = 0;     //!< set foward & standby rotation.
    static const uint8_t ROTATION_RV_BK = 3;     //!< set reverse & brake rotation.
    static const uint8_t ROTATION_RV_SB = 2;     //!< set reverse & standby rotation.
    static const uint8_t ROTATION_SB_SB = 4;     //!< set stanby & standby rotation.(Stop)
    static const uint8_t ROTATION_BK_BK = 5;     //!< set brake & brake rotation.(Stop)
    static const uint8_t MOTOR_NO1      = 0;     //!< set motor number1.
    static const uint8_t MOTOR_NO2      = 1;     //!< set motor number2.
	

	
    // Serial message identiffer types
    static const byte SRMES_GET_ID      = 0x03;         //  3
    static const byte SRMES_POLLING_ID  = 0x04;         //  4
    static const byte SRMES_CTL_VOLTAGE = 0x41;         // 'A'
    static const byte SRMES_ROTATION    = 0x44;         // 'D'
    static const byte SRMES_PWM_FREQSET = 0x67;         // 'g'
    static const byte SRMES_START_FLAG  = 0x68;         // 'h'

    
    // Segment definition of serial message
    typedef struct
    {
      byte identiffer;
      byte motorNo;
      byte duty;
    } SrMesDivCtlVoltage;

    typedef struct
    {
      byte identiffer;
      byte motorNo;
      byte select;
    } SrMesDivStartFlag;

    typedef struct
    {
      byte identiffer;
      byte motorNo;
      byte select;
    } SrMesDivRotation;
    
    typedef struct
    {
      byte identiffer;
      byte hz;
    } SrMesDivPwmFreqset;
};

#endif
