// ----------------------------------------------------------------------------

// Copyright (c) 2016 Semiconductor Components Industries LLC

// (d/b/a "ON Semiconductor").  All rights reserved.

// This software and/or documentation is licensed by ON Semiconductor under

// limited terms and conditions.  The terms and conditions pertaining to the

// software and/or documentation are available at

// http://www.onsemi.com/site/pdf/ONSEMI_T&C.pdf ("ON Semiconductor Standard

// Terms and Conditions of Sale, Section 8 Software") and if applicable the

// software license agreement.  Do not use this software and/or documentation

// unless you have carefully read and you agree to the limited terms and

// conditions.  By using this software and/or documentation, you agree to the

// limited terms and conditions.

// ----------------------------------------------------------------------------/


#include "LV8548_DC_Lib.h"

#define ID "LV8548DC_Ver.2.0.0\n"

// --------------------------------------------------------------------
// GuiSerialInterface common class
// --------------------------------------------------------------------
/*!
 *  \fn     GuiSerialInterface::guiSerialRead()
 *  \brief  Read serial messages and execeute that values.
 *
 *          If serial message recieved then parse and execute.
 *          (That minimum onecharacter or more characters.)
 *          Will be calling abstruct method (guiSerialParse) of motor class.
 *
 *  \return void
 */
void GuiSerialInterface::guiSerialRead()
{
  char *serialRecvStr = new char[_SMES_BUFFER_SIZE] {};
  _serialReadArduino(serialRecvStr, _SMES_BUFFER_SIZE);
  uint32_t polCountMax;
  switch(pwmFreq){
        case PWM_FREQ_DividingRatio_8: polCountMax = POLCOUNTMAX * 8; break;
        case PWM_FREQ_DividingRatio_64: polCountMax = POLCOUNTMAX; break;
        case PWM_FREQ_DividingRatio_256: polCountMax = POLCOUNTMAX / 4; break;
        case PWM_FREQ_DividingRatio_1024: polCountMax = POLCOUNTMAX / 16; break;
  }
  if (serialRecvStr[0] != '\0')
  {
  	polCounter = millis();
    // Call to abstruct method.
    guiSerialParse(serialRecvStr);
  }
	else{
		if(millis() - polCounter > polCountMax){
			timeoutPol();
			polCounter = millis();
		}
	}
  delete[] serialRecvStr;
}

/*!
 *  \fn     GuiSerialInterface::_serialReadArduino()
 *  \brief  Read serial message into the buffer from computer.
 *
 *  \param  *serialRecvStr   Buffer for received message.
 *  \param  bufferSize       Read buffer size.(max 63)
 *  \return void
 */
void GuiSerialInterface::_serialReadArduino(char *serialRecvStr, const int bufferSize)
{
  int i = 0;

  while (Serial.available())
  {
    char receptChar = Serial.read();
    serialRecvStr[i++] = receptChar;

    // Interrupted if buffer was overflowed.
    if (i >= (bufferSize - 1))
    {
      serialRecvStr[i] = '\0';
      return;
    }
  }
}

// --------------------------------------------------------------------
// LV8548 API class
// --------------------------------------------------------------------
/*!
 *  \fn     Lib_LV8548::initLib()
 *  \brief  Initialize Arduino.
 *
 *          Below set the parameters.
 *          - setRotation(0, 4)
 *          - setCtlVoltage(0, 0)
 *          - setRotation(1, 4)
 *          - setCtlVoltage(1, 0)
 *          - setStartFlag(0, 0)
 *          - setStartFlag(1, 0)
 *          - setPWMFrequency(1)
 *
 *  \param  None.
 *  \return The result of executing Initialize.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::initLib()
{
  /* 
   *  Set D3,IO11Pin PWMF 7.813[kHz]
   */
  TCCR0B &= B11111000;
  TCCR0B |= B00000010;
  
  /*
   * Set IO10Pin Fast PWM Mode & PWMF 7.813[kHz]
   */
  TCCR1B &= B11100000;
  TCCR1B |= B00001010;
  TCCR1A &= B11111100;
  TCCR1A |= B00000001;
   /*
   * Set D5Pin Fast PWM Mode & PWMF 7.813[kHz]
   */
  TCCR3B &= B11100000;
  TCCR3B |= B00001010;
  TCCR3A &= B11111100;
  TCCR3A |= B00000001;
  
  inPin[0] = D5;
  inPin[1] = D3;
  inPin[2] = IO10;
  inPin[3] = IO11;
  //set PinMode
  for(int i=0;i<4;i++){
    setPinMode(inPin[i]);
  }
  
  // set init status.
  uint8_t result = SUCCESS;
  result |= setRotation(MOTOR_NO1,ROTATION_SB_SB);
  result |= setCtlVoltage(MOTOR_NO1,DUTY_MIN);
  result |= setRotation(MOTOR_NO2,ROTATION_SB_SB);
  result |= setCtlVoltage(MOTOR_NO2,DUTY_MIN);
  result |= setStartFlag(MOTOR_NO1,FLAG_OFF);
  result |= setStartFlag(MOTOR_NO2,FLAG_OFF);
  result |= setPWMFrequency(1);

  if(SUCCESS == result)
  {
    return SUCCESS;
  }
  else
  {
    return FAILURE;
  }
}

/*!
 *  \fn     Lib_LV8548DC::setTCCR0()
 *  \brief  Local method.
 */
int Lib_LV8548DC::setTCCR0()
{
  /* 
   *  Set D3,IO11Pin PWMF 7.813[kHz]
   */
  TCCR0B &= B11111000;
  TCCR0B |= B00000010;
}

/*!
 *  \fn     Lib_LV8548DC::setTCCR1()
 *  \brief  Local method.
 */
int Lib_LV8548DC::setTCCR1()
{
  /*
   * Set IO10Pin,IO9Pin Fast PWM Mode & PWMF 7.813[kHz]
   */
  TCCR1B &= B11100000;
  TCCR1B |= B00001010;
  TCCR1A &= B11111100;
  TCCR1A |= B00000001;
}

/*!
 *  \fn     Lib_LV8548DC::setTCCR3()
 *  \brief  Local method.
 */
int Lib_LV8548DC::setTCCR3()
{
   /*
   * Set D5Pin Fast PWM Mode & PWMF 7.813[kHz]
   */
  TCCR3B &= B11100000;
  TCCR3B |= B00001010;
  TCCR3A &= B11111100;
  TCCR3A |= B00000001;
}

/*!
 *  \fn     Lib_LV8548DC::setTCCR4()
 *  \brief  Local method.
 */
int Lib_LV8548DC::setTCCR4()
{
   /*
   * Set D6Pin,IO13Pin Fast PWM Mode & PWMF 7.813[kHz]
   */
  TCCR4A = 0b10100011;
  TCCR4B &= B11110000;
  TCCR4B |= B00000100;
  TCCR4D = 0x00;
}

/*!
 *  \fn     Lib_LV8548DC::setPinMode()
 *  \brief  Local method.
 */
int Lib_LV8548DC::setPinMode(byte pin)
{
    pinMode(pin, OUTPUT);
}

/*!
 *  \fn     Lib_LV8548DC::initLibPortSet(byte in1,byte in2,byte in3,byte in4)
 *  \brief  Initialize Arduino and set the selected pin to IN1-IN4 for PWM output.
 *
 *          Below set the parameters.
 *          - setRotation(0, 4)
 *          - setCtlVoltage(0, 0)
 *          - setRotation(1, 4)
 *          - setCtlVoltage(1, 0)
 *          - setStartFlag(0, 0)
 *          - setStartFlag(1, 0)
 *          - setPWMFrequency(1)
 *
 *  \param  in1    set the IN1.
 *                 0x03(3)  Arduino micro D3 pin.
 *                 0x05(5)  Arduino micro D5 pin.
 *                 0x06(6)  Arduino micro D6 pin.
 *                 0x09(9)  Arduino micro D9 pin.
 *                 0x10(10) Arduino micro IO10 pin.
 *                 0x11(11) Arduino micro IO11 pin.
 *                 0x13(13) Arduino micro IO13 pin.
 *  \param  in2    set the IN2.
 *                 0x03(3)  Arduino micro D3 pin.
 *                 0x05(5)  Arduino micro D5 pin.
 *                 0x06(6)  Arduino micro D6 pin.
 *                 0x09(9)  Arduino micro D9 pin.
 *                 0x10(10) Arduino micro IO10 pin.
 *                 0x11(11) Arduino micro IO11 pin.
 *                 0x13(13) Arduino micro IO13 pin.
 *  \param  in3    set the IN3.
 *                 0x03(3)  Arduino micro D3 pin.
 *                 0x05(5)  Arduino micro D5 pin.
 *                 0x06(6)  Arduino micro D6 pin.
 *                 0x09(9)  Arduino micro D9 pin.
 *                 0x10(10) Arduino micro IO10 pin.
 *                 0x11(11) Arduino micro IO11 pin.
 *                 0x13(13) Arduino micro IO13 pin.
 *  \param  in4    set the IN4.
 *                 0x03(3)  Arduino micro D3 pin.
 *                 0x05(5)  Arduino micro D5 pin.
 *                 0x06(6)  Arduino micro D6 pin.
 *                 0x09(9)  Arduino micro D9 pin.
 *                 0x10(10) Arduino micro IO10 pin.
 *                 0x11(11) Arduino micro IO11 pin.
 *                 0x13(13) Arduino micro IO13 pin.
 *
 *  \return The result of executing Initialize.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::initLibPortSet(byte in1,byte in2,byte in3,byte in4)
{
    byte in[4] = {in1,in2,in3,in4};
    for(int j = 0 ; j < 4 ; j++){
        switch(in[j]){
            case 0: return 1;//pin Error
            case 1: return 1;//pin Error
            case 2: return 1;//not PWM Port
            case 3: 
              setTCCR0();
              inPin[j] = D3;
              break;
            case 4: return 1;//not PWM Port
            case 5:
              setTCCR3();
              inPin[j] = D5;
              break;
            case 6:
              setTCCR4();
              inPin[j] = D6;
              break;
            case 7:return 1;//not PWM Port
            case 8:return 1;//not PWM Port
            case 9:
              setTCCR1();
              inPin[j] = IO9;
              break;
            case 10:
              setTCCR1();
              inPin[j] = IO10;
              break;
            case 11:
              setTCCR0();
              inPin[j] = IO11;
              break;
            case 12:return 1;//not PWM Port
            case 13:
              setTCCR4();
              inPin[j] = IO13;
              break;
            default: return 1;
        }
    }
    //set PinMode
    for(int i=0;i<4;i++){
        setPinMode(inPin[i]);
    }

    // set init status.
    uint8_t result = SUCCESS;
    result |= setRotation(MOTOR_NO1, ROTATION_SB_SB);
    result |= setCtlVoltage(MOTOR_NO1, DUTY_MIN);
    result |= setRotation(MOTOR_NO2, ROTATION_SB_SB);
    result |= setCtlVoltage(MOTOR_NO2, DUTY_MIN);
    result |= setStartFlag(MOTOR_NO1, FLAG_OFF);
    result |= setStartFlag(MOTOR_NO2, FLAG_OFF);
    result |= setPWMFrequency(1);

    if (SUCCESS == result)
    {
        return SUCCESS;
    }
    else
    {
        return FAILURE;
    }
}

/*!
 *  \fn     Lib_LV8548DC::setCtlVoltage()
 *  \brief  Set duty.
 *
 *  \param  motorNo     Specify motor number.
 *                      0x00(0) motor 1.
 *                      0x01(1) motor 2.
 *  \param  duty        Set duty ratio.
 *                      Duty range is 0x00(0) to 0x64(100).
 *                      0x00(0) is mean to set duty 0%, and 0x64(100) is mean to set duty 100%.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::setCtlVoltage(byte motorNo,byte duty)
{
  // motor range check
  if(( motorNo < MOTOR_NO1) || ( motorNo > MOTOR_NO2))
  {
    return FAILURE;
  }
  // duty range check
  if ((duty < DUTY_MIN) || (duty > DUTY_MAX))
  {
    // out of range
    return FAILURE;
  }
  
  // convert duty rate to duty value
  byte dutyVal = dutyRateToValue(duty);
  if(motorNo == MOTOR_NO1){
    motor1Duty = duty;
    motorRotation(MOTOR_NO1);
  }
  else{
    motor2Duty = duty;
    motorRotation(MOTOR_NO2);
  }
#ifdef GUI_DEBUG
  char buf[256];
  sprintf(buf,"Change Duty MotorNo = %d duty=%d \n",motorNo,duty);
  Serial.write(buf);
#endif
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548DC::setRotation()
 *  \brief  Set PWM Mode
 *
 *  \param  motorNo     Specify motor number.
 *                      0x00(0) motor 1.
 *                      0x01(1) motor 2.
 *  \param  select      set PWM Mode[0-5].
 *                      0x00(0) Forward - Open(Standby).
 *                      0x01(1) Forward - Brake.
 *                      0x02(2) Reverse - Open(Standby).
 *                      0x03(3) Reverse - Brake.
 *                      0x04(4) Open(Standby) - Open(Standby).
 *                      0x05(5) Brake.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::setRotation(byte motorNo,byte select)
{
  // motor range check
  if(( motorNo < MOTOR_NO1) || ( motorNo > MOTOR_NO2))
  {
    return FAILURE;
  }
  // If select is out of the range, return FAILURE
  if (select < ROTATION_FW_SB || select > ROTATION_BK_BK)
  {
    return FAILURE;
  }
#ifdef GUI_DEBUG
  char buf[256];
  sprintf(buf,"Change PWM Mode  MotorNo = %d mode=%d \n",motorNo,select);
  Serial.print(buf);
#endif
  if(motorNo == MOTOR_NO1){
    motor1RotateMode = select;
    motorRotation(MOTOR_NO1);
  }
  else{
    motor2RotateMode = select;
    motorRotation(MOTOR_NO2);
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548DC::setStartFlag()
 *  \brief  Set StartFlag
 *
 *  \param  motorNo     Specify motor number.
 *                      0x00(0) motor 1.
 *                      0x01(1) motor 2.
 *  \param  select      set Flag [0-1].
 *                      0x00(0) motor start flag OFF.
 *                      0x01(1) motor start flag ON.
 *
 *  \return The result of executing ChipSelect.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::setStartFlag(byte motorNo,byte select)
{
   // motor range check
  if(( motorNo < MOTOR_NO1) || ( motorNo > MOTOR_NO2))
  {
    return FAILURE;
  }
  // If select is out of the range, return FAILURE
  if (select < FLAG_OFF || select > FLAG_BRAKE)
  {
    return FAILURE;
  }
#ifdef GUI_DEBUG
  char buf[256];
  sprintf(buf,"Change StartFlag  MotorNo = %d flag=%d \n",motorNo,select);
  Serial.print(buf);
#endif
  if(motorNo == MOTOR_NO1){
    motor1StartFlag = select;
    motorRotation(MOTOR_NO1);
  }
  else{
    motor2StartFlag = select;
    motorRotation(MOTOR_NO2);
  }
  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548DCV::dutyRateToValue(int rate)
 *  \brief  Local method.
 */
byte Lib_LV8548DC::dutyRateToValue(int rate)
{
  return (byte)round(((double )(rate * 255) / 100));
}

/*!
 *  \fn     Lib_LV8548DCV::dutyRateToReverseValue(byte rate)
 *  \brief  Local method.
 */
byte Lib_LV8548DC::dutyRateToReverseValue(byte rate)
{
  return (255 - rate);
}

/*!
 *  \fn     Lib_LV8548DCV::motorRotation(byte motorNo)
 *  \brief  Convert duty rate (%) to duty value.

 *  \param  motorNo     Specify motor number.
 *                      0x00(0) motor 1.
 *                      0x01(1) motor 2.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::motorRotation(byte motorNo)
{  
  // convert duty rate to duty value
  if(motorNo == MOTOR_NO1){
    if(motor1StartFlag == FLAG_OFF){
        digitalWrite(inPin[1],LOW);
        digitalWrite(inPin[0],LOW);
        return SUCCESS;
    }
      if(motor1StartFlag == FLAG_BRAKE){
        digitalWrite(inPin[1],HIGH);
        digitalWrite(inPin[0],HIGH);
        return SUCCESS;
    }
    byte dutyVal = dutyRateToValue(motor1Duty);
    switch(motor1RotateMode){
        case ROTATION_FW_BK:
            analogWrite(inPin[0],dutyRateToReverseValue(dutyVal));
            digitalWrite(inPin[1],HIGH);
            break;
        case ROTATION_FW_SB:
            analogWrite(inPin[1],dutyVal);
            digitalWrite(inPin[0],LOW);
            break;
        case ROTATION_RV_BK:
            analogWrite(inPin[1],dutyRateToReverseValue(dutyVal));
            digitalWrite(inPin[0],HIGH);
            break;
        case ROTATION_RV_SB:
            analogWrite(inPin[0],dutyVal);
            digitalWrite(inPin[1],LOW);
            break;
        case ROTATION_SB_SB:
            digitalWrite(inPin[1],LOW);
            digitalWrite(inPin[0],LOW);
            break;
        case ROTATION_BK_BK:
            digitalWrite(inPin[1],HIGH);
            digitalWrite(inPin[0],HIGH);
            break;
    }
  }
  else{
    if(motor2StartFlag == FLAG_OFF){
        digitalWrite(inPin[2],LOW);
        digitalWrite(inPin[3],LOW);
        return SUCCESS;
    }
      if(motor2StartFlag == FLAG_BRAKE){
        digitalWrite(inPin[2],HIGH);
        digitalWrite(inPin[3],HIGH);
        return SUCCESS;
    }
    byte dutyVal = dutyRateToValue(motor2Duty);
    switch(motor2RotateMode){
        case ROTATION_FW_BK:
            analogWrite(inPin[3],dutyRateToReverseValue(dutyVal));
            digitalWrite(inPin[2],HIGH);
            break;
        case ROTATION_FW_SB:
            analogWrite(inPin[2],dutyVal);
            digitalWrite(inPin[3],LOW);
            break;
        case ROTATION_RV_BK:
            analogWrite(inPin[2],dutyRateToReverseValue(dutyVal));
            digitalWrite(inPin[3],HIGH);
            break;
        case ROTATION_RV_SB:
            analogWrite(inPin[3],dutyVal);
            digitalWrite(inPin[2],LOW);
            break;
        case ROTATION_SB_SB:
            digitalWrite(inPin[2],LOW);
            digitalWrite(inPin[3],LOW);
            break;
        case ROTATION_BK_BK:
            digitalWrite(inPin[2],HIGH);
            digitalWrite(inPin[3],HIGH);
            break;
    }
  }
}

/*!
 *  \fn     Lib_LV8548DC::setPWMFrequency()
 *  \brief  Set PWM frequency
 *
 *  \param  select      set PWM Ratio.
 *                      0x00(0) division ratio 1/8, frequency 7.813kHz.
 *                      0x01(1) division ratio 1/64, frequency 0.977kHz.
 *                      0x02(2) division ratio 1/256, frequency 0.244kHz.
 *                      0x03(3) division ratio 1/1024, frequency 0.061kHz.
 *
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::setPWMFrequency(byte selectNo)
{
  byte cs = B00000000;
  pwmFreq = selectNo;
  if((selectNo < PWM_FREQ_DividingRatio_8)||(selectNo > PWM_FREQ_DividingRatio_1024)){
      return FAILURE;
  }
  for(int i = 0 ; i < 4 ;i++){
    switch(selectNo){
        case PWM_FREQ_DividingRatio_8: cs = B00000010; break;
        case PWM_FREQ_DividingRatio_64: cs = B00000011; break;
        case PWM_FREQ_DividingRatio_256: cs = B00000100; break;
        case PWM_FREQ_DividingRatio_1024: cs = B00000101; break;
        default: return FAILURE;
  }
    switch(inPin[i]){
        case IO11:
        case D3:
      {
        TCCR0B &= B11111000;
        TCCR0B |= cs;
        break;
      }
        case IO10:
        case IO9:
      {
        TCCR1B &= B11111000;
        TCCR1B |= cs;
        break;
      }
        case D5:
      {
        TCCR3B &= B11111000;
        TCCR3B |= cs;
        break;
      }
        case D6:
        case IO13:
      {
        switch(selectNo){
          case PWM_FREQ_DividingRatio_8: cs = B00000100; break;
          case PWM_FREQ_DividingRatio_64: cs = B00000111; break;
          case PWM_FREQ_DividingRatio_256: cs = B00001001; break;
          case PWM_FREQ_DividingRatio_1024: cs = B00001011; break;
        }
        TCCR4B &= B11110000;
        TCCR4B |= cs;
        break;
      }
    }
  }
#ifdef GUI_DEBUG
  char buf[256];
  sprintf(buf,"Change Frequency = %d \n",selectNo);
  Serial.write(buf);
#endif

  return SUCCESS;
}

/*!
 *  \fn     Lib_LV8548DC::serialParse(char *serialRecvStr)
 *  \brief  Run the API from the serial message.
 *
 *          It parses the serial message to be sent from the test GUI.
 *          From the message, and run the corresponding API.
 *          This method is a virtual method implementation of
 *          a common class of GuiSerialInterface.
 *
 *  \param  *serialRecvStr    Received serial messages from GUI.
 *  \return The result of executing the API.
 *          0 is success.
 *          1 is failure.
 */
int Lib_LV8548DC::guiSerialParse(char *serialRecvStr)
{
  // Implement for serial code execution.
  switch (serialRecvStr[0])
  {
    case SRMES_GET_ID:  // 3
      Serial.print(ID);
      return SUCCESS;
     case SRMES_POLLING_ID:  // 4
      return SUCCESS;
    case SRMES_CTL_VOLTAGE:
    {
      SrMesDivCtlVoltage voltValue = *(SrMesDivCtlVoltage *)serialRecvStr;
      return setCtlVoltage(voltValue.motorNo,voltValue.duty);
    }
    case SRMES_ROTATION:
    {
      SrMesDivRotation rotationValue = *(SrMesDivRotation *)serialRecvStr;
      return setRotation(rotationValue.motorNo,rotationValue.select);
    }
    case SRMES_PWM_FREQSET:
    {
      SrMesDivPwmFreqset freqValue = *(SrMesDivPwmFreqset *)serialRecvStr;
      return setPWMFrequency(freqValue.hz);
    }
    case SRMES_START_FLAG:
    {
        SrMesDivStartFlag startFlag = *(SrMesDivStartFlag *)serialRecvStr;
        return setStartFlag(startFlag.motorNo,startFlag.select);
    }
    default:
    {
      Serial.print("unknown command");
    }
  }
  return FAILURE;
}

/*!
 *  \fn     Lib_LV8548DC::timeoutPol()
 *  \brief  Stop the motor with polling timeout.
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548DC::timeoutPol(){
	setStartFlag(MOTOR_NO1,FLAG_OFF);
	setStartFlag(MOTOR_NO2,FLAG_OFF);
}

/*!
 *  \fn     Lib_LV8548DC::displayAllforTest()
 *  \brief  Display the parameter for API test.
 *  \param  None.
 *  \return None.
 */
void Lib_LV8548DC::displayAllforTest(){
    char buf[256];
    sprintf(buf,"*******************************************\n");
    Serial.write(buf);
    sprintf(buf,"MotorPWMMode1 : %d\n",motor1RotateMode);
    Serial.write(buf);
    sprintf(buf,"MotorPWMMode2 : %d\n",motor2RotateMode);
    Serial.write(buf);
    sprintf(buf,"motor1Duty    : %d\n",(int)motor1Duty);
    Serial.write(buf);
    sprintf(buf,"motor2Duty    : %d\n",(int)motor2Duty);
    Serial.write(buf);
    sprintf(buf,"motor1StartFlg: %d\n",motor1StartFlag);
    Serial.write(buf);
    sprintf(buf,"motor2StartFlg: %d\n",motor2StartFlag);
    Serial.write(buf);
    sprintf(buf,"In Pin        : %d,%d,%d,%d\n",inPin[0],inPin[1],inPin[2],inPin[3]);
    Serial.write(buf);
    sprintf(buf,"TimerRegister0: %x,%x\n",TCCR0A,TCCR0B);
    Serial.write(buf);
    sprintf(buf,"TimerRegister1: %x,%x\n",TCCR1A,TCCR1B);
    Serial.write(buf);
    sprintf(buf,"TimerRegister3: %x,%x\n",TCCR3A,TCCR3B);
    Serial.write(buf);
    sprintf(buf,"TimerRegister4: %x,%x\n",TCCR4A,TCCR4B);
    Serial.write(buf);
    sprintf(buf,"*******************************************\n");
    Serial.write(buf);
}
