/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      myPB.c
*
* Description:    Use port match interrupt to handle button presses
*
* Revision History:
*
*   01/14/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <si32_device.h>
#include <SI32_PBSTD_A_Type.h>
#include "myCpu.h"
#include "myPB.h"


/* Pressing the S1 button toggles the scale between Celsius and Fahrenheit. */
/* Pressing the S1 button causes the PB1.4 pin to go low.                   */
/* Pressing the S2 button toggles the between temperature and thermistor    */
/* Pressing the S2 button causes the PB1.5 pin to go low.                   */

#define S1   0x0010   /* PB1.4 */
#define S2   0x0020   /* PB1.5 */

/* Scale is 'C' for Celsius or 'F' for Fahrenheit */
char Scale = 'C';

/* If true this get the temperature from the thermistor */
bool Thermistor = false;

/* Tick count of the previous port match interrupt */
uint32_t PreviousMismatch = 0;


//==============================================================================
// 2nd Level Interrupt Handlers (Called from generated code)
//==============================================================================
void PMATCH_handler( void )
{
   uint32_t Pins;
   uint32_t Match;

   /* Debounce the button */
   if ( ElapsedTime(PreviousMismatch) > 10 )
   {
      /* Read PB1 pins */
      Pins = SI32_PBSTD_A_read_pins( SI32_PBSTD_1 );

      /* Read the PB1 match register */
      Match = SI32_PBSTD_A_read_pm( SI32_PBSTD_1 );

      /* If S1 caused the port match interrupt */
      if ( (Pins & S1) == (Match & S1) )
      {
         /* Change the match for S1 */
         SI32_PBSTD_A_write_pm( SI32_PBSTD_1, Match^S1 );

         /* If S1 is pressed */
         if ( (Pins & S1) == 0 )
         {
            /* Toggle the scale between Celsius and Fahrenheit */
            if ( Scale == 'C' )
               Scale = 'F';
            else
               Scale = 'C';  
         }
      }

      /* If S2 caused the port match interrupt */
      if ( (Pins & S2) == (Match & S2) )
      {
         /* Change the match for S2 */
         SI32_PBSTD_A_write_pm( SI32_PBSTD_1, Match^S2 );

         /* If S2 is pressed */
         if ( (Pins & S2) == 0 )
         {
            /* Switch between temperature and thermistor */
            if ( Thermistor )
               Thermistor = false;
            else
               Thermistor = true;   
         }
      }
   }   

   /* Save the tick count of this interrupt */
   PreviousMismatch = get_msTicks();
}

