/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Demo
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      PortMatch.c
*
* Description:    Handle changes in external signals
*
* Revision History:
*
*   09/17/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Tick.h"
#include "I2C.h"
#include "PortMatch.h"


/* Pressing the SW1 button toggles the scale between Celsius and Fahrenheit. */
/* Pressing the SW1 button causes the P0.0 pin to go low.                    */
/* Pressing the SW2 button toggles the between temperature and thermistor    */
/* Pressing the SW2 button causes the P0.1 pin to go low.                    */

#define SW1  0x01
#define SW2  0x02

/* Scale is 'C' for Celsius or 'F' for Fahrenheit */
char Scale;

/* If true this get the temperature from the thermistor */
U8 Thermistor;

/* Tick count of the previous port match interrupt */
U16 PreviousMismatch;


/*****************************************************************************/
/* PortMatch_Init                                                            */
/*****************************************************************************/

void PortMatch_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Initialize the scale to Celsius */
   Scale = 'C';

   /* Initialize thermistor to false */
   Thermistor = FALSE;

   /* There have been no previous mismatches */
   PreviousMismatch = 0;

   /* Generate a port match interrupt when a button is pressed */
   P0MASK = SW1 | SW2;
   P0MAT  = SW1 | SW2;

   /* Enable the Port Match interrupt */
   EIE2 |= 0x02;
}


/*****************************************************************************/
/* PortMatch_ISR                                                             */
/*****************************************************************************/

INTERRUPT( PortMatch_ISR, INTERRUPT_PORT_MATCH )
{
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;
   
   /* Debounce the button */
   if ( ElapsedTime(PreviousMismatch) > 10 )
   {
      /* If SW1 has changed state */
      if ( (SW1&P0) != (SW1&P0MAT) )
      {
         /* Change the match for SW1 */
         P0MAT ^= SW1;
         
         /* If SW1 is pressed */
         if ( (SW1&P0) == 0 )
         {
            /* Toggle the scale between Celsius and Fahrenheit */
            if ( Scale == 'C' )
               Scale = 'F';
            else
               Scale = 'C';  
         }
      }

      /* If SW2 has changed state */
      if ( (SW2&P0) != (SW2&P0MAT) )
      {
         /* Change the match for SW2 */
         P0MAT ^= SW2;
         
         /* If SW2 is pressed */
         if ( (SW2&P0) == 0 )
         {
            /* Switch between temperature and thermistor */
            if ( Thermistor )
               Thermistor = FALSE;
            else
               Thermistor = TRUE;   
         }
      }
   }

   /* Save the tick count of this interrupt */
   PreviousMismatch = TickCount();
   
   SFRPAGE = OldPage;
}

