/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2013 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      main.c
*
* Description:    Main file for the demo application
*
* Revision History:
*
*   02/01/13  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <SI32_PBCFG_A_Type.h>
#include <SI32_PBSTD_A_Type.h>
#include <SI32_WDTIMER_A_Type.h>
#include <SI32_PMU_A_Type.h>
#include <SI32_SCONFIG_A_Type.h>
#include <si32_device.h>
#include "gModes.h"
#include "myCPU.h"
#include "myLCD0.h"
#include "mySensor.h"
#include "myPB.h"
#include "myDisplay.h"
#include "myLog.h"
#include "myPacket.h"
#include "main.h"
#include "myTrace.h"


/* Version */
#define FW_VERSION   0x0102  /* 1.2 */

/* Display definitions */
#define DISPLAY_TEMPERATURE     0
#define DISPLAY_DATE            1
#define DISPLAY_TIME            2
#define DISPLAY_SAMPLE_COUNT    3

/* What to display on the LCD */
int Display = DISPLAY_TEMPERATURE;

/* Alarm definitions */
#define ALARM_HIGH_TEMPERATURE  0x01
#define ALARM_LOW_TEMPERATURE   0x02
#define ALARM_HIGH_HUMIDITY     0x04
#define ALARM_LOW_HUMIDITY      0x08
#define ALARM_COMM_ERROR        0x10

/* Alarms */
uint32_t Alarms = 0;

/* Indicates that a threshold is not used */
#define NO_THRESHOLD  32767

/* Alarm thresholds */
int HighTempThreshold     = NO_THRESHOLD;
int LowTempThreshold      = NO_THRESHOLD;
int HighHumidityThreshold = NO_THRESHOLD;
int LowHumidityThreshold  = NO_THRESHOLD;

/* Awake definitions */
#define AWAKE_TIMEOUT       15   /* seconds */
#define AWAKE_NO_TIMEOUT   (-1)

/* Prevents MCU from sleeping when not zero */
int Awake = 0;

/* Number of seconds since midnight January first 2000 */
uint32_t SystemTime;

/* When the next sample will be taken */
uint32_t SampleTime;

/* Number of seconds between samples */
uint32_t SampleInterval = 120;

/* If Timezone is TIMER_MODE then system time is a timer */
#define TIMER_MODE  16

/* Timezone has not yet been set by the PC */
#define NO_TIMEZONE (-16)

/* Timezone is the offset from UTC (-12 to +12) or TIMER_MODE */
int Timezone = NO_TIMEZONE;

/* Events such as RTC alarm or button pressed */
uint32_t Events = 0;

/* The last measured temperature */
int Temperature = 0;

/* The last measured humidity */
int Humidity = 0;

/* If TRUE then samples are stored in the log */
bool Logging = false;

/* Temperature scale is 'C' for Celsius or 'F' for Fahrenheit */
char Scale = 'C';

/* Packet for communicating with the PC */
PACKET Packet;

/* Output pin for sounding a buzzer */
#define OUTPUT  0x0040    /* PB0.6 */

/* Pause button */
#define S4      0x0080    /* PB1.7 */

/* Forward references */
void     SetConnected( bool Value );
uint16_t GetID( void );


/*****************************************************************************/
/* main                                                                      */
/*****************************************************************************/

int main( void )
{
   int Result;

   /* Initialize the trace buffer */
   TraceReset();

   /* Enable PM8 debugging */
   SI32_SCONFIG_A_enable_power_mode_8_debugging( SI32_SCONFIG_0 );

   /* Enter the default operating mode for this application */
   enter_default_mode_from_reset();
   Sensor_enter_default_mode_from_reset();
 
   /* Initialize the system time to the timestamp of the last sample */
   SystemTime = Log_Init();

   /* Take the first sample right away */
   SampleTime = SystemTime + 1;

   /* If the S4 button is pressed */
   if ( (SI32_PBSTD_A_read_pins(SI32_PBSTD_1) & S4) == 0 )
   {
      myLCD0_put_string("PAUSED");
      
      /* Pause until S4 is released */
      /* This gives the IDE or flash utility a chance to */
      /* get control before the MCU enters sleep mode    */
      while ( (SI32_PBSTD_A_read_pins(SI32_PBSTD_1) & S4) == 0 );
   }
   
   while ( true )
   {
      /* Receive a request packet */
      Result = Packet_Receive( &Packet );
      if ( Result != PACKET_SUCCESS ) continue;

      switch ( Packet.Request )
      {
         case REQ_GET_ID:                      Packet.Parameter = GetID();               break;
         case REQ_GET_VERSION:                 Packet.Parameter = FW_VERSION;            break;                   
         case REQ_GET_STATUS:                  Packet.Parameter = Alarms;                break;                   
         case REQ_GET_TEMPERATURE:             Packet.Parameter = Temperature;           break;                   
         case REQ_GET_HUMIDITY:                Packet.Parameter = Humidity;              break;                   
         case REQ_GET_TIME:                    Packet.Parameter = SystemTime;            break;                   
         case REQ_GET_TIME_ZONE:               Packet.Parameter = Timezone;              break;                   
         case REQ_GET_SAMPLE_INTERVAL:         Packet.Parameter = SampleInterval;        break;                   
         case REQ_GET_HIGH_TEMP_THRESHOLD:     Packet.Parameter = HighTempThreshold;     break;                   
         case REQ_GET_LOW_TEMP_THRESHOLD:      Packet.Parameter = LowTempThreshold;      break;                   
         case REQ_GET_HIGH_HUMIDITY_THRESHOLD: Packet.Parameter = HighHumidityThreshold; break;                   
         case REQ_GET_LOW_HUMIDITY_THRESHOLD:  Packet.Parameter = LowHumidityThreshold;  break;                   
         case REQ_GET_LOGGING:                 Packet.Parameter = Logging;               break;                   
         case REQ_GET_SAMPLE_COUNT:            Packet.Parameter = SampleCount;           break;                   
         case REQ_SET_TIME:                    SystemTime            = Packet.Parameter; break;                   
         case REQ_SET_TIME_ZONE:               Timezone              = Packet.Parameter; break;                   
         case REQ_SET_SAMPLE_INTERVAL:         SampleInterval        = Packet.Parameter; break;                   
         case REQ_SET_HIGH_TEMP_THRESHOLD:     HighTempThreshold     = Packet.Parameter; break;                   
         case REQ_SET_LOW_TEMP_THRESHOLD:      LowTempThreshold      = Packet.Parameter; break;                   
         case REQ_SET_HIGH_HUMIDITY_THRESHOLD: HighHumidityThreshold = Packet.Parameter; break;                   
         case REQ_SET_LOW_HUMIDITY_THRESHOLD:  LowHumidityThreshold  = Packet.Parameter; break;                   
         case REQ_SET_LOGGING:                 Logging               = Packet.Parameter; break;                   
         case REQ_GET_LOG:                     Packet_SendLog(&Packet); continue;        break;                   
         case REQ_ERASE_LOG:                   Log_Erase();                              break; 
         case REQ_SET_CONNECTED:               SetConnected(Packet.Parameter);           break;                  
      }

      /* If the system time or the sample interval is changed then update the sample time */
      if ( (Packet.Request==REQ_SET_TIME) || (Packet.Request==REQ_SET_SAMPLE_INTERVAL) )
         SampleTime = SystemTime + SampleInterval;

      /* If logging is changed and displaying the sample count then update the LCD */
      if ( ((Packet.Request==REQ_SET_LOGGING) || (Packet.Request==REQ_ERASE_LOG)) && 
           (Display==DISPLAY_SAMPLE_COUNT) )
         Events |= LCD_EVENT;

      /* Send back a response packet */
      Packet_Send( &Packet );
   }
}


/*****************************************************************************/
/* SetConnected                                                              */
/*****************************************************************************/

void SetConnected( bool Connected )
{
   /* If the GUI is connected */
   if ( Connected )
   {
      /* Turn on the DS8 LED to show connected */
      SI32_PBSTD_A_write_pins_low( SI32_PBSTD_1, DS8 );

      /* Prevent the MCU from sleeping while connected */
      Awake = AWAKE_NO_TIMEOUT;
   }
   else /* GUI is not connected */
   {
      /* Turn off the DS8 LED to to show not connected */
      SI32_PBSTD_A_write_pins_high( SI32_PBSTD_1, DS8 );

      /* Allow the MCU to sleep */
      Awake = 0;
   }

   /* Update the LCD */
   Events |= LCD_EVENT;
}


/*****************************************************************************/
/* GetID                                                                     */
/*****************************************************************************/

uint16_t GetID( void )
{
   uint16_t ID;

   /* Put 167 in the high byte to mean SiM3L167 */
   /* Currently the DEVICEID register cannot be trusted */
   ID = (167) << 8;

   /* Wake up the sensor */
   Sensor_WakeUp( I2C0_BUS_1 );

   /* Put the sensor ID in the lower byte */
   ID |= Sensor_ID( I2C0_BUS_1 );

   /* Put the sensor back to sleep */
   Sensor_Sleep( I2C0_BUS_1 );
   
   return ID;
}


/*****************************************************************************/
/* CheckThresholds                                                           */
/*****************************************************************************/

void CheckThresholds()
{
   /* If there is a high temperature threshold */
   if ( HighTempThreshold != NO_THRESHOLD )
   {
      /* Check for high temperature */
      if ( Temperature > HighTempThreshold )
         Alarms |=  ALARM_HIGH_TEMPERATURE;
      else
         Alarms &= ~ALARM_HIGH_TEMPERATURE;
   }
   else Alarms &= ~ALARM_HIGH_TEMPERATURE;

   /* If there is a low temperature threshold */
   if ( LowTempThreshold != NO_THRESHOLD )
   {
      /* Check for low temperature */
      if ( Temperature < LowTempThreshold )
         Alarms |=  ALARM_LOW_TEMPERATURE;
      else
         Alarms &= ~ALARM_LOW_TEMPERATURE;
   }
   else Alarms &= ~ALARM_LOW_TEMPERATURE;
   
   /* If there is a high humidity threshold */
   if ( HighHumidityThreshold != NO_THRESHOLD )
   {
      /* Check for high humidity */
      if ( Humidity > HighHumidityThreshold )
         Alarms |=  ALARM_HIGH_HUMIDITY;
      else
         Alarms &= ~ALARM_HIGH_HUMIDITY;
   }
   else Alarms &= ~ALARM_HIGH_HUMIDITY;

   /* If there is a low humidity threshold */
   if ( LowHumidityThreshold != NO_THRESHOLD )
   {
      /* Check for low humidity */
      if ( Humidity < LowHumidityThreshold )
         Alarms |=  ALARM_LOW_HUMIDITY;
      else
         Alarms &= ~ALARM_LOW_HUMIDITY;
   }
   else Alarms &= ~ALARM_LOW_HUMIDITY;
}


/*****************************************************************************/
/* TakeSample                                                                */
/*****************************************************************************/

void TakeSample()
{
   int Result;
   
   /* Blink DS8 to show that a sample is being taken */
   SI32_PBSTD_A_toggle_pins( SI32_PBSTD_1, DS8 );

   /* Wake up the sensor */
   Sensor_WakeUp( I2C0_BUS_1 );
     
   /* Read the temperature and humidity from the sensor */
   Result = Sensor_ReadTemperature( I2C0_BUS_1, &Temperature );
   if ( Result == I2C0_STATUS_SUCCESS )
      Result = Sensor_ReadHumidity( I2C0_BUS_1, Temperature, &Humidity );

   /* Put the sensor back to sleep */
   Sensor_Sleep( I2C0_BUS_1 );

   /* If successfully read the temperature and humidity */
   if ( Result == I2C0_STATUS_SUCCESS )
   {
      /* If logging is enabled */
      if ( Logging )
      {
         /* Write the sample to the log */
         Log_WriteSample( SystemTime, Temperature, Humidity );

         /* If displaying the sample count then update the LCD */
         if ( Display == DISPLAY_SAMPLE_COUNT )
            Events |= LCD_EVENT;
      } 
      
      /* Check if temperature and humidity are outside the thresholds */
      CheckThresholds(); 

      /* Clear any previous communications error */
      Alarms &= ~ALARM_COMM_ERROR;
   }
   else Alarms |= ALARM_COMM_ERROR;

   /* If there are alarms */
   if ( Alarms )
   {
      /* Update the LCD and pulse the output pin */
      Events |= (LCD_EVENT|OUT_EVENT);

      /* Wakeup to show alarms */
      if ( Awake >= 0 )
         Awake = AWAKE_TIMEOUT;
   }   
   else if ( Display == DISPLAY_TEMPERATURE )
   {
      /* Update the LCD */
      Events |= LCD_EVENT;
   }   

   /* Stop blinking DS8 */
   SI32_PBSTD_A_toggle_pins( SI32_PBSTD_1, DS8 );
}


/*****************************************************************************/
/* ProcessRTCEvent                                                           */
/*****************************************************************************/

void ProcessRTCEvent()
{
   /* Increment the system time */
   SystemTime++;

   /* Decrement the awake timeout */
   if ( Awake > 0 )
   {
      Awake--;

      /* If awake timed out then clear the LCD */
      if ( Awake == 0 )
         Events |= LCD_EVENT;
   }   
      
   /* If it's time to take another sample */
   if ( SystemTime == SampleTime )
   { 
      /* Calculate the time of the next sample */
      SampleTime += SampleInterval;

      /* Take the sample */
      TakeSample();
   }

   /* If displaying time or date or alarms then update the LCD */
   if ( Display==DISPLAY_TIME || Display==DISPLAY_DATE || Alarms )
      Events |= LCD_EVENT;
}


/*****************************************************************************/
/* ProcessJoyEvent                                                           */
/*****************************************************************************/

void ProcessJoyEvent()
{
   /* If not awake then wake up */
   if ( Awake == 0 )
      Awake = AWAKE_TIMEOUT;        
   /* If awake then go to sleep */
   else if ( Awake > 0 )
      Awake = 0;   

   /* Update the LCD */   
   Events |= LCD_EVENT;   
}
   

/*****************************************************************************/
/* ProcessS1Event                                                            */
/*****************************************************************************/

void ProcessS1Event()
{
   /* If asleep then ignore the S1 button */
   if ( Awake )
   {
      /* Restart the awake timeout */   
      if ( Awake > 0 )
         Awake = AWAKE_TIMEOUT;        

      /* If there are no alarms */
      if ( !Alarms )
      { 
         /* Change to the next display with wrap around */
         Display++;
         if ( Display > DISPLAY_SAMPLE_COUNT )
            Display = DISPLAY_TEMPERATURE;
      }
      /* Clear alarms so other info can be displayed */   
      else Alarms = 0;   

      /* Update the LCD */   
      Events |= LCD_EVENT;   
   }
}
   

/*****************************************************************************/
/* ProcessS2Event                                                            */
/*****************************************************************************/

void ProcessS2Event()
{
   /* If asleep then ignore the S2 button */
   if ( Awake )
   {
      /* Restart the awake timeout */   
      if ( Awake > 0 )
         Awake = AWAKE_TIMEOUT;        
         
      /* Toggle the scale between Celsius and Fahrenheit */
      if ( Scale == 'C' )
         Scale = 'F';
      else
         Scale = 'C';  

      /* If displaying the temperature then update the LCD */
      if ( Display == DISPLAY_TEMPERATURE )
         Events |= LCD_EVENT;   
   }   
}
   

/*****************************************************************************/
/* ProcessS3Event                                                            */
/*****************************************************************************/

void ProcessS3Event()
{
   /* If asleep then ignore the S3 button */
   if ( Awake )
   {
      /* Restart the awake timeout */   
      if ( Awake > 0 )
         Awake = AWAKE_TIMEOUT;        
         
      /* Start and stop logging */
      if ( Logging )
         Logging = false;
      else
         Logging = true; 
         
      /* If displaying the sample count then update the LCD */
      if ( Display == DISPLAY_SAMPLE_COUNT )
         Events |= LCD_EVENT;
   }   
}


/*****************************************************************************/
/* ProcessLCDEvent                                                           */
/*****************************************************************************/

void ProcessLCDEvent()
{
   /* Clear the display */
   myLCD0_put_char('\r');
   myLCD0_put_char('\n');

   /* If asleep then show a blank LCD */
   if ( Awake )
   {
      /* If there are alarms */
      if ( Alarms )
      {
         /* If the MCU failed to communicate with the sensor */
         if ( Alarms & ALARM_COMM_ERROR )
         {
            myLCD0_put_string("COMM ERR");
         }
         else /* Communications with the sensor is good */
         {
            /* Display humidity alarm on odd seconds */
            if ( SystemTime & 1 )
            {
               if ( Alarms & ALARM_HIGH_HUMIDITY )
               {
                  Display_Humidity( Humidity );
                  myLCD0_put_string(" HI");
               }
               else if ( Alarms & ALARM_LOW_HUMIDITY )
               {
                  Display_Humidity( Humidity );
                  myLCD0_put_string(" LOW");
               }
            }
            else /* Display temperature alarm on even seconds */
            {
               if ( Alarms & ALARM_HIGH_TEMPERATURE )
               {
                  Display_Temperature( Temperature, Scale );
                  myLCD0_put_string(" HI");
               }
               else if ( Alarms & ALARM_LOW_TEMPERATURE )
               {
                  Display_Temperature( Temperature, Scale );
                  myLCD0_put_string(" LOW");
               }
            }
         }
      }
      else /* There are no alarms */
      {
         switch ( Display )
         {
            case DISPLAY_TEMPERATURE:
               Display_Temperature( Temperature, Scale );
               Display_Humidity( Humidity );
               break;
            case DISPLAY_DATE:
               if ( Timezone == TIMER_MODE )
                  Display_Days( SystemTime );
               else   
                  Display_Date( SystemTime );
               break;
            case DISPLAY_TIME:
               if ( Timezone == TIMER_MODE )
                  Display_Time( SystemTime, 24 );
               else
                  Display_Time( SystemTime, 12 );
               break;
            case DISPLAY_SAMPLE_COUNT:
               Display_Count( SampleCount, Logging );
               break;
         }
      }
   }
}   


/*****************************************************************************/
/* ProcessOutEvent                                                           */
/*****************************************************************************/

void ProcessOutEvent( void )
{
   /* Generate a 500 ms negative pulse on the output pin */
   SI32_PBSTD_A_write_pins_low( SI32_PBSTD_0, OUTPUT );
   Delay( 500 );
   SI32_PBSTD_A_write_pins_high( SI32_PBSTD_0, OUTPUT );
}   


/*****************************************************************************/
/* ProcessEvents                                                             */
/*****************************************************************************/

void ProcessEvents( void )
{
   /* Process each event */
   if ( Events & RTC_EVENT ) { Events &= ~RTC_EVENT; ProcessRTCEvent(); }
   if ( Events & JOY_EVENT ) { Events &= ~JOY_EVENT; ProcessJoyEvent(); }
   if ( Events & S1_EVENT  ) { Events &= ~S1_EVENT;  ProcessS1Event();  }
   if ( Events & S2_EVENT  ) { Events &= ~S2_EVENT;  ProcessS2Event();  }
   if ( Events & S3_EVENT  ) { Events &= ~S3_EVENT;  ProcessS3Event();  }
   if ( Events & LCD_EVENT ) { Events &= ~LCD_EVENT; ProcessLCDEvent(); }
   if ( Events & OUT_EVENT ) { Events &= ~OUT_EVENT; ProcessOutEvent(); }   

   /* If all events have been processed and the MCU is allowed to sleep */
   if ( !Events && !Awake )
   {
      /* Disable the SysTick timer */
      SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

      /* Set the SLEEPDEEP bit in the core */
      SCB->SCR = SCB_SCR_SLEEPDEEP_Msk;

      /* Enter PM8 mode */
      __DSB();
      __ISB();
      __WFI();

      /* Wakeup from PM8 */

      /* Disable the watchdog timer */
      SI32_WDTIMER_A_stop_counter( SI32_WDTIMER_0 );

      /* Clear the wakeup flags */
      SI32_PMU_A_clear_wakeup_flags( SI32_PMU_0 );

      /* Re-enable the SysTick timer */
      SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
   }
}

