/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Tick.c
*
* Description:    Tick counter is used to implement timeouts and delays
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Tick.h"


/* TickCounter increments every 1 ms and wraps around every 65 seconds. */
/* TickCounter is used to implement timeouts and delays.                */
/* TickCounter does not increment while the MCU is in low power mode.   */

U16 TickCounter;


/*****************************************************************************/
/* Tick_Init                                                                 */
/*****************************************************************************/

void Tick_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Initialize the tick counter to zero */
   TickCounter = 0;

   /* Timer 2 uses SYSCLK */
   CKCON |= 0x10;

   /* Timer 2 generates periodic interrupts every 1 ms */
   TMR2RLH = 0xB1;
   TMR2RLL = 0xE0;
   
   /* Enable timer 2 */
   TMR2CN = 0x04;

   /* Enable timer 2 interrupt */
   ET2 = 1;
}


/*****************************************************************************/
/* Timer2_ISR                                                                */
/*****************************************************************************/

INTERRUPT( Timer2_ISR, INTERRUPT_TIMER2 )
{
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;

   /* Clear the timer 2 interrupt */
   TF2H = 0;

   /* Increment the tick counter */
   TickCounter++;
   
   SFRPAGE = OldPage;
}


/*****************************************************************************/
/* TickCount                                                                 */
/*****************************************************************************/

U16 TickCount( void )
{
   U16 Count;

   ET2 = 0;
   Count = TickCounter;
   ET2 = 1;
   
   return Count;
}


/*****************************************************************************/
/* ElapsedTime                                                               */
/*****************************************************************************/

U16 ElapsedTime( U16 StartTime )
{
   U16 CurrentTime = TickCount();
   
   if ( CurrentTime >= StartTime )
      return CurrentTime - StartTime;
   else
      return (0xFFFF - StartTime) + CurrentTime + 1;
}


/*****************************************************************************/
/* Delay                                                                     */
/*****************************************************************************/

void Delay( U16 Milliseconds )
{
   U16 StartTime = TickCount();
   
   while ( ElapsedTime(StartTime) < Milliseconds );
}

