/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      RTC.c
*
* Description:    Generate an RTC alarm once a second
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "Tick.h"
#include "RTC.h"


/* Number of RTC clocks in 1 second */
#define ALARM_TIME  32766

/* RTC0CN register */
#define RTC0CAP   0x01
#define RTC0SET   0x02
#define RTC0FAST  0x04
#define RTC0TR    0x10
#define OSCFAIL   0x20
#define MCLKEN    0x40
#define RTC0EN    0x80

/* RTC0CF register */
#define ALRM0EN   0x01
#define ALRM1EN   0x02
#define ALRM2EN   0x04
#define AUTORST   0x08
#define ALRM0     0x10
#define ALRM1     0x20
#define ALRM2     0x40

/* RTC0XCN register */
#define LFOEN     0x08
#define CLKVLD    0x10
#define BIASX2    0x20
#define XMODE     0x40
#define AGCEN     0x80

/* RTC0XCF register */
#define LOADCAP   0x0F
#define LOADRDY   0x40
#define AUTOSTP   0x80


/*****************************************************************************/
/* RTC_ Init                                                                 */
/*****************************************************************************/

void RTC_Init( void )
{
   SFRPAGE = LEGACY_PAGE;

   /* Set crystal pins to analog mode */
   P1MDIN = 0xF3;

   /* Set RTC to crystal mode with double bias */
   RTC_Write( RTC0XCN, XMODE|BIASX2 );

   /* Set the load capacitance to 7 PF */
   RTC_Write( RTC0XCF, AUTOSTP|0x06 );

   /* Enable RTC */
   RTC_Write( RTC0CN, RTC0EN );
   
   Delay( 20 );

   /* Wait for the crystal oscillation to be valid */
   while ( !(RTC_Read(RTC0XCN)&CLKVLD) ); 

   /* Wait for the load capacitance to be ready */
   while ( !(RTC_Read(RTC0XCF)&LOADRDY) ); 

   /* Enable automatic gain control */
   RTC_Write( RTC0XCN, XMODE|AGCEN );

   /* Configure RTC alarm 0 for 1 second with auto reset */
   RTC_Write( RTC0CF, 0x00 );
   RTC_WriteAlarm( 0, ALARM_TIME );
   RTC_Write( RTC0CF, ALRM0EN|AUTORST );

   /* Start the RTC timer */
   RTC_Write( RTC0CN, RTC0EN|RTC0FAST|RTC0TR );

   /* Enable the RTC alarm interrupt */
   EIE1 |= 0x02;
}


/*****************************************************************************/
/* RTC_Write                                                                 */
/*****************************************************************************/

void RTC_Write( U8 Register, U8 Data )
{
   RTC0ADR = Register;
   RTC0DAT = Data;
}


/*****************************************************************************/
/* RTC_Read                                                                  */
/*****************************************************************************/

U8 RTC_Read( U8 Register )
{
   RTC0ADR = Register;
   return RTC0DAT;
}


/*****************************************************************************/
/* RTC_WriteAlarm                                                            */
/*****************************************************************************/

void RTC_WriteAlarm( U8 Alarm, U32 Time )
{
   UU32 Value;

   Value.U32 = Time;

   RTC0ADR = ALARM0B0 + Alarm*4;
   RTC0DAT = Value.U8[b0];
   RTC0DAT = Value.U8[b1];
   RTC0DAT = Value.U8[b2];
   RTC0DAT = Value.U8[b3];
}


/*****************************************************************************/
/* RTC_ReadTimer                                                             */
/*****************************************************************************/

U32 RTC_ReadTimer( void )
{
   UU32 Timer;

   /* Capture the timer */
   RTC_Write( RTC0CN, RTC0EN|RTC0FAST|RTC0TR|RTC0CAP );

   /* Wait till the capture is done */
   while ( RTC_Read(RTC0CN)&RTC0CAP );

   /* Read the captured timer value */
   RTC0ADR = 0x40|CAPTURE0;
   Timer.U8[b0]= RTC0DAT;
   Timer.U8[b1]= RTC0DAT;
   Timer.U8[b2]= RTC0DAT;
   Timer.U8[b3]= RTC0DAT;
   
   return Timer.U32;
}


/*****************************************************************************/
/* RTC_ElapsedTime                                                           */
/*****************************************************************************/

U32 RTC_ElapsedTime( VARIABLE_SEGMENT_POINTER(pStartTime,U32,SEG_PDATA), 
                     VARIABLE_SEGMENT_POINTER(  pEndTime,U32,SEG_PDATA)  )
{
   if ( *pEndTime >= *pStartTime )
      return *pEndTime - *pStartTime;
   else
      return (ALARM_TIME - *pStartTime) + *pEndTime + 1;
}

   
/*****************************************************************************/
/* RTC_ISR                                                                   */
/*****************************************************************************/

INTERRUPT( RTC_ISR, INTERRUPT_RTC0ALARM )
{
   U8 OldPage = SFRPAGE;
   
   SFRPAGE = LEGACY_PAGE;

   /* Clear the RTC alarm 0 event flag */
   RTC_Write( RTC0CF, 0x00 );
   RTC_Write( RTC0CF, ALRM0EN|AUTORST );

   /* Set the RTC event */
   Events |= RTC_EVENT;

   SFRPAGE = OldPage;
}

