/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Packet.c
*
* Description:    Send and receive packets using the Data Logger Protocol
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "Main.h"
#include "Tick.h"
#include "UART.h"
#include "Log.h"
#include "Packet.h"


/* Start-of-Packet character */
#define SOP  0xAC


/*****************************************************************************/
/* Packet_Receive                                                            */
/*****************************************************************************/

S8 Packet_Receive( VARIABLE_SEGMENT_POINTER(pPacket,PACKET,SEG_IDATA) )
{
   U8 *pByte;
   U8 *pLimit;
   U8  Count;
   U8  Sum;
   
   /* Clear the start field */
   pPacket->Start = 0;

   /* Receive until a Start-Of-Packet character is found */   
   while ( pPacket->Start != SOP )
   {
      /* Don't timeout when trying to receive the SOP */
      Count = UART_Receive( &pPacket->Start, 1, FALSE );
      if ( Count == 0 ) return FAILURE;
   } 

   /* Receive the Request and DataLength fields */
   for ( pByte=&pPacket->Request,pLimit=pByte+3; pByte<pLimit; pByte+=Count )
   {
      Count = UART_Receive( pByte, pLimit-pByte, TRUE );
      if ( Count == 0 ) return FAILURE;
   }

   /* Receive the Data and Checksum fields */
   for ( pByte=pLimit,pLimit+=pPacket->DataLength+1; pByte<pLimit; pByte+=Count )
   {
      Count = UART_Receive( pByte, pLimit-pByte, TRUE );
      if ( Count == 0 ) return FAILURE;
   }

   /* Calculate the checksum */
   for ( pByte=&pPacket->Request,Sum=0; pByte<pLimit; pByte++ )
      Sum += *pByte;      

   /* Validate the checksum */
   if ( Sum == 0 )
      return SUCCESS;
   else
      return FAILURE;   
}


/*****************************************************************************/
/* Packet_Send                                                               */
/*****************************************************************************/

S8 Packet_Send( VARIABLE_SEGMENT_POINTER(pPacket,PACKET,SEG_IDATA) )
{
   U8 *pByte;
   U8  Sum;
   
   /* Calculate the checksum */      
   for ( pByte=&pPacket->Request,Sum=0; pByte<&pPacket->Checksum; pByte++ )
      Sum += *pByte;      

   /* Store the checksum in the packet */
   pPacket->Checksum = (~Sum) + 1;

   /* Send the packet */   
   UART_Send( (U8*)pPacket, pPacket->DataLength+5 );

   return SUCCESS;
}


/*****************************************************************************/
/* Packet_SendLog                                                            */
/*****************************************************************************/

S8 Packet_SendLog( VARIABLE_SEGMENT_POINTER(pPacket,PACKET,SEG_IDATA) )
{
   VARIABLE_SEGMENT_POINTER( pData,      U8, SEG_CODE );
   VARIABLE_SEGMENT_POINTER( pDataLimit, U8, SEG_CODE );
   U8 *pByte;
   U8  Sum;

   /* Convert xdata pointer to a code pointer */
   pDataLimit = (U8 SEG_CODE *)pLog;
   
   /* Store the size of the log in the packet */
   if ( SampleCount == MAX_SAMPLE_COUNT )
      pPacket->DataLength = 65535;
   else   
      pPacket->DataLength = SampleCount * SAMPLE_SIZE;

   /* Calculate the checksum across the header */      
   for ( pByte=&pPacket->Request,Sum=0; pByte<&pPacket->Parameter; pByte++ )
      Sum += *pByte;      
      
   /* If the log is currently in bank 2 */
   if ( PSBANK == BANK_2 )
   {
      PSBANK = BANK_1;

      /* Calculate the checksum across bank 1 */
      for ( pData=BANK_ADDRESS; pData>0; pData++ )
         Sum += *pData;      

      PSBANK = BANK_2;
   }

   /* If the log is full */
   if ( pDataLimit == 0 )
   {
      /* Calculate the checksum across bank 2 */
      for ( pData=BANK_ADDRESS; pData>0; pData++ )
         Sum += *pData;      
   }
   else /* The log is not full */
   {
      /* Calculate the checkum across the written part of the current bank */
      for ( pData=BANK_ADDRESS; pData<pDataLimit; pData++ )
         Sum += *pData;      
   }   

   /* Store the checksum in the packet */
   pPacket->Checksum = (~Sum) + 1;

   /* Send the packet header */
   UART_Send( (U8*)pPacket, 4 );

   /* Point to the beginning of log */
   pData = BANK_ADDRESS;

   /* If the log is currently in bank 2 */
   if ( PSBANK == BANK_2 )
   {
      PSBANK = BANK_1;

      /* Send all of bank 1 */
      UART_Send( pData, BANK_SIZE );

      PSBANK = BANK_2;
   }

   /* If the log is full */
   if ( pDataLimit == 0 )
   {
      /* Send all of bank 2 */
      UART_Send( pData, BANK_SIZE );
   }
   else /* The log is not full */
   {
      /* Send the written part of the current bank */
      UART_Send( pData, pDataLimit-pData );
   }   

   /* Send the checksum */
   UART_Send( &pPacket->Checksum, 1 );

   return SUCCESS;
}

