/*-----------------------------------------------------------------------------
*
* Project:        Silicon Labs Si7005 UDP Data Logger
*
* Copyright:      2012 Silicon Labs, Inc. (www.silabs.com)
*
* File Name:      Main.c
*
* Description:    Read temperature and humidity from the sensor device
*                 and store them in flash memory.
*
* Revision History:
*
*   10/08/12  QHS  Initial Release
*
*----------------------------------------------------------------------------*/

#include <compiler_defs.h>
#include <C8051F960_defs.h>
#include "lcdPutString.h"
#include "lcdConfig.h"
#include "lcdPutNumber.h"
#include "RTC.h"
#include "Main.h"
#include "Log.h"
#include "Tick.h"
#include "I2C.h"
#include "UART.h"
#include "Packet.h"
#include "Sensor.h"
#include "PortMatch.h"


/* Version */
#define FW_VERSION   0x0103  /* 1.3 */

/* Display definitions */
#define DISPLAY_TEMPERATURE     0
#define DISPLAY_DATE            1
#define DISPLAY_TIME            2
#define DISPLAY_SAMPLE_COUNT    3

/* What to display on the LCD */
U8 Display = DISPLAY_TEMPERATURE;

/* Alarm definitions */
#define ALARM_HIGH_TEMPERATURE  0x01
#define ALARM_LOW_TEMPERATURE   0x02
#define ALARM_HIGH_HUMIDITY     0x04
#define ALARM_LOW_HUMIDITY      0x08
#define ALARM_COMM_ERROR        0x10

/* Alarms */
U8 Alarms = 0;

/* The threshold parameter is not used */
#define NO_THRESHOLD  32767

///* Alarm thresholds */
S16 HighTempThreshold     = NO_THRESHOLD;
S16 LowTempThreshold      = NO_THRESHOLD;
S16 HighHumidityThreshold = NO_THRESHOLD;
S16 LowHumidityThreshold  = NO_THRESHOLD;

/* Power Management Configuration (PMU0CF) register   */
#define CPT0WK   0x01  /* Comparator wake-up          */
#define PMATWK   0x02  /* Port Match wake-up          */
#define RTCAWK   0x04  /* RTC Alarm wake-up           */
#define RTCFWK   0x08  /* RTC Fail wake-up            */
#define RSTWK    0x10  /* Reset wake-up               */
#define CLEAR    0x20  /* Clear the wake-up flags     */
#define SUSPEND  0x40  /* Put the MCU in suspend mode */
#define SLEEP    0x80  /* Put the MCU in sleep mode   */

/* Number of seconds since midnight January first 2000 */
U32 SystemTime;

/* When the next sample will be taken */
U32 SampleTime;

/* Number of seconds between samples */
U16 SampleInterval = 120;

/* If Timezone is TIMER_MODE then system time is a timer */
#define TIMER_MODE  16

/* Timezone has not yet been set by the PC */
#define NO_TIMEZONE (-16)

/* Timezone is the offset from UTC (-12 to +12) or TIMER_MODE */
S8 Timezone = NO_TIMEZONE;

/* Events such as RTC alarm or button pressed */
U8 Events = 0;

/* The last measured temperature */
S16 Temperature = 0;

/* The last measured humidity */
S16 Humidity = 0;

/* If TRUE then samples are stored in the log */
U8 Logging = FALSE;

/* Temperature scale is 'C' for Celsius or 'F' for Fahrenheit */
char Scale = 'C';

/* Packet for communicating with the PC */
SEGMENT_VARIABLE( Packet, PACKET, SEG_IDATA );

/* Output pin for sounding a buzzer */
SBIT( OUTPUT, SFR_P1, 0 );

/* Forward references */
U16 GetID( void );


/*****************************************************************************/
/* main                                                                      */
/*****************************************************************************/

void main (void)
{
   S8 Result;
   
   SFRPAGE = LEGACY_PAGE;

   /* Disable the watchdog timer */
   PCA0MD &= ~0x40;

   /* Use the internal low power oscillator (SYSCLK = 20 MHz) */   
   CLKSEL = 0x04;

   /* Configure P1.0 as a high-drive output pin to sound a buzzer */
   P1MDOUT = 0x01;
   P1DRV   = 0x01;
   OUTPUT  = 1;

   /* Clear the wake-up flags */
   PMU0CF = CLEAR;

   /* Configure the LCD */
   LCD0_Config( 4, 32, 1, 2900, 60, 1 );

   /* Initialize the other components */
   Tick_Init();
   I2C_Init();
   UART_Init();
   Sensor_Init();
   PortMatch_Init();

   /* Initialize the system time to the timestamp of the last sample */
   SystemTime = Log_Init();

   /* Take the first sample right away */
   SampleTime = SystemTime + 1;

   /* Enable interrupts */
   EA = 1;

   /* RTC must be initialized after interrupts are enabled */
   RTC_Init();

   /* If the SW4 button is pressed */
   if ( (P0 & SW4_EVENT) == 0 )
   {
      lcdPutString("PAUSED");
      
      /* Pause until SW4 is released */
      /* This gives the IDE or flash utility a chance to */
      /* get control before the MCU enters sleep mode    */
      while ( (P0 & SW4_EVENT) == 0 );
   }

   while ( TRUE )
   {
      /* Receive a request packet */
      Result = Packet_Receive( &Packet );
      if ( Result != SUCCESS ) continue;

      switch ( Packet.Request )
      {
         case REQ_GET_ID:                      Packet.Parameter = GetID();               break;
         case REQ_GET_VERSION:                 Packet.Parameter = FW_VERSION;            break;                   
         case REQ_GET_STATUS:                  Packet.Parameter = Alarms;                break;                   
         case REQ_GET_TEMPERATURE:             Packet.Parameter = Temperature;           break;                   
         case REQ_GET_HUMIDITY:                Packet.Parameter = Humidity;              break;                   
         case REQ_GET_TIME:                    Packet.Parameter = SystemTime;            break;                   
         case REQ_GET_TIME_ZONE:               Packet.Parameter = Timezone;              break;                   
         case REQ_GET_SAMPLE_INTERVAL:         Packet.Parameter = SampleInterval;        break;                   
         case REQ_GET_HIGH_TEMP_THRESHOLD:     Packet.Parameter = HighTempThreshold;     break;                   
         case REQ_GET_LOW_TEMP_THRESHOLD:      Packet.Parameter = LowTempThreshold;      break;                   
         case REQ_GET_HIGH_HUMIDITY_THRESHOLD: Packet.Parameter = HighHumidityThreshold; break;                   
         case REQ_GET_LOW_HUMIDITY_THRESHOLD:  Packet.Parameter = LowHumidityThreshold;  break;                   
         case REQ_GET_LOGGING:                 Packet.Parameter = Logging;               break;                   
         case REQ_GET_SAMPLE_COUNT:            Packet.Parameter = SampleCount;           break;                   
         case REQ_SET_TIME:                    SystemTime            = Packet.Parameter; break;                   
         case REQ_SET_TIME_ZONE:               Timezone              = Packet.Parameter; break;                   
         case REQ_SET_SAMPLE_INTERVAL:         SampleInterval        = Packet.Parameter; break;                   
         case REQ_SET_HIGH_TEMP_THRESHOLD:     HighTempThreshold     = Packet.Parameter; break;                   
         case REQ_SET_LOW_TEMP_THRESHOLD:      LowTempThreshold      = Packet.Parameter; break;                   
         case REQ_SET_HIGH_HUMIDITY_THRESHOLD: HighHumidityThreshold = Packet.Parameter; break;                   
         case REQ_SET_LOW_HUMIDITY_THRESHOLD:  LowHumidityThreshold  = Packet.Parameter; break;                   
         case REQ_SET_LOGGING:                 Logging               = Packet.Parameter; break;                   
         case REQ_GET_LOG:                     Packet_SendLog(&Packet); continue;        break;                   
         case REQ_ERASE_LOG:                   Log_Erase();                              break;                   
         case REQ_SET_CONNECTED:               /* Ignore */                              break;                  
      }

      /* If the system time or the sample interval is changed then update the sample time */
      if ( (Packet.Request==REQ_SET_TIME) || (Packet.Request==REQ_SET_SAMPLE_INTERVAL) )
         SampleTime = SystemTime + SampleInterval;

      /* If logging is changed and displaying the sample count then update the LCD */
      if ( ((Packet.Request==REQ_SET_LOGGING) || (Packet.Request==REQ_ERASE_LOG)) && 
           (Display==DISPLAY_SAMPLE_COUNT) )
         Events |= LCD_EVENT;

      /* Send back a response packet */
      Packet_Send( &Packet );
   }
}


/*****************************************************************************/
/* GetID                                                                     */
/*****************************************************************************/

U16 GetID( void )
{
   U16 ID;

   SFRPAGE = CONFIG_PAGE;

   /* Put the MCU ID in the upper byte */
   ID = DEVICEID << 8;

   SFRPAGE = LEGACY_PAGE;

   /* Wake up the sensor */
   Sensor_WakeUp( I2C_BUS_1 );

   /* Put the sensor ID in the lower byte */
   ID |= Sensor_ID( I2C_BUS_1 );

   /* Put the sensor back to sleep */
   Sensor_Sleep( I2C_BUS_1 );
   
   return ID;
}


/*****************************************************************************/
/* TakeSample                                                                */
/*****************************************************************************/

void TakeSample()
{
   S8 Result;
   
   /* Blink LED4 to show that a sample is being taken */
   LED4 ^= 1;

   /* Wake up the sensor */
   Sensor_WakeUp( I2C_BUS_1 );
     
   /* Read the temperature and humidity from the sensor */
   Result = Sensor_ReadTemperature( I2C_BUS_1, &Temperature );
   if ( Result == SUCCESS )
      Result = Sensor_ReadHumidity( I2C_BUS_1, Temperature, &Humidity );

   /* Put the sensor back to sleep */
   Sensor_Sleep( I2C_BUS_1 );

   /* If successfully read the temperature and humidity */
   if ( Result == SUCCESS )
   {
      /* If logging is enabled */
      if ( Logging )
      {
         /* Write the sample to the log */
         Log_WriteSample( SystemTime, Temperature, Humidity );

         /* If displaying the sample count then update the LCD */
         if ( Display == DISPLAY_SAMPLE_COUNT )
            Events |= LCD_EVENT;
      }  
      
      /* If there is a high temperature threshold */
      if ( HighTempThreshold != NO_THRESHOLD )
      {
         /* Check for high temperature */
         if ( Temperature > HighTempThreshold )
            Alarms |=  ALARM_HIGH_TEMPERATURE;
         else
            Alarms &= ~ALARM_HIGH_TEMPERATURE;
      }
      else Alarms &= ~ALARM_HIGH_TEMPERATURE;

      /* If there is a low temperature threshold */
      if ( LowTempThreshold != NO_THRESHOLD )
      {
         /* Check for low temperature */
         if ( Temperature < LowTempThreshold )
            Alarms |=  ALARM_LOW_TEMPERATURE;
         else
            Alarms &= ~ALARM_LOW_TEMPERATURE;
      }
      else Alarms &= ~ALARM_LOW_TEMPERATURE;
      
      /* If there is a high humidity threshold */
      if ( HighHumidityThreshold != NO_THRESHOLD )
      {
         /* Check for high humidity */
         if ( Humidity > HighHumidityThreshold )
            Alarms |=  ALARM_HIGH_HUMIDITY;
         else
            Alarms &= ~ALARM_HIGH_HUMIDITY;
      }
      else Alarms &= ~ALARM_HIGH_HUMIDITY;

      /* If there is a low humidity threshold */
      if ( LowHumidityThreshold != NO_THRESHOLD )
      {
         /* Check for low humidity */
         if ( Humidity < LowHumidityThreshold )
            Alarms |=  ALARM_LOW_HUMIDITY;
         else
            Alarms &= ~ALARM_LOW_HUMIDITY;
      }
      else Alarms &= ~ALARM_LOW_HUMIDITY;

      /* Clear any previous communications error */
      Alarms &= ~ALARM_COMM_ERROR;
   }
   else Alarms |= ALARM_COMM_ERROR;

   /* If there are alarms */
   if ( Alarms )
   {
      /* Update the LCD and pulse the output pin */
      Events |= (LCD_EVENT|OUT_EVENT);
   }   
   else if ( Display == DISPLAY_TEMPERATURE )
   {
      /* Update the LCD */
      Events |= LCD_EVENT;
   }   

   /* Stop blinking LED4 */
   LED4 ^= 1;
}


/*****************************************************************************/
/* ProcessRTCEvent                                                           */
/*****************************************************************************/

void ProcessRTCEvent()
{
   /* Increment the system time */
   SystemTime++;

   /* If displaying time or date or alarms then update the LCD */
   if ( Display==DISPLAY_TIME || Display==DISPLAY_DATE || Alarms )
      Events |= LCD_EVENT;
      
   /* If it's time to take another sample */
   if ( SystemTime == SampleTime )
   { 
      /* Calculate the time of the next sample */
      SampleTime += SampleInterval;

      /* Take the sample */
      TakeSample();
   }
}
   

/*****************************************************************************/
/* ProcessSW1Event                                                           */
/*****************************************************************************/

void ProcessSW1Event()
{
   /* If there are no alarms */
   if ( !Alarms )
   { 
      /* Change to the next display with wrap around */
      Display++;
      if ( Display > DISPLAY_SAMPLE_COUNT )
         Display = DISPLAY_TEMPERATURE;
   }
   /* Clear alarms so other info can be displayed */   
   else Alarms = 0;   

   /* Update the LCD */   
   Events |= LCD_EVENT;   
}
   

/*****************************************************************************/
/* ProcessSW2Event                                                           */
/*****************************************************************************/

void ProcessSW2Event()
{
   /* Toggle the scale between Celsius and Fahrenheit */
   if ( Scale == 'C' )
      Scale = 'F';
   else
      Scale = 'C';  
      
   /* If displaying the temperature then update the LCD */
   if ( Display == DISPLAY_TEMPERATURE )
      Events |= LCD_EVENT;   
}
   

/*****************************************************************************/
/* ProcessSW3Event                                                           */
/*****************************************************************************/

void ProcessSW3Event()
{
   /* Start and stop logging */
   if ( Logging )
      Logging = FALSE;
   else
      Logging = TRUE;      
      
   /* If displaying the sample count then update the LCD */
   if ( Display == DISPLAY_SAMPLE_COUNT )
      Events |= LCD_EVENT;
}
   

/*****************************************************************************/
/* ProcessLCDEvent                                                           */
/*****************************************************************************/

void ProcessLCDEvent()
{
   /* Clear the display */
   lcdPutChar('\r');
   lcdPutChar('\n');

   /* If there are alarms */
   if ( Alarms )
   {
      /* If the MCU failed to communicate with the sensor */
      if ( Alarms & ALARM_COMM_ERROR )
      {
         lcdPutString("COMM ERR");
      }
      else /* Communications with the sensor is good */
      {
         /* Display humidity alarm on odd seconds */
         if ( SystemTime & 1 )
         {
            if ( Alarms & ALARM_HIGH_HUMIDITY )
            {
               lcdPutHumidity( Humidity );
               lcdPutString(" HI");
            }
            else if ( Alarms & ALARM_LOW_HUMIDITY )
            {
               lcdPutHumidity( Humidity );
               lcdPutString(" LOW");
            }
         }
         else /* Display temperature alarm on even seconds */
         {
            if ( Alarms & ALARM_HIGH_TEMPERATURE )
            {
               lcdPutTemperature( Temperature, Scale );
               lcdPutString(" HI");
            }
            else if ( Alarms & ALARM_LOW_TEMPERATURE )
            {
               lcdPutTemperature( Temperature, Scale );
               lcdPutString(" LOW");
            }
         }
      }
   }
   else /* There are no alarms */
   {
      switch ( Display )
      {
         case DISPLAY_TEMPERATURE:
            lcdPutTemperature( Temperature, Scale );
            lcdPutHumidity( Humidity );
            break;
         case DISPLAY_DATE:
            if ( Timezone == TIMER_MODE )
               lcdPutDays( SystemTime );
            else   
               lcdPutDate( SystemTime );
            break;
         case DISPLAY_TIME:
            if ( Timezone == TIMER_MODE )
               lcdPutTime( SystemTime, 24 );
            else
               lcdPutTime( SystemTime, 12 );
            break;
         case DISPLAY_SAMPLE_COUNT:
            lcdPutSampleCount( SampleCount, Logging );
            break;
      }
   }
}   


/*****************************************************************************/
/* ProcessOutEvent                                                           */
/*****************************************************************************/

void ProcessOutEvent()
{
   /* Generate a 500 ms negative pulse on the output pin */
   OUTPUT = 0;
   Delay( 500 );
   OUTPUT = 1;
}   


/*****************************************************************************/
/* ProcessEvents                                                             */
/*****************************************************************************/

void ProcessEvents( void )
{
   /* Process each event */
   if ( Events & RTC_EVENT ) { Events &= ~RTC_EVENT; ProcessRTCEvent(); }
   if ( Events & SW1_EVENT ) { Events &= ~SW1_EVENT; ProcessSW1Event(); }
   if ( Events & SW2_EVENT ) { Events &= ~SW2_EVENT; ProcessSW2Event(); }
   if ( Events & SW3_EVENT ) { Events &= ~SW3_EVENT; ProcessSW3Event(); }
   if ( Events & LCD_EVENT ) { Events &= ~LCD_EVENT; ProcessLCDEvent(); }
   if ( Events & OUT_EVENT ) { Events &= ~OUT_EVENT; ProcessOutEvent(); }   

   /* If all events are processed */
   if ( Events == 0 )
   {
      /* Disable interrupts */
      EA = 0;

      /* Sleep until woken up by RTC alarm or port match */
      PMU0CF = SLEEP | RTCAWK | PMATWK;

      /* Clear the wake-up flags */
      PMU0CF = CLEAR;

      /* Re-enable interrupts */
      EA = 1;
   }
}

