/******************************************************************************/
//
// Software License Agreement
//
// CODE OWNERSHIP AND DISCLAIMER OF LIABILITY
// Microchip Technology Incorporated ("Microchip") retains
// all ownership and intellectual property rights in the code
// accompanying this message and in all derivatives hereto.
// You may use this code, and any derivatives created by any
// person or entity by or on your behalf, exclusively with
// Microchips proprietary products. Your acceptance and/or
// use of this code constitutes agreement to the terms and
// conditions of this notice.
// CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP "AS IS".
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING,
// BUT NOT LIMITED TO, IMPLIED WARRANTIES OF NON-INFRINGEMENT,
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO
// THIS CODE, ITS INTERACTION WITH MICROCHIPS PRODUCTS, COMBINATION
// WITH ANY OTHER PRODUCTS, OR USE IN ANY APPLICATION.
// YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL MICROCHIP BE
// LIABLE, WHETHER IN CONTRACT, WARRANTY, TORT (INCLUDING NEGLIGENCE
// OR BREACH OF STATUTORY DUTY), STRICT LIABILITY, INDEMNITY, CONTRIBUTION,
// OR OTHERWISE, FOR ANY INDIRECT, SPECIAL, PUNITIVE, EXEMPLARY, INCIDENTAL
// OR CONSEQUENTIAL LOSS, DAMAGE, FOR COST OR EXPENSE OF ANY KIND WHATSOEVER
// RELATED TO THE CODE, HOWSOEVER CAUSED, EVEN IF MICROCHIP HAS BEEN ADVISED
// OF THE POSSIBILITY OR THE DAMAGES ARE FORESEEABLE. TO THE FULLEST EXTENT
// ALLOWABLE BY LAW, MICROCHIP'S TOTAL LIABILITY ON ALL CLAIMS IN ANY WAY
// RELATED TO THIS CODE, SHALL NOT EXCEED THE PRICE YOU PAID DIRECTLY TO
// MICROCHIP SPECIFICALLY TO HAVE THIS CODE DEVELOPED.
// You agree that you are solely responsible for testing the code and
// determining its suitability. Microchip has no obligation to modify,
// test, certify, or support the code.
//
/*****************************************************************************/
// File: MCP8025_Evaluation.c
//
// This program validates the MCP8025 driver using a basic 6-step trapezoidal
// drive method.
//
// Written By: Matthew Williams
//             
// Microchip Technology Inc
//
//
// The following files should be included in the MPLAB project:
//
// MCP8025_Evaluation.c   -- Main source code file
// MCP802X.h              -- Driver header file
// XC16.h                 -- XC16 compiler header
// dsp.h                  -- dsPIC header file
// Uart.h                 -- Uart header file
// Uart.c                 -- Uart Source file
// libdsp-elf.a           -- XC16 dsPIC libray  file
//
//
/*****************************************************************************/
//
// print out the version during compilation
#warning ADM00600 Revision 1.0
// Revision History
//
// 08-SEP-2014 -- Initial version.
/*****************************************************************************/


/******************** Define DRIVER IC Selection First ***********************/
#define MCP8025 1
#define MCP8026 2

#define DRIVER MCP8025


/********************* Include Files *****************************************/
#include "MCP802X.h"        // must be first include for FCY
#include <XC.h>
#include <dsp.h>
#include "uart.h"
#include <libpic30.h>



/********************* System Configuration Definitions *************************/

/******************************************************************************
 * Motor operation: CLOSED/OPEN LOOP DEFINITION
 * If CLOSED loop is selected, the compiler will add the PID controller for speed
 *******************************************************************************/
#define CLOSEDLOOPMODE 1
#define OPENLOOPMODE   2
#define LOOPMODE      OPENLOOPMODE

/*******************************************************************************
 *                           PIC Configuration Bits
 ********************************************************************************/
/* external 8 MHz crystal is on OSC1, OSC2 */
/* Oscillator Source Selection Register (FOSCSEL) */
/* startup at FRC then switch to PLL later in code */
_FOSCSEL(FNOSC_FRC & PWMLOCK_OFF & IESO_OFF);  /* Internal Fast RC (FRC) Oscillator (7.37 MHz) = 0b000 */


/* FCY = FOSC / 2 = 7.37 MHz/2 = 3.685 MHz or 271 ns cycle time */

/* Oscillator Configuration Register (FOSC) */
/* FCKSM_CSECMD = 0xFF7F = Clock switching is enabled,
   Fail-Safe Clock Monitor is disabled */
/* OSCIOFNC_ON  = 0xFFFB = OSC2 pin has digital I/O function */
/* POSCMD_NONE  = 0xFFFF = Primary Oscillator Disabled */
/* IOL1WAY_OFF = 0xFFDF = Peripheral pin select configuration: Allow multiple reconfigurations */
_FOSC(FCKSM_CSECMD & OSCIOFNC_ON & POSCMD_NONE & IOL1WAY_OFF);

/*Watchdog Timer Enabled/disabled by user software*/
_FWDT(FWDTEN_OFF & PLLKEN_ON);

/* Register FPOR (0xf8000c)
 ** PWM Output Pin Reset: Controlled by Port Register
 ** PWM High Side Polarity: Active High
 ** PWM Low Side Polarity: Active High
 **   POR Timer Value:
 **     FPWRT_PWR1           Disabled
 **     FPWRT_PWR2           2ms
 **     FPWRT_PWR4           4ms
 **     FPWRT_PWR8           8ms
 **     FPWRT_PWR16          16ms
 **     FPWRT_PWR32          32ms
 **     FPWRT_PWR64          64ms
 **     FPWRT_PWR128         128ms
 **
 **   Alternate I2C  pins:
 **     ALTI2C_ON            I2C mapped to ASDA1/ASCL1 pins
 **     ALTI2C_OFF           I2C mapped to SDA1/SCL1 pins
 **
 **   Motor Control PWM Low Side Polarity bit:
 **     LPOL_OFF             PWM module low side output pins have active-low output polarity
 **     LPOL_ON              PWM module low side output pins have active-high output polarity
 **
 **   Motor Control PWM High Side Polarity bit:
 **     HPOL_OFF             PWM module high side output pins have active-low output polarity
 **     HPOL_ON              PWM module high side output pins have active-high output polarity
 **
 **   Motor Control PWM Module Pin Mode bit:
 **     PWMPIN_OFF           PWM module pins controlled by PWM module at device Reset
 **     PWMPIN_ON            PWM module pins controlled by PORT register at device Reset
 */
//_FPOR(PWMPIN_ON & HPOL_ON & LPOL_ON);


/* Register FICD (0xf8000e)
 ** Background Debug Enable Bit: Device will Reset in user mode
 ** Debugger/Emulator Enable Bit: Reset in operational mode
 ** JTAG Enable Bit: JTAG is disabled
 **   ICD Communication Channel Select bits:
 **     ICS_NONE             Reserved, do not use
 **     ICS_PGD3             Communicate on PGEC3 and PGED3
 **     ICS_PGD2             Communicate on PGEC2 and PGED2
 **     ICS_PGD1             Communicate on PGEC1 and PGED1
 **
 **   JTAG Enable bit:
 **     JTAGEN_OFF           JTAG is disabled
 **     JTAGEN_ON            JTAG is enabled
 **
 */
_FICD(JTAGEN_OFF & ICS_PGD2);
/********************** End of System Configuration Definitions ***************/



/****************************************************************************
*                   General System Definitions
*****************************************************************************/
#define DEBOUNCE_DELAY 20   /* Push button debounce delay, expressed in milliseconds */


/*****************************************************************************
*                       Motor Control Definitions
******************************************************************************/
/* define PWM timer Counts per PWM period */
#define PWM_COUNTS_PER_PERIOD (FPLLO/FPWM -1)

/* duty cycle counts used for A/D sample start times */
#define PWM_DUTY_CYCLE_P1_PERCENT  ((unsigned int)((FPLLO * 0.001)/FPWM))
#define PWM_DUTY_CYCLE_P2_PERCENT  ((unsigned int)((FPLLO * 0.002)/FPWM))
#define PWM_DUTY_CYCLE_P3_PERCENT  ((unsigned int)((FPLLO * 0.003)/FPWM))
#define PWM_DUTY_CYCLE_P4_PERCENT  ((unsigned int)((FPLLO * 0.004)/FPWM))
#define PWM_DUTY_CYCLE_P5_PERCENT  ((unsigned int)((FPLLO * 0.005)/FPWM))
#define PWM_DUTY_CYCLE_1_PERCENT   ((unsigned int)((FPLLO * 0.01)/FPWM))
#define PWM_DUTY_CYCLE_2_PERCENT   ((unsigned int)((FPLLO * 0.02)/FPWM))
#define PWM_DUTY_CYCLE_4_PERCENT   ((unsigned int)((FPLLO * 0.04)/FPWM))
#define PWM_DUTY_CYCLE_5_PERCENT   ((unsigned int)((FPLLO * 0.05)/FPWM))
#define PWM_DUTY_CYCLE_6_PERCENT   ((unsigned int)((FPLLO * 0.06)/FPWM))
#define PWM_DUTY_CYCLE_8_PERCENT   ((unsigned int)((FPLLO * 0.08)/FPWM))
#define PWM_DUTY_CYCLE_10_PERCENT  ((unsigned int)((FPLLO * 0.10)/FPWM))
#define PWM_DUTY_CYCLE_15_PERCENT  ((unsigned int)((FPLLO * 0.15)/FPWM))
#define PWM_DUTY_CYCLE_20_PERCENT  ((unsigned int)((FPLLO * 0.20)/FPWM))
#define PWM_DUTY_CYCLE_30_PERCENT  ((unsigned int)((FPLLO * 0.30)/FPWM))
#define PWM_DUTY_CYCLE_40_PERCENT  ((unsigned int)((FPLLO * 0.40)/FPWM))
#define PWM_DUTY_CYCLE_50_PERCENT  ((unsigned int)((FPLLO * 0.50)/FPWM))
#define PWM_DUTY_CYCLE_60_PERCENT  ((unsigned int)((FPLLO * 0.60)/FPWM))
#define PWM_DUTY_CYCLE_90_PERCENT  ((unsigned int)((FPLLO * 0.90)/FPWM))
#define PWM_DUTY_CYCLE_95_PERCENT  ((unsigned int)((FPLLO * 0.95)/FPWM))
#define PWM_DUTY_CYCLE_96_PERCENT  ((unsigned int)((FPLLO * 0.96)/FPWM))
#define PWM_DUTY_CYCLE_100_PERCENT ((unsigned int)(FPLLO/FPWM))

/* set open loop acceleration and deceleration rates */
#define OPEN_LOOP_ACCEL_RATE    PWM_DUTY_CYCLE_P5_PERCENT
#define OPEN_LOOP_DECEL_RATE    PWM_DUTY_CYCLE_P1_PERCENT

/* Time for initial rotor rampup.  Number of PWM periods per commutation. */
/* should match KV of motor at RAMP_END_PERIOD based on Motor voltage and dutycycle */
#define RAMPUP_START_PERIOD  150  /* start with 140 PWM periods per commutation */
#define RAMPUP_END_PERIOD    140  /* end with 130 PWM periods per commutation */

/* duty cycle for starting the motor */
/* adjust the startup duty cycle for each specific motor and supply voltage */
#define STARTUP_DUTY_CYCLE PWM_DUTY_CYCLE_15_PERCENT

/* maximum duty cycle P1DC1=P1DC2=P1DC3 (1474 or 736 100%) */
#define MAX_DUTY_CYCLE  PWM_DUTY_CYCLE_96_PERCENT

/* minimum duty cycle P1DC1=P1DC2=P1DC3 */
#define MIN_DUTY_CYCLE  PWM_DUTY_CYCLE_10_PERCENT

/* Phase Advance Angle - (0 < advance <10 )
 ** Phase Advance adjusts the 30 degree calculated commutation timer
 ** count. A n advance of 0 degrees triggers commutation at the 30
 ** degree calculated time.  A 1 degree advance would subtract 1
 ** degree of timer counts from the calculated count.  That would
 ** trigger a commutation 1 degree earlier than calculated.  May be
 ** useful to handle latencies and high speeds.
 */
#define PHASE_ADVANCE_DEGREES 0

/* Blanking count expressed in PWM periods used to avoid false zero-crossing
 ** detection after commutating motor.  (20 us/period @ 50 kHz PWM)
 ** The counter is used to delay the ADC input for ZC_BLANKING_COUNT PWM periods
 ** More than 2 counts may cause the ADC conversion to occur after the crossover
 ** occurs, causing the motor to stall at high speeds.
 ** use 1 for fast R/C hobby motors.
 ** use 2 for Hurst motor.
 */
#define ZC_BLANKING_COUNT 2


/********************* PID Varibles *********************************/
#if (LOOPMODE == CLOSEDLOOPMODE)
  int           DesiredSpeed;
  int           CurrentSpeed;
  int           SpeedControl_P =    1;  // The Proportional term for the PI speed control loop
  int           SpeedControl_I = 1000;  // The Integral term for the PI speed control loop
  int           SpeedControl_D =    0;  // The Derivative term for the PI speed control loop
  fractional    PIDGainCoefficients[3]; //Control gains for the speed control loop
  fractional    abcCoefficients[3] __attribute__ ((space(xmemory),__aligned__(4)));
  fractional    controlHistory[3] __attribute__ ((space(ymemory),__aligned__(4)));
  tPID          PIDStructure;           // PID Structure
#endif // (LOOPMODE == CLOSEDLOOPMODE)


/*************************** PWM Register Definitions *************************
 *
 * PWM Definitions for the IOCONx register
 *
 *  15   14   13   12  :  11    10     9      8    :    7       6       5       4    :   3      2     1     0
 * PENH PENL POLH POLL : PMOD1 PMOD0 OVRENH OVRENL : OVRDAT1 OVRDAT0 FLTDAT1 FLTDAT0 : CLDAT1 CLDAT0 SWAP OSYNC
 *  *    *    0    0   :   0     0     *      *    :    *       *       0       0    :   0      0      0    1
 *
 ******************************************************************************/
// COMPLEMENTARY mode allows high side and low side to be compliments of each other
#define PMOD_SEL 0x0000     /* complimentary mode bits <11:10> */

/* define IOCONx.PENH and IOCONx.PENL PWM register bit positions */
#define PENH_ENA 0x8000     /* enable PWMH ouput for PWM */
#define PENL_ENA 0x4000     /* enable PWML ouput for PWM */
#define PENH_DIS 0x0000     /* enable PWMH ouput for GPIO */
#define PENL_DIS 0x0000     /* enable PWMH ouput for GPIO */

/* define IOCONx.POLH and IOCONx.POLL PWM polarity */
#define POLH_HI 0x0000      /* PWMxH is active high */
#define POLH_LO 0x2000      /* PWMxH is active low  */
#define POLL_HI 0x0000      /* PWMxL is active high */
#define POLL_LO 0x1000      /* PWMxL is active low  */

/* define IOCONx.OVRENH and IOCONx.OVRENL override bits */
#define OVRENH_OVR  0x0200  /* PWMxH override controled by GPIO */
#define OVRENH_PWM  0x0000  /* PWMxH override controled by PWM  */
#define OVRENL_OVR  0x0100  /* PWMxL override controled by GPIO */
#define OVRENL_PWM  0x0000  /* PWMxL override controled by PWM  */
#define OVRDATH_1   0x0080  /* PWMxH override is "1" */
#define OVRDATH_0   0x0000  /* PWMxH override is "0" */
#define OVRDATL_1   0x0040  /* PWMxL override is "1" */
#define OVRDATL_0   0x0000  /* PWMxL override is "0" */

/* define IOCONx.OSYNC synchronization bit */
#define OSYNC_PWM 0x0001  /* output override synchronized with PWM timebase */
#define OSYNC_CLK 0x0000  /* output override synchronized with next CPU CLK */

/* base IOCON bits for use with PWM over-riding phases */
#define IOCON_PWM_CFG     PMOD_SEL | PENH_ENA | PENL_ENA | POLH_HI | POLL_HI | OSYNC_PWM

//const unsigned int PWM_STATE[] = {0x0000,0x3001,0x3004,0x0304,0x0310,0x0C10,0x0C01,0x0000};  dsPIC FJ
/* PWM Commutation State Table for dsPIC EP
 * const unsigned int PWM_STATEn[] = {
 *                                                    OVRENH1  OVRENL1 OVRDAT1 OVRENH2 OVRENL2 OVRDAT2 OVRENH3 OVRENL3 OVRDAT3  BEMF
 * 0x0000,      OVR => 3H=0 3L=0 2H=0 2L=0 1H=0 1L=0     1         1       00      1       1       00      1       1       00     x
 * 0x3001,  PWM => 3H 3L, OVR => 2H=0 2L=0 1H=0 1L=1     1         1       01      1       1       00      0       0       xx     2
 * 0x3004,  PWM => 3H 3L, OVR => 2H=0 2L=1 1H=0 1L=0     1         1       00      1       1       01      0       0       xx     1
 * 0x0304,  PWM => 1H 1L, OVR => 3H=0 3L=0 2H=0 2L=1     0         0       xx      1       1       01      0       0       00     3
 * 0x0310,  PWM => 1H 1L, OVR => 3H=0 3L=1 2H=0 2L=0     0         0       xx      1       1       00      1       1       01     2
 * 0x0C10,  PWM => 2H 2L, OVR => 3H=0 3L=1 1H=0 1L=0     1         1       00      0       0       xx      1       1       01     1
 * 0x0C01,  PWM => 2H 2L, OVR => 3H=0 3L=0 1H=0 1L=1     1         1       01      0       0       xx      1       1       00     3
 * 0x3C01,  PWM => 3H 3L, 2H, 2L,   OVR => 1H=0, 1L=1    1         1       01      0       0       xx      0       0       xx     x  Align Rotor
 */

// 0x0000,
#define IOCON_0x0000  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)

// 0x3001, 1=L, 2=F, 3=H  Falling BEMF 2
#define IOCON1_0x3001  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)
#define IOCON2_0x3001  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)
#define IOCON3_0x3001  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)

// 0x3004, 1=R, 2=L, 3=H  Rising BEMF 1
#define IOCON1_0x3004  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)
#define IOCON2_0x3004  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)
#define IOCON3_0x3004  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)

// 0x0304, 1=H, 2=L, 3=F Falling BEMF 3
#define IOCON1_0x0304  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)
#define IOCON2_0x0304  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)
#define IOCON3_0x0304  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)

// 0x0310,  1=H, 2=R, 3=L  Rising BEMF 2
#define IOCON1_0x0310  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)
#define IOCON2_0x0310  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)
#define IOCON3_0x0310  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)

// 0x0C10, 1=F, 2=H, 3=L  Falling BEMF 1
#define IOCON1_0x0C10  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)
#define IOCON2_0x0C10  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)
#define IOCON3_0x0C10  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)

// 0x0C01, 1=L, 2=H, 3=R  Rising BEMF 3
#define IOCON1_0x0C01  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)
#define IOCON2_0x0C01  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)
#define IOCON3_0x0C01  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)

// 0x3C01, Lockstate - PHASE 1 L on, PHASE 2,3 PWM ON
#define IOCON1_LOCKSTATE  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)
#define IOCON2_LOCKSTATE  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)
#define IOCON3_LOCKSTATE  (IOCON_PWM_CFG | OVRENH_PWM | OVRENL_PWM | OVRDATH_0 | OVRDATL_0)

// now create the PWM state table
const unsigned int PWM_STATE1[] = {IOCON_0x0000,IOCON1_0x3001,IOCON1_0x3004,IOCON1_0x0304,IOCON1_0x0310,IOCON1_0x0C10,IOCON1_0x0C01,IOCON_0x0000};
const unsigned int PWM_STATE2[] = {IOCON_0x0000,IOCON2_0x3001,IOCON2_0x3004,IOCON2_0x0304,IOCON2_0x0310,IOCON2_0x0C10,IOCON2_0x0C01,IOCON_0x0000};
const unsigned int PWM_STATE3[] = {IOCON_0x0000,IOCON3_0x3001,IOCON3_0x3004,IOCON3_0x0304,IOCON3_0x0310,IOCON3_0x0C10,IOCON3_0x0C01,IOCON_0x0000};


// define all lowside drivers ON
#define IOCON_LOWSIDE_ON  (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_1)

// define all drivers OFF
#define IOCON_ALL_OFF     (IOCON_PWM_CFG | OVRENH_OVR | OVRENL_OVR | OVRDATH_0 | OVRDATL_0)

#define BOOTSTRAP_CHARGE_STATE  IOCON_LOWSIDE_ON  /* Charge Bootstrap Capacitors, PHA_L, PHB_L, PHC_L to -Vbus */
#define MOTOR_SHUTDOWN_STATE    IOCON_LOWSIDE_ON  /* Turn off highside, turn on lowside, PHA_L, PHB_L, PHC_L to -Vbus */

/* AND / OR operators for masking the active BEMF signal*/
const unsigned int ADC_MASK[7] = {0x0000,0x0002,0x0001,0x0004,0x0002,0x0001,0x0004};//,0x0000};
const unsigned int ADC_XOR[7] = {0x0000,0x0000,0xFFFF,0x0000,0xFFFF,0x0000,0xFFFF};//,0x0000};

/* BEMF Majority Function Filter values*/
const unsigned char ADC_BEMF_FILTER[64] = { 0x00,0x02,0x04,0x06,0x08,0x0A,0x0C,0x0E,
  0x10,0x12,0x14,0x16,0x18,0x1A,0x1C,0x1E,
  0x20,0x22,0x24,0x26,0x28,0x2A,0x2C,0x2E,0x01,0x01,0x01,0x36,0x01,0x3A,0x3C,0x3E,
  0x00,0x02,0x04,0x06,0x08,0x0A,0x0C,0x0E,0x01,0x01,0x01,0x16,0x01,0x1A,0x1C,0x1E,
  0x01,0x01,0x01,0x26,0x01,0x2A,0x2C,0x2E,0x01,0x01,0x01,0x36,0x01,0x3A,0x3C,0x3E};

/* Application Flags to indicate the motor status */
union
{
  unsigned char all;
  struct
  {
    unsigned char RunMotor        : 1;  /* 1 = Run */
    unsigned char RotorAlignment  : 1;  /* 1 = Rotor Alignment during startup */
    unsigned char ClosedLoopMode  : 1;  /* 0 = open loop, 1 = closed loop */
    unsigned char Startup         : 1;  /* 1 = Startup Motor */
    unsigned char Stop            : 1;  /* 1 = Stop Motor */
    unsigned char Fault           : 1;  /* 1 = motor fault */
    unsigned char unused          : 1;  /* unused */
  } ;
} Flags;

// Boolean variables used to save the comparison between
// the phases and the neutral point
struct
{
  unsigned char PhaseAOutput : 1;  /* PHASE A comparator output bit 0 */
  unsigned char PhaseBOutput : 1;  /* PHASE B comparator output bit 1 */
  unsigned char PhaseCOutput : 1;  /* PHASE C comparator output bit 2 */
  unsigned char unused : 5;
} Comparator;

/********************* Motor Control Variables *******************************/
unsigned char RampUpCommState;            /* motor startup ramp commutation state */
int           ReferenceSpeed;             /* SPEED pot input reference value */
int           PhaseAdvance = 0;           /* amount of desired phase advance */
int           Timer2Average;                       /* average of timer 2 (60 degree timer) */
unsigned int  MotorNeutral;               /* Motor Neutral value */
unsigned int  MotorPhaseA;                /* Motor Phase A monitor input */
unsigned int  MotorPhaseB;                /* Motor Phase B monitor input */
unsigned int  MotorPhaseC;                /* Motor Phase C monitor input */
unsigned int  ComparatorOutputs;          /* bitmap of Phase comparator outputs */
unsigned int  ExpectedCrossover;          /* timer count when crossover is expected to occur */
unsigned int  Timer1Value;                /* timer 1 value */
unsigned int  DesiredPWMDutyCycle;        /* target PWM duty cycle */
volatile int  CurrentPWMDutyCycle;        /* current PWM duty cycle */
volatile char BlankingCounter;            /* A/D blanking counter */
volatile char Startup_Commutations;       /* number of commutations to ignore during startup */
volatile unsigned char ADCCommState;      /* ADC commutation state */
volatile unsigned char adcBackEMFFilter;  /* back EMF filter value */
volatile unsigned int  PWMticks;          /* number of PWM periods during startup */


/********************* Function Definitions *********************************/
void InitMCPWM(void);                 /* initialize PWM module */
void InitADC10(void);                 /* initialize ADC module */
void InitTMR1(void);                  /* initialize Timer 1 module */
void InitTMR2(void);                  /* initialize Timer 2 module */
void PreCommutationState(void);       /* commutaion timer calculations */
void SpeedPILoopController(void);     /* PID closed loop control calculations */
void OpenLoopController(void);        /* open loop control calculations */
void ConfigureDriver(void);           /* configure motor driver */
void CheckSwitches(void);             /* check onboard switches for changes */
void DoMotorStartup(void);            /* start the motor operation */
void DoMotorStop(void);               /* stop the motor operation */
void SpeedController(void);           /* motor speed calculation */
void CheckDriverCommunications(void); /* check for driver communications */
void CheckIlimit(void);               /* check for Ilimit Fault signal */


/******************************************************************************
 * Function: main(void)
 *
 * Output: None
 *
 * Overview: Main function used to init the ADC, PWM and TIMER2 modules.
 * It also inits the global variables used in the interrupts and
 * PI controller.
 * The main task executed here is to start and stop the motor
 * as well as setting the ramp-up initial parameters to
 * spin the motor
 *
 * Note: None
 *******************************************************************************/
int main(void)
{
  // Configure Clock
  PLLFBD = PLL_DIV;
  CLKDIV = 0;
  CLKDIVbits.PLLPOST = 0; // N1=2
  CLKDIVbits.PLLPRE = 0;  // N2=2
  OSCTUN = 0;             // Tune FRC oscillator, if FRC is used

  // Disable Watch Dog Timer
  RCONbits.SWDTEN = 0;

  // Clock switch to incorporate PLL
  __builtin_write_OSCCONH(0x01);            // Initiate Clock Switch to FRC with PLL (NOSC=0b001)
  __builtin_write_OSCCONL( OSCCON | 0x01);  // Start clock switching
  while (OSCCONbits.COSC != 0b001) { };     // Wait for Clock switch to occur
  while (OSCCONbits.LOCK != 1)      { };     // Wait for PLL to lock

  /****************** I/O port Init *********************************
   * Configuring Digital PORTS multiplexed with PWMs as outputs
   * TRISx = data direction.  1=input, 0=output
   * LATx = latch.  All reads and writes go through port latch
   * ODCx = open drain control register
   * ANSELx and TRISx control A/D port pins.
   * ANSELx = 1 for Analog, 0 for Digital (default: 0xFFFF)
   * CNEN1/2 = interrupt on change enable.  1=enable CN interupt
   * CNPU1/2 = interrupt on change pullup enable.  1=enable pullup, not for DIGITAL node
   * RPINRx = Peripheral Pin Select
   */

  /* Port A */
  TRISAbits.TRISA0 = 1;   // AN0, A/D input, MONITOR1, RA0
  TRISAbits.TRISA1 = 1;   // AN1, A/D input, MONITOR2, RA1
  TRISAbits.TRISA2 = 0;   LATAbits.LATA2 = 0;   // output - RA2 spare
  TRISAbits.TRISA3 = 0;   LATAbits.LATA3 = 0;   // output - RA3 spare
  TRISAbits.TRISA4 = 0;   LATAbits.LATA4 = 0;   // output - RA4 Spare LED
  TRISAbits.TRISA7 = 0;   LATAbits.LATA7 = 0;   // output - Driver CE pin 1=enable
  TRISAbits.TRISA8 = 1;   LATAbits.LATA8 = 0;   // input - Switch SW1
  TRISAbits.TRISA9 = 0;   LATAbits.LATA9 = 0;   // output - RA9 spare
  TRISAbits.TRISA10 = 0;  LATAbits.LATA10 = 0;  // output - RA10 Spare

  /* Port B */
  TRISBbits.TRISB0  = 1;  // AN2, A/D input, MONITOR3, RB0
  TRISBbits.TRISB1  = 1;  // AN3, A/D input, IOUTB, RB1
  TRISBbits.TRISB2  = 1;  // AN4, A/D input, IOUTA, RB2
  TRISBbits.TRISB3  = 1;  // AN5, A/D input, MONITOR4, RB3
  TRISBbits.TRISB4  = 1;  LATBbits.LATB4  = 0;  // input, Switch SW2
  TRISBbits.TRISB5  = 1;  LATBbits.LATB5  = 0;  // input, PGD Programmer
  TRISBbits.TRISB6  = 1;  LATBbits.LATB6  = 0;  // input, PGC Programmer
  TRISBbits.TRISB7  = 1;  LATBbits.LATB7  = 1;  // Redirected to RP38, UART RX from DE2
  TRISBbits.TRISB8  = 1;  LATBbits.LATB8  = 1;  // input, ILIMIT from Driver, 1=0k, 0=fault or Current Limit
  TRISBbits.TRISB9  = 0;  LATBbits.LATB9  = 1;  // Redirected to RP41, UART TX to DE2
  TRISBbits.TRISB10 = 0;  LATBbits.LATB10 = 0;  //PWM, PWM1H3
  TRISBbits.TRISB11 = 0;  LATBbits.LATB11 = 0;  //PWM, PWM1L3
  TRISBbits.TRISB12 = 0;  LATBbits.LATB12 = 0;  //PWM, PWM1H2
  TRISBbits.TRISB13 = 0;  LATBbits.LATB13 = 0;  //PWM, PWM1L2
  TRISBbits.TRISB14 = 0;  LATBbits.LATB14 = 0;  //PWM, PWM1H1
  TRISBbits.TRISB15 = 0;  LATBbits.LATB15 = 0;  //PWM, PWM1L1

  /* Port C */
  TRISCbits.TRISC0 = 1;   // AN6, IOUTC
  TRISCbits.TRISC1 = 1;   // AN7, VDD_DIV16
  TRISCbits.TRISC2 = 1;   // AN8, Speed Pot

  #if (DRIVER == MCP8025)
    TRISCbits.TRISC3 = 0;   LATCbits.LATC3 = 0;   // output: Mux select signal 1
    TRISCbits.TRISC4 = 0;   LATCbits.LATC4 = 0;   // output: Mux select signal 1
    TRISCbits.TRISC5 = 1;   LATCbits.LATC5 = 0;   // input: Zero Crossing signal
    TRISCbits.TRISC6 = 1;   LATCbits.LATC6 = 0;   // input: LIN RX port
    TRISCbits.TRISC7 = 0;   LATCbits.LATC7 = 1;   // output: LIN TX port
  #elif (DRIVER == MCP8026)
    TRISCbits.TRISC3 = 0;   LATCbits.LATC3 = 0;   // output, RC3 unused
    TRISCbits.TRISC4 = 0;   LATCbits.LATC4 = 0;   // output, RC4 unused
    TRISCbits.TRISC5 = 0;   LATCbits.LATC5 = 0;   // output, RC5 unused
    TRISCbits.TRISC6 = 0;   LATCbits.LATC6 = 0;   // output, RC6 unused
    TRISCbits.TRISC7 = 0;   LATCbits.LATC7 = 0;   // output, RC7 unused
  #else
    #pragma error "Driver not selected"
  #endif

  TRISCbits.TRISC8 = 0;   LATCbits.LATC8 = 0;   // output, RC8 unused
  TRISCbits.TRISC9 = 0;   LATCbits.LATC9 = 0;   // output, RC9 unused

  /* Analog Ports */
  ANSELA = 0x0000;        // initialize all to digital I/O
  ANSELAbits.ANSA0 = 1;   // AN0, analog
  ANSELAbits.ANSA1 = 1;   // AN0, analog
  ANSELAbits.ANSA4 = 0;   // Digital I/O

  ANSELB = 0x0000;        // initialize all to digital I/O
  ANSELBbits.ANSB0 = 1;   // AN2, analog
  ANSELBbits.ANSB1 = 1;   // AN3, analog
  ANSELBbits.ANSB2 = 1;   // AN4, analog
  ANSELBbits.ANSB3 = 1;   // AN5, analog
  ANSELBbits.ANSB8 = 0;   // Digital I/O

  ANSELC = 0x0000;        // initialize all to digital I/O
  ANSELCbits.ANSC0 = 1;   // AN6, analog
  ANSELCbits.ANSC1 = 1;   // AN7, analog
  ANSELCbits.ANSC2 = 1;   // AN8, analog

  /****************** Functions init *********************************/
  INTCON1bits.NSTDIS = 0;   // Enabling nested interrupts
  InitMCPWM();              //Configuring MC PWM module
  InitADC10();              //Configuring ADC
  InitTMR2();               //Configuring TIMER 3, used to measure speed
  InitTMR1();               //Configuring TIMER 1, used for commutation

  /****************** Flags init *********************************/
  Flags.RotorAlignment = 0; // Turn off ramp up
  Flags.RunMotor = 0;       // indication the run motor condition
  Flags.ClosedLoopMode = 0; // running in Open Loop Mode
  Flags.Startup = 0;        // Start motor
  Flags.Stop = 1;           // Stop motor


  /****************** UART init **************************************/
  UARTInit();

  /****************** Initialize Driver **********************************/
  DRIVER_CE = 0;            // Toggle CE to clear any startup faults
  __delay_ms(10);

  DRIVER_CE = 1;
  __delay_ms(20);

  /* initialize MCP802x configuration registers */
  ConfigureDriver();

  /* clear dsPIC reset cause register */
  RCON = 0x00;

  /****************** Infinite Loop **********************************/
  for(;;)
  {
    CheckSwitches();

    if (Flags.Stop)	    // Stop Motor
    {
      DoMotorStop();
    }

    if (Flags.Startup)	// Startup Motor
    {
      DoMotorStartup();
    }

    /* check for motor faults */
    CheckIlimit();

    /* if a new fault is present, stop motor and turn on SPARE LED */
    if (Flags.Fault && !SPARE_LED)
    {
      // add user code here for faults and current comparator
      DoMotorStop();
      SPARE_LED = 1;
    }

    /* reset SPARE LED if fault has been cleared */
    if (!Flags.Fault)
    {
      SPARE_LED = 0;
    }

  }//end of infinite loop
  return 0;

}//end of main function


/******************************************************************************
 * Function:  void CheckIlimit()
 *
 * Output:		None
 *
 * Overview:	Checks for Motor Faults.
 *
 * Note:			Driver fault pin is filtered and monitored for a fault condition.
 *            Fault flag is set if a fault has occurred.
 *******************************************************************************/
void CheckIlimit ()
{
  static char  Filter_count = 0;

  /* check for motor driver fault condition */
  if (ILIMIT)
  {
    /* no fault = increment counter */
    ++Filter_count;
  }
  else
  {
    /* fault = decrement counter */
    --Filter_count;
  }

  /* see if filter is complete */
  if (Filter_count > 10)
  {
    Flags.Fault = 0;  // Fault has been reset
    Filter_count = 0;
  }
  else if (Filter_count < -10)
  {
    Flags.Fault = 1;  // Fault has occurred
    Filter_count = 0;
  }
  else
  {
    // Filtering in progress
  }
}

/******************************************************************************
 * Function:  void CheckSwitches()
 *
 * Output:		None
 *
 * Overview:	Checks for any switch changes.
 *
 * Note:			SW1 is a spare switch that may be user programmed.
 *            SW2 is the Start/Stop switch for motor operation.
 *******************************************************************************/
void CheckSwitches()
{
  static char  SW1_debounce_count = 0;
  static char  Sw1_Pressed = 0;
  static char  Sw1_Last = 1;
  static char  SW2_debounce_count = 0;
  static char  Sw2_Pressed = 0;
  static char  Sw2_Last = 1;

  __delay_ms(2);

  /****************** Check SW1 *********************/
  /* debounce at least 5 iterations */
  if (SW1)  // 1 = button pressed
  {
    ++SW1_debounce_count;
  }
  else  // 0 = button released
  {
    --SW1_debounce_count;
  }

  /* see if debounce is complete */
  if (SW1_debounce_count > 5)
  {
    Sw1_Pressed = 1;  // button pressed
    SW1_debounce_count = 0;
  }
  else if (SW1_debounce_count < -5)
  {
    Sw1_Pressed = 0;  // button released
    SW1_debounce_count = 0;
  }
  else
  {
    // button in transition
  }


  if (Sw1_Pressed != Sw1_Last)
  {
    Sw1_Last = Sw1_Pressed;

    /* switch changed state */
    if (Sw1_Pressed)
    {
      /*****************************
       * do SW1 Pressed stuff here *
       *****************************/

    }
    else
    {
      /******************************
       * do SW1 Released stuff here *
       ******************************/


    }
  } // end if (Sw1_Pressed != Sw1_Last)


  /****************** Check SW2 *********************/
  /* debounce at least 5 iterations */
  if (SW2)  // 1 = button pressed
  {
    ++SW2_debounce_count;
  }
  else  // 0 = button released
  {
    --SW2_debounce_count;
  }

  /* see if debounce is complete */
  if (SW2_debounce_count > 5)
  {
    Sw2_Pressed = 1;  // button pressed
    SW2_debounce_count = 0;
  }
  else if (SW2_debounce_count < -5)
  {
    Sw2_Pressed = 0;  // button released
    SW2_debounce_count = 0;
  }
  else
  {
    // button in transition
  } // end if (SW2_debounce_count > 5)


  if (Sw2_Pressed != Sw2_Last)
  {
    Sw2_Last = Sw2_Pressed;

    /* switch changed state */
    if (Sw2_Pressed)
    {
      /* always reset startup flag */
      Flags.Startup = 0;

      /* do SW2 Pressed stuff */
      if ((!Flags.RunMotor) && (!Flags.Stop))
      {
        /* motor is not running yet, start up in main loop */
        Flags.Startup = 1;
      }
      else
      {
        /* motor is running, shutdown now */
        DoMotorStop();

        /* Toggle Driver Enable line to clear any faults */
        DRIVER_CE = 0;
        __delay_us(100);
        DRIVER_CE = 1;
        __delay_ms(10);
      } // end if ((!Flags.RunMotor) && (!Flags.Stop))
    }
    else
    {
      /* do SW2 Released stuff */

    } // end if (Sw2_Pressed)
  } // end if (Sw2_Pressed != Sw2_Last)
} // end void CheckSwitches()


/******************************************************************************
 * Function:  void DoMotorStartup(void)
 *
 * Output:		None
 *
 * Overview:	Initializes the PID calculations, sets the initail duty cycle,
 *            sets the motor flags, and starts the motor PWM's.
 *
 * Note:			None
 *******************************************************************************/
void DoMotorStartup()
{
  int RampRate;     /* desired Ramp rate for each commutation state */

  Flags.Startup = 0;

  /* turn off timers and turn on lowside drivers */
  DoMotorStop();

  /* clear any motor driver faults */
  DRIVER_CE = 1;
  __delay_us(100);
  DRIVER_CE = 0;
  __delay_us(100);
  DRIVER_CE = 1;
  __delay_us(100);
  Flags.Fault = 0;

  /* make sure driver is configured */
  ConfigureDriver();

  /* reset commutation related variables */
  ADCCommState = 0;
  adcBackEMFFilter = 0;
  ComparatorOutputs = 0;		// Precondition all comparator bits as zeros

  /* set number of commutations to skip before switching from open loop to closed loop */
  Startup_Commutations = 5;


  /* initialize Motor Control Loop */
  #if (LOOPMODE == CLOSEDLOOPMODE)
    //Init PI controller variables
    DesiredSpeed = STARTUP_DUTY_CYCLE;
    CurrentSpeed = 0;

    // load the PID gain coeffecients into an array;
    PIDGainCoefficients[0] = SpeedControl_P;
    PIDGainCoefficients[1] = SpeedControl_I;
    PIDGainCoefficients[2] = SpeedControl_D;

    // Initialize the PIDStructure variable before calculation the K1, K2, and K3 terms
    PIDStructure.abcCoefficients = abcCoefficients;
    PIDStructure.controlHistory = controlHistory;

    // initialize control history
    PIDCoeffCalc(PIDGainCoefficients, &PIDStructure);
    PIDInit(&PIDStructure);
    PIDStructure.controlOutput = STARTUP_DUTY_CYCLE;
    Flags.ClosedLoopMode = 0; // running in Open Loop Mode
  #endif

  /*ROTOR ALIGNMENT SEQUENCE*/
  CurrentPWMDutyCycle = STARTUP_DUTY_CYCLE;	//Init PWM values
  DesiredPWMDutyCycle = STARTUP_DUTY_CYCLE;	//Init PWM values
  MDC = CurrentPWMDutyCycle;                // set PWM duty cycle

  BlankingCounter = 0;                      // initialize ADC blanking counter
  Flags.RotorAlignment = 1;                 // TURN ON rotor alignment sequence
  Flags.RunMotor = 1;                       // Indicating that motor is running
  Flags.ClosedLoopMode = 0;                 // start in openloop mode

  /* Toggle Driver Enable line again clear any faults */
  DRIVER_CE = 0;
  __delay_us(100);
  DRIVER_CE = 1;
  __delay_us(100);


  /***************************************************
   *            Rotor alignment sequence
   ***************************************************/
  /* initialize PWM outputs */
  PTCONbits.PTEN = 0;                       // disable PWM's before changing bits
  IFS5bits.PWM1IF = 0;                      // Clearing the PWM Interrupt Flag
  PWMCON1bits.TRGIEN = 1;                   // enable individual PWM interrupt
  IEC5bits.PWM1IE = 1;                      // Enabling the PWM interrupt
  IOCON1 = IOCON2 = IOCON3 = IOCON_ALL_OFF; // Manual Override all PWM outputs
  PTCONbits.PTEN = 1;                       // enable PWM control

  /* force all low side drives ON and high side OFF to charge bootstrap caps */
  IOCON1 = IOCON2 = IOCON3 = BOOTSTRAP_CHARGE_STATE;
  __delay_ms(100);

  /* Lock Rotor to a known initial start position */
  IOCON1 = IOCON2 = IOCON3 = IOCON_ALL_OFF; // Manual Override all PWM outputs
  IOCON1 = IOCON1_LOCKSTATE;
  IOCON2 = IOCON2_LOCKSTATE;
  IOCON3 = IOCON3_LOCKSTATE;
  __delay_ms(100);

  /* Stop PWM and prepare for rampup */
  IOCON1 = IOCON2 = IOCON3 = IOCON_ALL_OFF; // Manual Override all PWM outputs
  PTCONbits.PTEN = 0;                       // disable PWM's
  IEC5bits.PWM1IE = 0;                      // disable the PWM interrupt
  IFS5bits.PWM1IF = 0;                      // Clearing the PWM Interrupt Flag

  /* an external MOSFET undervoltage/overcurrent lockout fault may occur here if the blanking
   * time is too short and the supply voltage is below 8V.
   * Caused by low Highside drive voltage and high RDSon.
   * Fix by increasing driver blanking time and/or supply voltage.
   */

  /*************************************** ramp-up sequence **************************************/
  Flags.RotorAlignment = 0;                 // turn off rotor alignment sequence
  ADCCommState = RampUpCommState = 0;       // initialize current 6 step commutation state
  
  /* Each pass through loop waits for a shorter number of PWM periods before commutating */
  IEC5bits.PWM1IE = 1;                      // enable the PWM interrupt
  PTCONbits.PTEN = 1;                       // enable PWM's

  /* initialize ramp rate with number of PWM periods per commutation */
  RampRate = RAMPUP_START_PERIOD;

  /* initialize expected BEMF crossover timer count */
  ExpectedCrossover  = 0xFFFF;

  /* open loop: start by manually forcing 6-step cummutation */
  while (RampRate-- > RAMPUP_END_PERIOD)
  {

    /* Check for 6-step commutation wrap */
    if (++RampUpCommState > 6) RampUpCommState = 1;

    ADCCommState = RampUpCommState;         // initialize ADC commutation state

    /* initialize PWM period counter.  PWM interrupt increments counter */
    PWMticks = 0;                           // ramp rate PWM interrupt counter
    BlankingCounter = 0;                    // PWM period blanking counter

    /* force a commutation */
    IOCON1 = IOCON2 = IOCON3 = IOCON_ALL_OFF; // Manual Override all PWM outputs
    IOCON1 = PWM_STATE1[RampUpCommState];     // set driver 1 PWM output
    IOCON2 = PWM_STATE2[RampUpCommState];     // set driver 2 PWM output
    IOCON3 = PWM_STATE3[RampUpCommState];     // set driver 3 PWM output

    if (RampRate <= RAMPUP_END_PERIOD) break; // if this is last commutation, break out of loop and use BEMF

    /* reset the 60 degree commutation counter */
    T2CONbits.TON = 0;                      // stop TIMER2
    TMR2 = 0;                               // reset commutation time counter
    T2CONbits.TON = 1;                      // Start TIMER2

    /* loop until PWM interrupt increments PWMticks counter to commutation value */
    while(PWMticks < RampRate)              // Ramp Up delay until next commutation time
    {
      /* wait for commutation counter to time out or BEMF detected */
      if (Flags.ClosedLoopMode) break;		  // exit if BEMF is available
    }

    /* exit ramp loop if we switched to BEMF commutation */
    if (Flags.ClosedLoopMode) break;        // exit if BEMF is available

    /* save 60 degree timer count */
    T2CONbits.TON = 0;                      // stop TIMER2
    Timer2Average = TMR2;                   // save timer count for one commutation period
    TMR2 = 0;
    T2CONbits.TON = 1;                      // Start TIMER2

    /* reset the expected 30 degree BEMF crossover count to 1/2 of 60 degree count */
    ExpectedCrossover  = Timer2Average >> 1;
  }

  /* exit forced commutation and let BEMF take over */
  Flags.ClosedLoopMode = 1;            // startup complete

}



/******************************************************************************
 * Function:  void DoMotorStop(void)
 *
 * Output:		None
 *
 * Overview:	Stops the motor and resets the counters.
 *
 * Note:			None
 *******************************************************************************/
void DoMotorStop()
{
  /* disable PWM outputs */
  PTCONbits.PTEN = 0;                       // must disable PWM's before changing bits
  PWMCON1bits.TRGIEN = 0;                   // Disable individual PWM interrupt
  IFS5bits.PWM1IF = 0;                      // Clear the PWM Interrupt Flag
  IOCON1 = IOCON2 = IOCON3 = IOCON_ALL_OFF; // Manual Override all PWM outputs

  /* stop Timer1 */
  InitTMR1();

  /* Init timer2 speed measurement variables & timer registers */
  InitTMR2();

  Flags.RotorAlignment = 0;                 // turn of rotor alignement
  Flags.RunMotor = 0;                       // reset run flag
  Flags.Stop = 0;                           // motor is stopped
  Flags.Startup = 0;                        // do not startup
  Flags.ClosedLoopMode = 0;                 // open loop mode

  CurrentPWMDutyCycle = MIN_DUTY_CYCLE;     // Set PWM to the min value
  IEC5bits.PWM1IE = 0;                      // disable the PWM interrupt
}



/******************************************************************************
 * Function:  _PWM1Interrupt(void)
 *
 * Output:		None
 *
 * Overview:	PWM reload interrupt used to filter the BEMF signals using the
 *				Majority detection filter to detect a valid zero-crossing event
 *				if a valid zero-crossing event was detected then PreCommutationState.
 *				This function also includes the start-up sequence for detecting
 *				the initial rotor position
 *
 * Note:			None
 *******************************************************************************/
void __attribute__((__interrupt__,auto_psv)) _PWM1Interrupt(void)
{
  //Sets the ADC sampling point according to the PWM duty cycle
  SEVTCMP = CurrentPWMDutyCycle >> 1;  // sample at 1/2 current duty cycle
  
  /* increment open loop PWM interrupt counter */
  ++PWMticks;

  /* if we are not just aligning the motor for startup */
  if (!Flags.RotorAlignment)
  {
    /* motor is ramping up or Running */
    if (T1CONbits.TON)
    {
      /* commutation timer already running.  Skip detection until timer finishes. */
    }
    else
    {
      /* ignore crossover detection until we expect one to occur */
      if ( TMR2 > ExpectedCrossover)
      {
        /* start looking for a BEMF crossover */

        BlankingCounter++;
        if(BlankingCounter > ZC_BLANKING_COUNT)         // ignore crossover during blanking time
        {
          /* blanking time complete.  Start looking for BEMF crossover */
          BlankingCounter = ZC_BLANKING_COUNT + 1;

          // Masking the BEMF signals according to the SECTOR in order to determine the ACTIVE BEMF signal
          // XOR operator helps to determine the direction of the upcoming zero-crossing slope
          if((ComparatorOutputs^ADC_XOR[ADCCommState]) & ADC_MASK[ADCCommState]) adcBackEMFFilter |= 0x01;

          //Majority detection filter
          adcBackEMFFilter = ADC_BEMF_FILTER[adcBackEMFFilter];
          if (adcBackEMFFilter & 0b00000001)
          {
            /* crossover detected, initialize 30 degree commutation timer and prepare for commutation */
            PreCommutationState();
          }
        }
      }
    }
  }

  _PWM1IF = 0;
}



/******************************************************************************
 * Function:  PreCommutationState(void)
 *
 * Output:		None
 *
 * Overview:	This function measures the 60 and 30 electrical degrees
 *				using the TIMER2. The 60 electrical degrees is proportional
 *				to the elpased time between zero-crossing events.
 *				The zero-crossing events occur 30 electrical degrees in advance
 *				of the commutation point. Hence a delay proportional to the 30
 *				electrical degrees is added using TIMER1
 *
 * Note:			None
 *******************************************************************************/
void PreCommutationState(void)
{
//  UNUSED_RC9 = 1;

  // Calculate the time proportional to the 60 electrical degrees
  T2CONbits.TON = 0;        // Stop TIMER2

  /* see if we are still in open loop mode */
  if(!Flags.ClosedLoopMode)
  {
    /* open loop mode.  Check counter to determine if time to go closed loop */
    if (--Startup_Commutations <= 0) Flags.ClosedLoopMode = 1;
    /* TImer2 is started at forced commutation time during startup
    * instead of at crossover.  Forced startup is only 30 degrees from
    * crossover point where Timer 1 expects to be loaded. Must use
    * twice Timer 2 count at crossover during openloop only. */
    TMR2 = TMR2 + TMR2;

    /* average the 60 degree timer values */
    Timer2Average = TMR2;
  }
  else
  {
    /* average the 60 degree timer values */
    Timer2Average = (((unsigned long)(Timer2Average + TMR2)) / 2);
  }

  /* restart 60 degree timer */
  TMR2 = 0;
  T2CONbits.TON = 1;        // Start TIMER2

  /* Calculate the delay in TIMER1 counts proportional to the Phase Adv angle */
  PhaseAdvance = (((long)Timer2Average * PHASE_ADVANCE_DEGREES) / 60);

  /* Calculate and load TIMER1 with the counts porportional to 30 deg	minus the PHASE ADV angle delay */
  Timer1Value = (Timer2Average >> 1) - PhaseAdvance;

  /* bounds check */
  if(Timer1Value < 10)  Timer1Value = 10;

  /* set Timer 1 commutation timer comparison value */
  PR1 = Timer1Value;
  
  /* calculate the earliest possible crossover timer value */
  ExpectedCrossover = Timer1Value >> 1;

  /* only start commutation timer when in closed loop mode */
  if(Flags.ClosedLoopMode)
  {
    /* Change The Six-Step Commutation Sector in preparation for commutation in 30 degrees */
    if (++ADCCommState > 6) ADCCommState = 1;
      adcBackEMFFilter = 0;

    TMR1 = 0x00;            // reset starting timer count
    _T1IF = 0;              // Clear Timer 1 Interrupt Flag
    _T1IE = 1;              // Enable Timer1 interrupt
    T1CONbits.TON = 1;


    /* Calculate new speed parameters */
    #if (LOOPMODE == CLOSEDLOOPMODE)
      SpeedPILoopController();
    #elif (LOOPMODE == OPENLOOPMODE)
      OpenLoopController();
    #else
      Not Good!
    #endif // (LOOPMODE == CLOSEDLOOPMODE)
  }
//  UNUSED_RC9 = 0;
}


/******************************************************************************
 * Function:      _T1Interrupt(void)
 *
 * Output:		None
 *
 * Overview:		Here is where the motor commutation occurs,
 *				Timer1 ISR is utilized as the commutation delay used to
 *       commutate the motor windings at the right time
 *
 *******************************************************************************/
void __attribute__((__interrupt__,auto_psv)) _T1Interrupt(void)
{
//  UNUSED_RC8 = 1;

  /* Commutate */
  PTCONbits.PTEN = 0;                 // Reset PWM's for new commutation frame
  IOCON1 = PWM_STATE1[ADCCommState];  // commutate PWM1
  IOCON2 = PWM_STATE2[ADCCommState];  // commutate PWM2
  IOCON3 = PWM_STATE3[ADCCommState];  // commutate PWM3
  IFS5bits.PWM1IF = 0;                // Clearing the PWM Interrupt Flag
  IEC5bits.PWM1IE = 1;                // Enable the PWM interrupt
  PTCONbits.PTEN = 1;                 // Enable PWM's

  /* reset blanking timer for new commutation frame */
  BlankingCounter = 0;

  /* reset commutation Timer 1 */
  T1CONbits.TON = 0;                  // Stop TIMER1
  TMR1 = 0;                           // reset timer1 UP counter
  _T1IE = 0;                          // Disable Timer1 interrupt
  _T1IF = 0;                          // Clear Timer 1 Interrupt Flag
//  UNUSED_RC8 = 0;
}



/******************************************************************************
 * Function:  _ADCInterrupt(void)
 *
 * Output:		None
 *
 * Overview:	ADC interrupt used to measure the BEMF signals, reconstruct
 * 				the Motor Virtual Neutral Point and compare the BEMF signals
 *				against the neutral point reference to detect the zero-crossing
 *				event
 *
 * Note:			None
 *******************************************************************************/
void __attribute__((__interrupt__,auto_psv)) _AD1Interrupt(void)
{
  /* adjust SPEED POT value relative to 100% PWM count */
  ReferenceSpeed = (int) ((unsigned long)((unsigned long)ADC1BUF0 * (unsigned long) PWM_COUNTS_PER_PERIOD) >> ADC_BITS);

  /* bounds check lower limit */
  if(ReferenceSpeed < MIN_DUTY_CYCLE)
  {
    ReferenceSpeed = MIN_DUTY_CYCLE;
  }
  else
  {
    /* bounds check upper limit */
    if(ReferenceSpeed > MAX_DUTY_CYCLE)
    {
      ReferenceSpeed = MAX_DUTY_CYCLE;
    }
    else
    {
      /* within limits */
    }
  }

  //Reconstruct Voltage at the  Motor Neutral Point.
  MotorPhaseA = ADC1BUF1;		//ADC CH1 holds the Phase A value
  MotorPhaseB = ADC1BUF2;		//ADC CH2 holds the Phase B value
  MotorPhaseC = ADC1BUF3;		//ADC CH3 holds the Phase C value
  MotorNeutral = (MotorPhaseA + MotorPhaseB + MotorPhaseC) / 3;

  /********************* ADC SAMPLING & BMEF signals comparison ****************/
  if(BlankingCounter > ZC_BLANKING_COUNT)
  {
    ComparatorOutputs = 0;		// Precondition all comparator bits as zeros
    if(MotorPhaseA > MotorNeutral)
    {
      ComparatorOutputs |= 1;  // Set bit 0 when Phase A is higher than Neutural
    }
    if(MotorPhaseB > MotorNeutral)
    {
      ComparatorOutputs |= 2;	// Set bit 1 when Phase B is higher than Neutural
    }
    if(MotorPhaseC > MotorNeutral)
    {
      ComparatorOutputs |= 4;	// Set bit 2 when Phase C is higher than Neutral
    }
  }
  
  _DONE = 0;
  _AD1IF = 0;
}




#if (LOOPMODE == CLOSEDLOOPMODE)
  /******************************************************************************
   * Function:     SpeedPILoopController(void)
   *
   * Description:  PID closed loop speed controller
   *              Calculates a new duty cycle based upon PID control loop
   *              difference  between desired speed and current speed of motor.
   *
   * Output:		None
   *
   * Note:			None
   *******************************************************************************/
  void SpeedPILoopController(void)
  {

    //10 bit ADC POT RANGE 0-1024
    DesiredSpeed = (int)ReferenceSpeed;

    // Normalizing TIMER2 counts to electrical RevPerSec expressed in PWM counts
    // Timer 2 Counts are converted to PWM counts
    // and then multipied by the number of sectors
    // required to complete 1 electrical RPS
    CurrentSpeed = CurrentPWMDutyCycle;

    //PID controller
    PIDStructure.controlReference = (fractional)DesiredSpeed;
    PIDStructure.measuredOutput = (fractional)CurrentSpeed;
    PID(&PIDStructure);

    //Max and Min Limits for the PID output
    if (PIDStructure.controlOutput < MIN_DUTY_CYCLE)
    {
      PIDStructure.controlOutput = MIN_DUTY_CYCLE;
    }
    else	if (PIDStructure.controlOutput > MAX_DUTY_CYCLE)
    {
      PIDStructure.controlOutput = MAX_DUTY_CYCLE;
    }

    CurrentPWMDutyCycle = PIDStructure.controlOutput;
    MDC = CurrentPWMDutyCycle;
  }

#elif (LOOPMODE == OPENLOOPMODE)
  /******************************************************************************
   * Function:    OpenLoopController(void)
   *
   * Description: open loop speed controller.
   *              Calculates a new duty cycle based upon difference between
   *              Desired speed and current speed of motor.
   *
   * Output:      None
   *
   * Note:        None
   *******************************************************************************/
  void OpenLoopController(void)
  {
    // PWM duty cycle
    DesiredPWMDutyCycle = ReferenceSpeed;

    // Check for duty cycle change
    if(CurrentPWMDutyCycle != DesiredPWMDutyCycle)
    {
      if(CurrentPWMDutyCycle < DesiredPWMDutyCycle)
      {
        if(DesiredPWMDutyCycle - CurrentPWMDutyCycle < PWM_DUTY_CYCLE_10_PERCENT)
        {
          // slow acceleration for exact match
          CurrentPWMDutyCycle ++;
        }
        else
        {
          // fast acceleration
          CurrentPWMDutyCycle += OPEN_LOOP_ACCEL_RATE;
        }
      }
      else
      {
        // (CurrentPWMDutyCycle > DesiredPWMDutyCycle)
        if(CurrentPWMDutyCycle - DesiredPWMDutyCycle > PWM_DUTY_CYCLE_2_PERCENT)
        {
          // fast deceleration
          CurrentPWMDutyCycle -= OPEN_LOOP_DECEL_RATE;
        }
        else
        {
          // slow deceleration for exact match
          CurrentPWMDutyCycle --;
        }
      }
    }

    // Max and Min PWM duty cycle limits
    if (CurrentPWMDutyCycle < MIN_DUTY_CYCLE)
      CurrentPWMDutyCycle = MIN_DUTY_CYCLE;
    if (CurrentPWMDutyCycle > MAX_DUTY_CYCLE)
      CurrentPWMDutyCycle = MAX_DUTY_CYCLE;

    //Assigning new duty cycles to the PWM channels
    MDC = CurrentPWMDutyCycle;
  }

#else
  Not Good!
#endif // #if (LOOPMODE == CLOSEDLOOPMODE)


/******************************************************************************
 * Function:  InitADC10(void)
 *
 * Output:		None
 *
 * Overview:	Initializes the ADC module to operate in simultaneous mode
 *				sampling terminals AN0,AN1, AN2, AN3 using MUX A. The ADC channels are
 *				assigned as follows in the MCLV board
 *				CH0->AN8  (POT)
 *				CH1->AN3  PHASE A FILTERED
 *				CH2->AN4  PHASE B FILTERED
 *				CH3->AN5  PHASE C FILTERED
 *				CH4->AN0  IOUT A
 *				CH5->AN1  IOUT B
 *				CH6->AN2  IOUT C
 *				CH7->AN6  NEUTRAL FILTERED
 *				CH8->AN7 VDD Divided by 16
 *
 * 				ADC is sync with the PWM. ADC conversion is triggered
 *				every time a PWM reload event occurs. Tadc = 101.76 nSec.
 *				ADC resulting samples are formatted as unsigned 10-bits
 *				Right-justified
 *
 * Note:			None
 *******************************************************************************/
void InitADC10(void)
{

  AD1CON1 = 0x006C;
  // <15>  = 0 = ADON: ADC is off
  // <14>  = 0 = unimplemented
  // <13>  = 0 = ADSIDL: Continue module operation in Idle mode
  // <12>  = 0 = ADDMABM: DMA Buffer Mode, Scatter / Gather
  // <11>  = 0 = unimplemented
  // <10>  = 0 = AD12B: 10-bit, 4-channel ADC operation
  // <9:8> = 00 = FORM: Data Output Format bits Integer (0000 00dd dddd dddd)
  // <7:5> = 011 = SSRC: PWM Primary Special Event Trigger ends sampling and starts conversion
  // <4>   = 0 = SSRCG: Selects SSRC group two bits <7:5> above
  // <3>   = 1 = SIMSAM: Samples CH0, CH1, CH2, CH3 simultaneously when CHPS<1:0> = 1x
  // <2>   = 1 = ASAM: Sampling begins immediately after last conversion; SAMP bit is auto-set
  // <1>   = status = SAMP: 1=sampling, 0=holding
  // <0>   = status = DONE: 1=Conversion DONE, 0=Conversion not started or is in progress

  AD1CON2 = 0x0300;
  //<15:13> = 000: VREFH = AVDD,  VREFL = AVSS
  //<10> = 0:  Do not scan inputs
  //<9:8> = 11: 1x = Converts CH0, CH1, CH2 and CH3
  //<7> status: 0=A/D is currently filling first half of buffer, 1=A/D is currently filling second half of buffer
  //<6:2> = 00000: Interrupts at the completion of conversion for each sample/convert sequence
  //<1> = 0: Always starts filling buffer from the beginning
  //<0> = 0: Always uses channel input selects for Sample MUX A

  AD1CON3 = 0x0000 + ADC_ADCS;
  //<15> = 0: ADC Clock derived from system clock, minimum of 76 ns
  //<12:8> = 0: Autosample time bits = 0 TAD since PWM is controlling sampling time
  //<7:0> = 2: TAD = (AD1CON3<7:0> + 1) * TCY = 3*TCY, @FOSC=60 MHz => 3*32.89338432 nSec, TAD = 98.68 nSec

  AD1CON4 = 0x0000;
  //<8> = 0: No DMA, Results stored in ADC1BUF0 thru 15

  AD1CHS123 = 0x0100;
  //<15:11> not implemented
  //<10:9> = 00: MUX B CH1, CH2, CH3 negative input is VREF-
  //<8> = 1: MUX B CH1 positive input is AN3(IOUTB), CH2 is AN0(MONITOR1), CH3 is AN6(IOUTC)
  //<7:3> not implemented
  //<2:1> = 00: MUX A CH1, CH2, CH3 negative input is VREF-
  //<0> = 0: MUX A CH1 positive input is AN0, CH2 positive input is AN1, CH3 positive input is AN2

  AD1CHS0 = 0x0408;
  //<15> = 0: MUX B Channel 0 negative input is VREF-
  //<14:13> not implemented
  //<12:8> 4: MUX B Channel 0 positive input is AN4 (IOUTA)
  //<7> = 0: MUX A Channel 0 negative input is VREF-
  //<6:5> not implemented
  //<4:0> = 8: MUX A Channel 0 positive input is AN8 (SPEED)

  AD1CSSH = 0x0000;	//Skip all ANx channels for input scan
  AD1CSSL = 0x0000;	//Skip all ANx channels for input scan

  _DONE = 0;    //Making sure that there is no conversion in progress
  _AD1IP = 5;		//Assigning ADC ISR priority (IPC3<6:4>)
  _AD1IF = 0;		//Clearing the ADC Interrupt Flag (IFS0<13>)
  _AD1IE = 1;		//Enabling the ADC conversion complete interrupt (IEC0<13>)
  _ADON = 1;    //Enabling the ADC module

}



/******************************************************************************
 * Function:     InitMCPWM(void)
 *
 * Output:		None
 *
 * Overview:	Master timebase interrupt is used to trigger A/D.
 *            PWM1 interrupt is used to reload dutycycle.
 *
 * 
 * Notes:
 *   Initializes the PWM module to operate in edge-aligned mode
 *	 at 50KHz (set in header file).
 *   PWM pins are configured in independent mode.
 *   PWM Resolution (bits) = Log2 (2 * Fcy / Fpwm)
 *                @50kHz  = Log2 (2 * 74,621,250 / 50,000)
 *                        = Log2 (2984)
 *                        = 8.00 bits = 78.125 ns / count, (20 us / 2^8)
 *
 *      FPWM = 50KHz PWM Freq
 *      P1TMR Prescaler = 1:1
 *      FCY  = 74,621,250, Tcy = 13.40 ns
 *      MAX_DUTY_CYCLE = FCY/FPWM = 74,621,250 / 50,000 = 1492 tics = 20 usec
 *
 *******************************************************************************/
void InitMCPWM(void)
{
  /* PTCON - PWM Time Base Control Register */
  PTCONbits.PTEN    = 0;      // <15>  disable PWM until initialized
  PTCONbits.PTSIDL  = 1;      // <13>  PWM time base halted in CPU IDLE mode
  PTCONbits.SEIEN   = 0;      // <11>  Special Event Interrupt Enable is disabled (main PWM interrupt)
  PTCONbits.EIPU    = 0;      // <10>  0=Period Updates occur in PWM cycle boundaries
  PTCONbits.SYNCPOL = 0;      // <9>   0=SYNC in/out is active high
  PTCONbits.SYNCOEN = 0;      // <8>   0=SYNCO output is disabled
  PTCONbits.SYNCEN  = 0;      // <7>   0=External SYNC disabled
  PTCONbits.SYNCSRC = 0;      // <6:4> 0=Synchronous source selection is SYNCI1 from PPS
  PTCONbits.SEVTPS  = 0;      // <3:0> 0=Special event trigger on every compare match event

  /* PTCON2 - PWM Clock Divider Select Register 2 */
  PTCON2bits.PCLKDIV = PWM_INPUT_CLOCK_PRESCALER; // <2:0> 0 = PWM clock prescaler divide by 1.

  /* PTPER - Primary Master Time Base Period Register */
  PTPER = FPLLO / FPWM;       // reload value is one period

  /* MDC: PWMx MASTER DUTY CYCLE REGISTER */
  MDC = STARTUP_DUTY_CYCLE;   // PWM Master duty cycle register used instead of individual duty cycles

  /* SEVTCMP: PWMx PRIMARY SPECIAL EVENT COMPARE REGISTER */
  SEVTCMP = 0;                // initialize count for ADC Conversion trigger

  /* CHOP: PWMx CHOP CLOCK GENERATOR REGISTER */
  CHOP = 0;                   // disable chop clock

  /* TRIGx: PWMx PRIMARY TRIGGER COMPARE VALUE REGISTER */
  TRIG1 = TRIG2 = TRIG3 = 0;  // trigger control value bits that generate a trigger event for the ADC

  /* PWMCONx: PWMx CONTROL REGISTER */
  PWMCON1bits.FLTIEN = 0;     // <12> disable fault interrupt
  PWMCON1bits.CLIEN = 0;      // <11> disable current limit interrupt
  PWMCON1bits.TRGIEN = 0;     // <10> disable individual PWM event interrupts
  PWMCON1bits.ITB = 0;        // <9> 0=PTPER provides timing, 1=Phase provides timing for this PWM (1 for Center Aligned)
  PWMCON1bits.MDCS = 1;       // <8> 0=PDCx provides duty cycle info, 1=MDC provides duty cycle for this PWM
  PWMCON1bits.DTC = 0;        // <7:6> Positive dead time is applied to all output modes
  PWMCON1bits.MTBS = 0;       // <3> Master Timebase uses primary generator
  PWMCON1bits.CAM = 0;        // <2> 0=edge aligned mode, 1=Center Aligned Mode
  PWMCON1bits.XPRES = 0;      // <1> External pins do not affect PWM timebase
  PWMCON1bits.IUE = 0;        // <0> PWM register updates are synchronized on PWM period boundary
  PWMCON2 = PWMCON3 = PWMCON1;

  /* PDCx: PWMx GENERATOR DUTY CYCLE REGISTER */
  PDC1 = PDC2 = PDC3 = STARTUP_DUTY_CYCLE;  // USE MDC register instead

  /* PHASEx: PWMx PRIMARY PHASE-SHIFT REGISTER or TIMEBASE in Center-Aligned mode */
  PHASE1 = PHASE2 = PHASE3 = PTPER; // Center mode uses PHASE for timebase period;

  /* DTRx: PWMx DEAD-TIME REGISTER */
  DTR1 = DTR2 = DTR3 = 2;                 // 2 cycle dead time for PWMH rising

  /* ALTDTRx: PWMx ALTERNATE DEAD-TIME REGISTER */
  ALTDTR1 = ALTDTR2 = ALTDTR3 = 2;        // 2 cycle Dead Time for PWML rising

  /* TRGCONx: PWMx TRIGGER CONTROL REGISTER */
  TRGCON1bits.TRGDIV = 0;                 // <15:12> Trigger output for every trigger event
  TRGCON1bits.TRGSTRT = 0;                // <5:0> Trigger after 0 PWM cycles after module is enabled.
  TRGCON2 = TRGCON3 = TRGCON1;

  /* TRIGx: PWMx PRIMARY TRIGGER COMPARE VALUE REGISTER */
  TRIG1 = TRIG2 = TRIG3 = 0;              // this register contains the compare values that trigger the ADC module

  /* IOCONx: PWMx I/O CONTROL REGISTER */
  IOCON1 = PMOD_SEL | PENH_ENA | PENL_ENA | POLH_HI | POLL_HI | OVRENH_OVR | OVRENL_OVR | OSYNC_PWM | OVRDATH_0 | OVRDATL_0;
  IOCON2 = IOCON3 = IOCON1;

  /* FCLCONx: PWMx FAULT CURRENT-LIMIT CONTROL REGISTER */
  FCLCON1 = FCLCON2 = FCLCON3 = 0x0003;   // Current Limit mode and Fault Input are disabled

  /* LEBCONx: PWMx LEADING-EDGE BLANKING CONTROL REGISTER */
  LEBCON1 = LEBCON2 = LEBCON3 = 0;        // no leading edge blanking

  /* LEBDLYx: PWMx LEADING-EDGE BLANKING DELAY REGISTER */
  LEBDLY1 = LEBDLY2 = LEBDLY3 = 0;        // no leading edge blanking delay

  /* AUXCONx: PWMx AUXILIARY CONTROL REGISTER */
  AUXCON1 = AUXCON2 = AUXCON3 = 0;

  /* keep interrupts disabled for now. */
  _PWM1IP = 4;                            // PWM Interrupt Priority 4
  _PWM1IE = 0;                            // disabling the PWM interrupt
  _PTEN = 0;                              // disabling the PWM module
  _PWM1IF = 0;                            // Clearing the PWM Interrupt Flag

  /* PWM special event Interrupt is not used */
  _PSEMIF = 0;
  _PSEMIE = 0;
  _PSEMIP = 0;                            // PWM special event Interrupt Priority
}



/******************************************************************************
 * Function:  InitTMR2(void)
 *
 * Output:		None
 *
 * Overview:	Initializes the TIMER2 module to operate in free-running
 *            up counting mode. The TIMER2 time base is Tcy * 64.
 *            This timer is used to calculate the motor speed
 *
 * Note:			None
 *******************************************************************************/
void InitTMR2(void)
{
  // stop Timer2
  T2CONbits.TON = 0;        // wait to Enable Timer2

  // Tcy = 33.921 nSec
  TMR2 = 0;               // Resetting TIMER
  T2CONbits.TSIDL = 0;    //Continue Timer in idle mode
  T2CONbits.TGATE = 0;    // gated time accumulation disabled
  T2CONbits.TCKPS = TIMER2_DIVISOR_TCKPS; // use calculated timer 2 clock rate
  T2CONbits.T32 = 0;      // 16 bit timer
  T2CONbits.TCS = 0;      // internal clock = FOSC/2

  PR2 = 0x1000;           // initial TIMER periond
  //Timer2Value = Timer2Average = 0x1000;
  Timer2Average = 0x1000;
  
  IPC1bits.T2IP = 0;
  IFS0bits.T2IF = 0;
  IEC0bits.T2IE = 0;

}



/******************************************************************************
 * Function:  InitTMR1(void)
 *
 * Output:		None
 *
 * Overview:	Initializes the TIMER1 module to operate in free-running
 *            up counting mode. The TIMER1 time base is Tcy * 64.
 *            This timer is used to calculate the commutation delay
 *
 * Note:			None
 *******************************************************************************/
void InitTMR1(void)
{
  TMR1 = 0;             // Resetting TIMER
  PR1 = 50;             // Initial commutation delay value 43.4 uSec

  // use calculated timer 1 clock rate
  T1CONbits.TCKPS = TIMER1_DIVISOR_TCKPS;

  _T1IP = 6;            // Set Timer 1 Interrupt Priority Level
  _T1IF = 0;            // Clear Timer 1 Interrupt Flag
  _T1IE = 0;            // wait to Enable Timer1 interrupt
  T1CONbits.TON = 0; 		// disable Timer1
}



/******************************************************************************
 * Function:  ConfigureDriver(void)


 *
 * Output:		None
 *
 * Overview:	Sends configuration info to the motor driver
 *            Must be sent after a brown out, power failure, or anytime when
 *            the driver needs to be re-configured.
 *
 * Note:
 *  ************************* CONFIG Register 0 *********************************
 *  SET_CFG_0 81H
 *  Bit Val Description
 *  7     0 Unused (Startup Default)
 *  6     0 Disconnect of 30K LIN Bus/Level Translator Pullup When CE=0 is Disabled
 *        1 Disconnect of 30K LIN Bus/Level Translator Pullup When CE=0 is Enabled
 *  5     1 Sleep when CE=0
 *        0 Standby when CE=0
 *  4     0 Internal Neutral Simulator Disabled (Startup Default)
 *        1 Internal Neutral Simulator Enabled
 *  3     0 Undervoltage Lockout Enabled (Default)
 *        1 Undervoltage Lockout Disabled
 *  2     0 External MOSFET Overcurrent Detection Enabled (Default)
 *        1 External MOSFET Overcurrent Detection Disabled
 *  1:0  00 0.250V External MOSFET Overcurrent Limit (Default)
 *       01 0.500V External MOSFET Overcurrent Limit
 *       10 0.750V External MOSFET Overcurrent Limit
 *       11 1.000V External MOSFET Overcurrent Limit
 *
 *  ************************* CONFIG Register 1 *********************************
 *  SET_CFG_2 87H
 *  Bit Val Description
 *  7:5  0 Unused (Startup Default)
 *  4:2  000 2 us driver dead time (startup default)
 *       001 1.750 us driver dead time
 *       010 1.500 us driver dead time
 *       011 1.250 us driver dead time
 *       100 1.000 us driver dead time
 *       101 0.750 ns driver dead time
 *       110 0.500 ns driver dead time
 *       111 0.250 ns driver dead time
 *  1:0  00 4 us driver current limit blanking time (startup default)
 *       01 2 us driver current limit blanking time
 *       10 1 us driver current limit blanking time
 *       11 500 ns driver current limit blanking time
 *******************************************************************************/
void ConfigureDriver(void)
{
  //Send configuration data
  writeUART1(0x81); // SET_CFG_0
  writeUART1(0x0B); // UVLO disabled, 1.00V External MOSFET Over Current Reference
  __delay_ms(3);

  // Send Current Sense Limit Value
  // DAC range is 0.991V (0x00) to 4.503V(FF) = 13.77mV/bit.
  // Minimum DAC output is 0.991V with 0 programmed current.
  // DACbits = ((Desired Current * Rsense * Gain) - 0.991V) / (13.77 mV/Bit)
  // Ilimit = (DACbits * 13.77 mV/bit + 0.991V) / (Rsense * Gain)
  // Minimum Ilimit = ( 0 * 13.77 mV/Bit + 0.991V) / (0.01 * 11) = 9.0 amps with 0.01 ohm sense.
  // ex:  10A limit, 0.01 Rsense, 11 Av = (((10 * 0.01 * 11) - 0.991) / 13.77 mV/Bit) = 7.916 = 0x08 (rounded up)

  writeUART1(0x83); // SET_CFG_1
  writeUART1(0xFF); // ((15 Amps * 0.01 * 11) - 0.991V) / (13.77 mV/Bit) = 119.8 = 0x77
  __delay_ms(5);

  // Send Blanking Interval
  writeUART1(0x87); // SET_CFG_2
  writeUART1(0x19); // 500 ns driver dead time, 2 us current limit Blanking time
  __delay_ms(30);

  #if (DE2_SERIAL_RECEIVE == ENABLED)
    /* throw away comm data, not using it */
    ClearCommunicationsPort();
  #endif // (DE2_SERIAL_RECEIVE == ENABLED)

}


#if (DE2_SERIAL_RECEIVE == ENABLED)
  /******************************************************************************
   * Function:     void CheckDriverCommunications(void)
   *
   * Output:      None
   *
   * Overview:		Polls the DE2 communications line for any received characters.
   *              Use this to check for faults that set ILIMIT low, indicating
   *              a driver shutdown.
   *
   *
   *
   * Note:        None
   *******************************************************************************/
  void CheckDriverCommunications()
  {
    Msg_Type databyte;

    if (readUART1(&databyte.Cmd))
    {
      /* get UART data from driver */
      switch (databyte.Cmd)
      {
        case STAT0_MSG:
        {
          if (readUART1(&databyte.Data))
          {
            if (databyte.Status0.all)
            {
              /* see if fault occured */
              if ( databyte.Status0.all && 0x97)
              {
                /* fault occurred */
              }
            }
          }
          break;
        }

        case STAT1_MSG:
        {
          if (readUART1(&databyte.Data))
          {
            if (databyte.Status1.all)
            {
              /* see if fault occured */
              if (databyte.Status1.all && 0x3C)
              {
                /* fault occurred */
              }
            }
          }
          break;
        }

        case SET_CFG0_MSG_RSP:
        case GET_CFG0_MSG_RSP:
        case SET_CFG1_MSG_RSP:
        case GET_CFG1_MSG_RSP:
        case STAT0_MSG_RSP:
        case STAT1_MSG_RSP:
        case SET_CFG2_MSG_RSP:
        case GET_CFG2_MSG_RSP:
        {
          /* get response databyte */
          readUART1(&databyte.Data);
          break;
        }

        case SET_CFG0_MSG:  // need to remove echoed databyte from SET commands
        case SET_CFG1_MSG:
        case SET_CFG2_MSG:
        {
          /* get response databyte */
          readUART1(&databyte.Data);
          break;
        }

        default:
        {
          /* ignore */
          break;
        }
      } // switch()
    } // if readUART1()

  }
#endif // if (0)
